/*
 * pcm9211.c  --  PCM9211 ALSA SoC Audio driver
 *
 * Copyright (C) 2023 AlphaTheta Corporation
 */

#include <linux/mutex.h>
#include <linux/init.h>
#include <linux/module.h>
#include <linux/of_device.h>
#include <linux/regmap.h>
#include <linux/regulator/consumer.h>
#include <linux/spi/spi.h>
#include <linux/gpio.h>
#include <linux/delay.h>
#include <sound/soc.h>
#include <sound/pcm_params.h>
#include <sound/tlv.h>

#include "pcm9211.h"

#define GPIO_XRST_18		69	// GPIO2_A5 32*2+0+5
#define GPIO_CLK_CLR_18		67	// GPIO2_A3 32*2+0+3
#define GPIO_MCLK_SEL_18	155	// GPIO4_D3 32*4+24+3
#define GPIO_MCLK_SEL48_18	2	// GPIO0_A2 32*0+0+2

#define CATEGORY_CODE				(0x01)	/* cd player */
//#define CATEGORY_CODE				(0x12)	/* digital signal mixer */
//#define CATEGORY_CODE				(0x22)	/* digital sound sampler */
//#define CATEGORY_CODE				(0x2A)	/* digital sound processor */
#define LEVEL_II					(0x00)

#define CATEGORY_LASEROPT_MASK		(0b00000111)
#define CATEGORY_LASEROPT_VALUE		(0b00000001)
#define CATEGORY_BROADCAST_MASK1	(0b00000111)
#define CATEGORY_BROADCAST_VALUE1	(0b00000100)
#define CATEGORY_BROADCAST_MASK2	(0b00001111)
#define CATEGORY_BROADCAST_VALUE2	(0b00001110)
#define COPYPARAM_FREE		(0)
#define COPYPARAM_ONCE		(1)
#define COPYPARAM_NEVER		(2)

#define COPYPARAM_DEFAULT	COPYPARAM_FREE	/* initial value of copy parameter */

/* PCM9211 Codec Private Data */
struct pcm9211_priv {
	struct regmap *regmap;
	unsigned int fmt;
	unsigned int pcm_format;
	unsigned int pcm_rate;
	struct mutex lock;
	int copyparam;
};

/* PCM9211 Default Register Value */
static const struct reg_default pcm9211_reg_defaults[] = {
	{ 0x20, 0x00 },
	{ 0x21, 0x00 },
	{ 0x22, 0x01 },
	{ 0x23, 0x04 },
	{ 0x24, 0x00 },
	{ 0x25, 0x01 },
	{ 0x26, 0x01 },
	{ 0x27, 0x00 },
	{ 0x28, 0x03 },
	{ 0x29, 0x0C },
	{ 0x2A, 0xFF },
	{ 0x2B, 0xFF },
	{ 0x2E, 0x00 },
	{ 0x2F, 0x04 },
	{ 0x30, 0x02 },
	{ 0x31, 0x1A },
	{ 0x32, 0x22 },
	{ 0x33, 0x22 },
	{ 0x34, 0xC2 },
	{ 0x35, 0x02 },
	{ 0x36, 0x02 },
	{ 0x37, 0x00 },
	/* Reserved: 0x3E - 0x3F */
	{ 0x40, 0xC0 },
	/* Reserved: 0x41 */
	{ 0x42, 0x02 },
	/* Reserved: 0x43 - 0x45 */
	{ 0x46, 0xD7 },
	{ 0x47, 0xD7 },
	{ 0x48, 0x00 },
	{ 0x49, 0x00 },
	/* Reserved: 0x4A - 0x59 */
	{ 0x60, 0x44 },
	{ 0x61, 0x10 },
	{ 0x62, 0x00 },
	{ 0x63, 0x00 },
	{ 0x64, 0x00 },
	{ 0x65, 0x00 },
	{ 0x66, 0x00 },
	{ 0x67, 0x00 },
	{ 0x68, 0x00 },
	/* Reserved: 0x69 */
	{ 0x6A, 0x00 },
	{ 0x6B, 0x00 },
	{ 0x6C, 0x00 },
	{ 0x6D, 0x00 },
	{ 0x6E, 0x0F },
	{ 0x6F, 0x40 },
	{ 0x70, 0x00 },
	{ 0x71, 0x00 },
	{ 0x72, 0x00 },
	{ 0x73, 0x00 },
	{ 0x74, 0x00 },
	{ 0x75, 0x00 },
	{ 0x76, 0x00 },
	{ 0x77, 0x00 },
	{ 0x78, 0x3D },
	{ 0x79, 0x00 },
	{ 0x7A, 0x00 },
	{ 0x7B, 0x00 },
	{ 0x7C, 0x00 },
};

/* regmap */
static const struct regmap_range pcm9211_readable_ranges[] = {
	regmap_reg_range(0x20, 0x3D),
	regmap_reg_range(0x40, 0x40),
	regmap_reg_range(0x42, 0x42),
	regmap_reg_range(0x46, 0x49),
	regmap_reg_range(0x5A, 0x68),
	regmap_reg_range(0x6A, 0x7E)
};

static const struct regmap_access_table pcm9211_readable_table = {
	.yes_ranges = pcm9211_readable_ranges,
	.n_yes_ranges = ARRAY_SIZE(pcm9211_readable_ranges),
};

static const struct regmap_range pcm9211_writable_ranges[] = {
	regmap_reg_range(0x20, 0x2B),
	regmap_reg_range(0x2E, 0x37),
	regmap_reg_range(0x40, 0x40),
	regmap_reg_range(0x42, 0x42),
	regmap_reg_range(0x46, 0x49),
	regmap_reg_range(0x60, 0x68),
	regmap_reg_range(0x6A, 0x7C)
};

static const struct regmap_access_table pcm9211_writable_table = {
	.yes_ranges = pcm9211_writable_ranges,
	.n_yes_ranges = ARRAY_SIZE(pcm9211_writable_ranges),
};

static const struct regmap_range pcm9211_volatile_ranges[] = {
	regmap_reg_range(0x2C, 0x2D),
	regmap_reg_range(0x38, 0x3D),
	regmap_reg_range(0x40, 0x40),
	regmap_reg_range(0x5A, 0x5F),
	regmap_reg_range(0x7D, 0x7E)
};

static const struct regmap_access_table pcm9211_volatile_table = {
	.yes_ranges = pcm9211_volatile_ranges,
	.n_yes_ranges = ARRAY_SIZE(pcm9211_volatile_ranges),
};

static const struct regmap_config pcm9211_regmap = {
	.reg_bits = 8,
	.val_bits = 8,
	.cache_type = REGCACHE_RBTREE,
	.max_register = 0x7E,

	.reg_defaults = pcm9211_reg_defaults,
	.num_reg_defaults = ARRAY_SIZE(pcm9211_reg_defaults),

	.rd_table = &pcm9211_readable_table,
	.wr_table = &pcm9211_writable_table,
	.volatile_table = &pcm9211_volatile_table,

	.use_single_rw = true,		/* 連続Read/Write機能はないのでシングル動作 */
};


static void pcm9211_set_channel_status(struct pcm9211_priv *pcm9211)
{
	u8 spi_data[6];

	/* Set channel status */
	spi_data[0] = 0; /* bit7-0 */
	spi_data[1] = CATEGORY_CODE; /* bit15-8 */
	spi_data[2] = 0; /* bit23-16 */
	spi_data[3] = (LEVEL_II & 0x30); /* bit31-24 */;
	spi_data[4] = 0; /* bit39-32 */
	spi_data[5] = 0;

	if (((CATEGORY_CODE & CATEGORY_LASEROPT_MASK) == CATEGORY_LASEROPT_VALUE)
	||  ((CATEGORY_CODE & CATEGORY_BROADCAST_MASK1) == CATEGORY_BROADCAST_VALUE1)
	||  ((CATEGORY_CODE & CATEGORY_BROADCAST_MASK2) == CATEGORY_BROADCAST_VALUE2)) {
		switch (pcm9211->copyparam) {
		case COPYPARAM_FREE:	/* copy free */
			spi_data[0] |= 0x04; /* bit7-0 */	/* COPY=1 */
			spi_data[1] |= 0x80; /* bit15-8 */	/* L=1 */
			break;
		case COPYPARAM_ONCE:	/* copy once */
			spi_data[0] &= 0xFB; /* bit7-0 */	/* COPY=0 */
			spi_data[1] &= 0x7F; /* bit15-8 */	/* L=0 */
			break;
		case COPYPARAM_NEVER:	/* copy never */
		default:				/* parameter NG */
			spi_data[0] &= 0xFB; /* bit7-0 */	/* COPY=0 */
			spi_data[1] |= 0x80; /* bit15-8 */	/* L=1 */
			break;
		}
	}
	else {
		switch (pcm9211->copyparam) {
		case COPYPARAM_FREE:	/* copy free */
			spi_data[0] |= 0x04; /* bit7-0 */	/* COPY=1 */
			spi_data[1] &= 0x7F; /* bit15-8 */	/* L=0 */
			break;
		case COPYPARAM_ONCE:	/* copy once */
			spi_data[0] &= 0xFB; /* bit7-0 */	/* COPY=0 */
			spi_data[1] |= 0x80; /* bit15-8 */	/* L=1 */
			break;
		case COPYPARAM_NEVER:	/* copy never */
		default:				/* parameter NG */
			spi_data[0] &= 0xFB; /* bit7-0 */	/* COPY=0 */
			spi_data[1] &= 0x7F; /* bit15-8 */	/* L=0 */
			break;
		}
	}

	switch (pcm9211->pcm_rate) {
	case 44100:
	default:
		spi_data[3] |= 0x00; /* bit27-24 */
		spi_data[4] = 0xF0; /* bit39-36 */
		break;
	case 48000:
		spi_data[3] |= 0x02; /* bit27-24 */
		spi_data[4] = 0xD0; /* bit39-36 */
		break;
	case 88200:
		spi_data[3] |= 0x08; /* bit27-24 */
		spi_data[4] = 0x70; /* bit39-36 */
		break;
	case 96000:
		spi_data[3] |= 0x0A; /* bit27-24 */
		spi_data[4] = 0x50; /* bit39-36 */
		break;
	}
	switch (pcm9211->pcm_format) {
	case SNDRV_PCM_FORMAT_S16_LE:
		spi_data[4] |= 0x02; /* bit32, 33-35 */
		break;
	case SNDRV_PCM_FORMAT_S24_LE:
	default:
		spi_data[4] |= 0x0B; /* bit32, 33-35 */
		break;
	}

	/* update DIT channel status */
	regmap_bulk_write(pcm9211->regmap, 0x63, spi_data, ARRAY_SIZE(spi_data));
}

static ssize_t pcm9211_set_copyparam(struct device *dev,
		struct device_attribute *attr,
		const char *buf, size_t count)
{
	struct pcm9211_priv *pcm9211 = dev_get_drvdata(dev);
	long copyparam = COPYPARAM_NEVER;
	int ret;

	ret = kstrtol(buf, 10, &copyparam);
	if (ret < 0) {
		return ret;
	}

	switch (copyparam) {
	case COPYPARAM_FREE:	/* parameter OK */
	case COPYPARAM_ONCE:	/* parameter OK */
	case COPYPARAM_NEVER:	/* parameter OK */
		break;
	default:				/* parameter NG */
		copyparam = COPYPARAM_NEVER;
		break;
	}

	mutex_lock(&(pcm9211->lock));
	pcm9211->copyparam = copyparam;
	pcm9211_set_channel_status(pcm9211);
	mutex_unlock(&(pcm9211->lock));

	return count;
}
static DEVICE_ATTR(copyparam, 0200, NULL, pcm9211_set_copyparam);

static int pcm9211_soc_probe(struct snd_soc_codec *codec)
{
	int ret;

	codec->control_data = dev_get_drvdata(codec->dev);

	ret = device_create_file(codec->dev, &dev_attr_copyparam);
	if (ret < 0) {
		pr_err("Failed to create copyparam sysfs file: %d\n", ret);
		return ret;
	}

	return 0;
}

static int pcm9211_soc_remove(struct snd_soc_codec *codec)
{
	device_remove_file(codec->dev, &dev_attr_copyparam);
	return 0;
}

static struct snd_soc_codec_driver pcm9211_codec_driver = {
	.probe = pcm9211_soc_probe,
	.remove = pcm9211_soc_remove,
};


static const uint32_t pcm9211_dai_rates_slave[] = {
	44100, 48000, 88200, 96000
};

static const struct snd_pcm_hw_constraint_list pcm9211_constraint_list_rates = {
	.list  = pcm9211_dai_rates_slave,
	.count = ARRAY_SIZE(pcm9211_dai_rates_slave),
};


static int pcm9211_dai_startup_slave(
		struct snd_pcm_substream *substream, struct snd_soc_dai *dai)
{
	struct snd_soc_codec *codec = dai->codec;
	int ret;
	u64 formats = SNDRV_PCM_FMTBIT_S16_LE
				| SNDRV_PCM_FMTBIT_S24_LE;

	ret = snd_pcm_hw_constraint_list(substream->runtime, 0,
			SNDRV_PCM_HW_PARAM_RATE, &pcm9211_constraint_list_rates);
	if (ret != 0) {
		dev_err(codec->dev, "Failed to setup rates constraints: %d\n", ret);
	}

	ret = snd_pcm_hw_constraint_mask64(substream->runtime,
			SNDRV_PCM_HW_PARAM_FORMAT, formats);
	if (ret != 0) {
		dev_err(codec->dev, "Failed to setup formats constraints: %d\n", ret);
	}

	return ret;
}

static int pcm9211_dai_startup(
		struct snd_pcm_substream *substream, struct snd_soc_dai *dai)
{
	struct snd_soc_codec *codec = dai->codec;
	struct pcm9211_priv *pcm9211
					= snd_soc_codec_get_drvdata(codec);

	switch (pcm9211->fmt & SND_SOC_DAIFMT_MASTER_MASK) {
	case SND_SOC_DAIFMT_CBS_CFS:
		return pcm9211_dai_startup_slave(substream, dai);

	default:
		return (-EINVAL);
	}
}

static int pcm9211_hw_params(
		struct snd_pcm_substream *substream, struct snd_pcm_hw_params *params,
		struct snd_soc_dai *dai)
{
	struct snd_soc_codec *codec = dai->codec;
	struct pcm9211_priv *pcm9211
					= snd_soc_codec_get_drvdata(codec);

	pr_debug("%s(): rate=%d \n" , __func__, params_rate(params));

	switch (params_format(params)) {
	case SNDRV_PCM_FORMAT_S16_LE:
	case SNDRV_PCM_FORMAT_S24_LE:
		break;
	default:
		pr_err("[%s] invalid format\n", __func__);
		return -EINVAL;
	}

	switch (params_rate(params)) {
	case 44100:
		gpio_set_value(GPIO_MCLK_SEL_18, 0);
		gpio_set_value(GPIO_MCLK_SEL48_18, 0);
		break;
	case 48000:
		gpio_set_value(GPIO_MCLK_SEL_18, 1);
		gpio_set_value(GPIO_MCLK_SEL48_18, 0);
		break;
	case 88200:
		gpio_set_value(GPIO_MCLK_SEL_18, 0);		// dummy
		gpio_set_value(GPIO_MCLK_SEL48_18, 1);		// dummy
		break;
	case 96000:
		gpio_set_value(GPIO_MCLK_SEL_18, 1);
		gpio_set_value(GPIO_MCLK_SEL48_18, 1);
		break;
	default:
		pr_err("[%s] unsupported sampling rate\n", __func__);
		return -EINVAL;
	}

	/* Set channel status */
	mutex_lock(&(pcm9211->lock));
	pcm9211->pcm_format = params_format(params);
	pcm9211->pcm_rate = params_rate(params);
	pcm9211_set_channel_status(pcm9211);
	mutex_unlock(&(pcm9211->lock));

	return 0;
}

static void pcm9211_dai_shutdown(
		struct snd_pcm_substream *substream, struct snd_soc_dai *dai)
{
	struct snd_soc_codec *codec = dai->codec;
	struct pcm9211_priv *pcm9211
					= snd_soc_codec_get_drvdata(codec);

	if (pcm9211 == NULL) {
		pr_err("[%s] invalid device private data\n", __func__);
		return;
	}
}

static int pcm9211_digital_mute(struct snd_soc_dai *dai, int mute)
{
	struct snd_soc_codec *codec = dai->codec;
	u8 mute_reg;

	/* control TXDMUT(DIT Output Audio Data Mute) bit */
	mute_reg = (mute) ? 0x20U : 0x00U;
	return snd_soc_update_bits(codec, 0x62, 0x20, mute_reg);
}

static const struct snd_soc_dai_ops pcm9211_dai_ops = {
	.startup = pcm9211_dai_startup,
	.shutdown = pcm9211_dai_shutdown,
	.hw_params = pcm9211_hw_params,
	.digital_mute = pcm9211_digital_mute,
};


#define PCM9211_RATES (SNDRV_PCM_RATE_44100 | \
					   SNDRV_PCM_RATE_48000 | \
					   SNDRV_PCM_RATE_96000)

#define PCM9211_FORMATS (SNDRV_PCM_FMTBIT_S16_LE | \
						 SNDRV_PCM_FMTBIT_S24_LE)

static struct snd_soc_dai_driver pcm9211_dai = {
	.name = "pcm9211-dai",
	.playback = {
		.stream_name = "Playback",
		.channels_min = 2,
		.channels_max = 2,
		.rates = PCM9211_RATES,
		.formats = PCM9211_FORMATS,
	},
	.ops = &pcm9211_dai_ops,
};

static const struct reg_sequence pcm9211_init_sequence[] = {
	//XTI Source, Clock (SCK/BCK/LRCK) Frequency Setting
	//XTI CLK source 12.288 and BCK 3.072, LRCK 48k = XTI/512
	{ 0x31, 0x1A },	//0
	{ 0x33, 0x22 },	//1
	{ 0x20, 0x00 },	//2
	{ 0x24, 0x00 },	//3
	//ADC clock source is chosen by REG42
	{ 0x26, 0x81 },	//4
	//XTI Source, Secondary Bit/LR Clock (SBCK/SLRCK) Frequency Setting
	{ 0x33, 0x22 }, //5
	//CLOCK AUTO
	//# REG. 30h, DIR Recovered System Clock (SCK) Ratio Setting
	{ 0x30, 0x12 },	//6
	//DIT
	{ 0x60, 0x44 }, //7  DITのソースをAUXINに
	{ 0x61, 0x10 }, //8  フォーマットを24bit I2S
	{ 0x62, 0x20 }, //9  Vflag 初期設定はMUTEにしておく
	//INPUT ROUTING
	//MPIO_C
	{ 0x6E, 0x0F }, //10
	{ 0x6F, 0x45 }, //11 MPIOC0-3をDIT入力へ変更
	//OUTPUT
	{ 0x35, 0x0F }, //12 DITの出力をRECOUT0につなげる
	{ 0x78, 0xED }, //13 MPO1 の出力をRECOUT0につなげる
};

static bool pcm9211_init(struct pcm9211_priv *pcm9211)
{
	int ret;

	ret = regmap_multi_reg_write(pcm9211->regmap,
			pcm9211_init_sequence, ARRAY_SIZE(pcm9211_init_sequence));
	if (ret) {
		return false;
	}
	/* Set channel status */
	mutex_lock(&(pcm9211->lock));
	pcm9211_set_channel_status(pcm9211);
	mutex_unlock(&(pcm9211->lock));

	msleep(50);

	return true;
}

static bool pcm9211_check_register(struct pcm9211_priv *pcm9211)
{
	unsigned int value = 0;
	int ret;

	/* reg.0x6Fはwritable設定レジスタなので、regmapキャッシュバイパス関数 */
	/* regcache_cache_bypass() を制御して読む                             */
	/* check reg.0x6F = 0x45 (MPIO_C DIT Standalone Operation) */
	regcache_cache_bypass(pcm9211->regmap, true);
	ret = regmap_read(pcm9211->regmap, 0x6F, &value);
	regcache_cache_bypass(pcm9211->regmap, false);
	if (ret != 0) {
		return false;
	}
	else if (value != 0x45) {
		return false;
	}

	return true;
}

static int pcm9211_probe(struct spi_device *spi, struct regmap *regmap)
{
	struct device *dev = &spi->dev;
	struct pcm9211_priv *pcm9211;
	int ret;

	/* initialize gpio */
	/* gpio_direction_output(GPIO_XRST_18, 1); */		/* ES9017ドライバで実施 */
	/* gpio_direction_output(GPIO_CLK_CLR_18, 1); */	/* ES9017ドライバで実施 */
	gpio_direction_output(GPIO_MCLK_SEL_18, 1);
	gpio_direction_output(GPIO_MCLK_SEL48_18, 1);
	usleep_range(1000, 1000);

	pcm9211 = devm_kzalloc(dev, sizeof(*pcm9211), GFP_KERNEL);
	if (!pcm9211) {
		dev_err(dev, "devm_kzalloc");
		return (-ENOMEM);
	}

	mutex_init(&(pcm9211->lock));
	pcm9211->regmap = regmap;
	pcm9211->fmt |= SND_SOC_DAIFMT_CBS_CFS;
	pcm9211->pcm_format = SNDRV_PCM_FORMAT_S24_LE;
	pcm9211->pcm_rate = 44100;
	pcm9211->copyparam = COPYPARAM_DEFAULT;

	if (!pcm9211_init(pcm9211)) {
		dev_err(dev, "Failed to setup.\n");
		return (-EIO);
	}
	if (!pcm9211_check_register(pcm9211)) {
		dev_err(dev, "Device not found.\n");
		return (-ENXIO);
	}
	dev_set_drvdata(dev, pcm9211);

	ret = snd_soc_register_codec(dev,
			&pcm9211_codec_driver, &pcm9211_dai, 1);
	if (ret !=  0) {
		dev_err(dev, "Failed to register CODEC: %d\n", ret);
		return ret;
	}
	pr_info("%s(): \n" , __func__);

	return 0;
}

static void pcm9211_remove(struct spi_device *spi)
{
	snd_soc_unregister_codec(&spi->dev);
}

static int pcm9211_spi_probe(struct spi_device *spi)
{
	struct regmap *regmap;
	int ret;

	/* chip_select は DeviceTreeで設定する */
	spi->bits_per_word = 8;
	spi->mode = SPI_MODE_0;
	ret = spi_setup(spi);
	if (ret < 0)
		return ret;

	pr_info("pcm9211: cs:%d mode:%04x bits_per_word:%d\n",
			spi->chip_select, spi->mode, spi->bits_per_word);

	regmap = devm_regmap_init_spi(spi, &pcm9211_regmap);
	if (IS_ERR(regmap)) {
		return PTR_ERR(regmap);
	}
	pr_info("%s(): \n" , __func__);

	return pcm9211_probe(spi, regmap);
}

static int pcm9211_spi_remove(struct spi_device *spi)
{
	pcm9211_remove(spi);

	return 0;
}

static const struct spi_device_id pcm9211_spi_id[] = {
	{ "pcm9211", },
	{ },
};
MODULE_DEVICE_TABLE(spi, pcm9211_spi_id);

static const struct of_device_id pcm9211_of_match[] = {
	{ .compatible = "ti,pcm9211", },
	{ }
};
MODULE_DEVICE_TABLE(of, pcm9211_of_match);

static struct spi_driver pcm9211_spi_driver = {
	.driver = {
		.name           = "pcm9211",
		.owner          = THIS_MODULE,
		.of_match_table = pcm9211_of_match,
	},
	.probe    = pcm9211_spi_probe,
	.remove	  = pcm9211_spi_remove,
	.id_table = pcm9211_spi_id,
};
module_spi_driver(pcm9211_spi_driver);


MODULE_DESCRIPTION("PCM9211 codec driver");
MODULE_AUTHOR("AlphaTheta Corp.");
MODULE_LICENSE("GPL");
