/*
 * es9017.c  --  ES9017 ALSA SoC Audio driver
 *
 * Copyright (C) 2023 AlphaTheta Corporation
 */

#include <linux/mutex.h>
#include <linux/init.h>
#include <linux/module.h>
#include <linux/of_device.h>
#include <linux/regmap.h>
#include <linux/regulator/consumer.h>
#include <linux/i2c.h>
#include <linux/gpio.h>
#include <linux/delay.h>
#include <sound/soc.h>
#include <sound/pcm_params.h>
#include <sound/tlv.h>

#include "es9017.h"

#define GPIO_XRST_18		69	// GPIO2_A5 32*2+0+5
#define GPIO_CLK_CLR_18		67	// GPIO2_A3 32*2+0+3
#define GPIO_MCLK_SEL_18	155	// GPIO4_D3 32*4+24+3
#define GPIO_MCLK_SEL48_18	2	// GPIO0_A2 32*0+0+2

/* ES9017 Codec Private Data */
struct es9017_priv {
	struct regmap *regmap;
	unsigned int fmt;
};


/* ES9017 Default Register Value */
static const struct reg_default es9017_reg_defaults[] = {
	{ ES9017_SYSTEM_CONFIG,					0x04 },
	{ ES9017_SYS_MODE_CONFIG,				0x38 },
	{ ES9017_DAC_CLOCK_CONFIG,				0x80 },
	{ ES9017_CLOCK_CONFIG,					0x07 },
	{ ES9017_CONFIG,						0x00 },
	/* Reserved: 0x05 - 0x0F */
	{ ES9017_GPIO1_2_CONFIG,				0x0D },
	{ ES9017_GPIO3_4_CONFIG,				0x00 },
	{ ES9017_GPIO5_6_CONFIG,				0x00 },
	{ ES9017_GPIO7_8_CONFIG,				0x00 },
	{ ES9017_GPIO_OUTPUT_ENABLE,			0x01 },
	{ ES9017_GPIO_INPUT,					0x1C },
	{ ES9017_GPIO_OUTPUT_LOGIC_LOW,			0x07 },
	{ ES9017_GPIO_OUTPUT_LOGIC_HIGH,		0x00 },
	{ ES9017_INPUT_SELECTION,				0x40 },
	{ ES9017_SERIAL_MASTER_ENCODER_CONFIG,	0x01 },
	{ ES9017_TDM_CONFIG,					0x01 },
	{ ES9017_TDM_CONFIG1,					0x00 },
	{ ES9017_TDM_CONFIG2,					0x80 },
	{ ES9017_BCK_WS_MONITOR_CONFIG,			0x30 },
	/* Reserved: 0x1F */
	{ ES9017_TDM_CH1_CONFIG,				0x00 },
	{ ES9017_TDM_CH2_CONFIG,				0x01 },
	{ ES9017_TDM_CH3_CONFIG,				0x20 },
	{ ES9017_TDM_CH4_CONFIG,				0x21 },
	{ ES9017_TDM_CH5_CONFIG,				0x40 },
	{ ES9017_TDM_CH6_CONFIG,				0x41 },
	{ ES9017_TDM_CH7_CONFIG,				0x60 },
	{ ES9017_TDM_CH8_CONFIG,				0x61 },
	{ ES9017_VOLUME1,						0x00 },
	{ ES9017_VOLUME2,						0x00 },
	{ ES9017_VOLUME3,						0x00 },
	{ ES9017_VOLUME4,						0x00 },
	{ ES9017_VOLUME5,						0x00 },
	{ ES9017_VOLUME6,						0x00 },
	{ ES9017_VOLUME7,						0x00 },
	{ ES9017_VOLUME8,						0x00 },
	{ ES9017_DAC_VOL_UP_RATE,				0x04 },
	{ ES9017_DAC_VOL_DOWN_RATE,				0x04 },
	{ ES9017_DAC_VOL_DOWN_RATE_FAST,		0xFF },
	{ ES9017_DAC_MUTE,						0x00 },
	{ ES9017_DAC_INVERT,					0x00 },
	{ ES9017_FILTER_SHAPE,					0x60 },
	{ ES9017_IIR_BANDWIDTH,					0x04 },
	{ ES9017_DAC_PATH_CONFIG,				0x00 },
	{ ES9017_AUTOMUTE_ENABLE,				0xFF },
	{ ES9017_AUTOMUTE_TIME_LOW,				0x0F },
	{ ES9017_AUTOMUTE_TIME_HIGH,			0x08 },
	{ ES9017_AUTOMUTE_LEVEL_LOW,			0x08 },
	{ ES9017_AUTOMUTE_LEVEL_HIGH,			0x00 },
	{ ES9017_AUTOMUTE_OFF_LEVEL_LOW,		0x0A },
	{ ES9017_AUTOMUTE_OFF_LEVEL_HIGH,		0x00 },
	{ ES9017_SOFT_RAMP_CONFIG,				0xC3 }
};

/* regmap */
static const struct regmap_range es9017_readable_ranges[] = {
	regmap_reg_range(ES9017_SYSTEM_CONFIG,
			ES9017_CONFIG),
	regmap_reg_range(ES9017_GPIO1_2_CONFIG,
			ES9017_BCK_WS_MONITOR_CONFIG),
	regmap_reg_range(ES9017_TDM_CH1_CONFIG,
			ES9017_SOFT_RAMP_CONFIG),
	regmap_reg_range(ES9017_SYS_READ,
			ES9017_CHIP_ID_READ),
	regmap_reg_range(ES9017_RATIO_VALID_READ,
			ES9017_INPUT_READBACK),
};

static const struct regmap_access_table es9017_readable_table = {
	.yes_ranges = es9017_readable_ranges,
	.n_yes_ranges = ARRAY_SIZE(es9017_readable_ranges),
};

static const struct regmap_range es9017_writable_ranges[] = {
	regmap_reg_range(ES9017_SYSTEM_CONFIG,
			ES9017_CONFIG),
	regmap_reg_range(ES9017_GPIO1_2_CONFIG,
			ES9017_BCK_WS_MONITOR_CONFIG),
	regmap_reg_range(ES9017_TDM_CH1_CONFIG,
			ES9017_SOFT_RAMP_CONFIG),
};

static const struct regmap_access_table es9017_writable_table = {
	.yes_ranges = es9017_writable_ranges,
	.n_yes_ranges = ARRAY_SIZE(es9017_writable_ranges),
};

static const struct regmap_range es9017_volatile_ranges[] = {
	regmap_reg_range(ES9017_SYSTEM_CONFIG,
			ES9017_SYSTEM_CONFIG),
	regmap_reg_range(ES9017_SYS_READ,
			ES9017_CHIP_ID_READ),
	regmap_reg_range(ES9017_RATIO_VALID_READ,
			ES9017_INPUT_READBACK),
};

static const struct regmap_access_table es9017_volatile_table = {
	.yes_ranges = es9017_volatile_ranges,
	.n_yes_ranges = ARRAY_SIZE(es9017_volatile_ranges),
};

static const struct regmap_config es9017_regmap = {
	.reg_bits = 8,
	.val_bits = 8,
	.cache_type = REGCACHE_RBTREE,
	.max_register = ES9017_INPUT_READBACK,

	.reg_defaults = es9017_reg_defaults,
	.num_reg_defaults = ARRAY_SIZE(es9017_reg_defaults),

	.rd_table = &es9017_readable_table,
	.wr_table = &es9017_writable_table,
	.volatile_table = &es9017_volatile_table,
};


static struct snd_soc_codec_driver es9017_codec_driver;


static const uint32_t es9017_dai_rates_slave[] = {
	44100, 48000, 88200, 96000
};

static const struct snd_pcm_hw_constraint_list es9017_constraint_list_rates = {
	.list  = es9017_dai_rates_slave,
	.count = ARRAY_SIZE(es9017_dai_rates_slave),
};


static int es9017_dai_startup_slave(
		struct snd_pcm_substream *substream, struct snd_soc_dai *dai)
{
	struct snd_soc_codec *codec = dai->codec;
	int ret;
	u64 formats = SNDRV_PCM_FMTBIT_S16_LE
				| SNDRV_PCM_FMTBIT_S24_LE
				| SNDRV_PCM_FMTBIT_S32_LE;

	ret = snd_pcm_hw_constraint_list(substream->runtime, 0,
			SNDRV_PCM_HW_PARAM_RATE, &es9017_constraint_list_rates);
	if (ret != 0) {
		dev_err(codec->dev, "Failed to setup rates constraints: %d\n", ret);
	}

	ret = snd_pcm_hw_constraint_mask64(substream->runtime,
			SNDRV_PCM_HW_PARAM_FORMAT, formats);
	if (ret != 0) {
		dev_err(codec->dev, "Failed to setup formats constraints: %d\n", ret);
	}

	return ret;
}

static int es9017_dai_startup(
		struct snd_pcm_substream *substream, struct snd_soc_dai *dai)
{
	struct snd_soc_codec *codec = dai->codec;
	struct es9017_priv *es9017
					= snd_soc_codec_get_drvdata(codec);

	switch (es9017->fmt & SND_SOC_DAIFMT_MASTER_MASK) {
	case SND_SOC_DAIFMT_CBS_CFS:
		return es9017_dai_startup_slave(substream, dai);

	default:
		return (-EINVAL);
	}
}

static int es9017_hw_params(
		struct snd_pcm_substream *substream, struct snd_pcm_hw_params *params,
		struct snd_soc_dai *dai)
{
	struct snd_soc_codec *codec = dai->codec;

	pr_debug("%s(): rate=%d \n" , __func__, params_rate(params));

	switch (params_format(params)) {
	case SNDRV_PCM_FORMAT_S16_LE:
	case SNDRV_PCM_FORMAT_S24_LE:
	case SNDRV_PCM_FORMAT_S32_LE:
		break;
	default:
		pr_err("[%s] invalid format\n", __func__);
		return -EINVAL;
	}

	/* disable DAC data path */
	snd_soc_update_bits(codec, ES9017_SYSTEM_CONFIG,
			ES9017_SYSTEM_CONFIG_DAC_MODE_REG, 0);

	pr_debug("%s(): rate=%d output \n" , __func__, params_rate(params));
	switch (params_rate(params)) {
	case 44100:
		gpio_set_value(GPIO_MCLK_SEL_18, 0);
		gpio_set_value(GPIO_MCLK_SEL48_18, 0);
		break;
	case 48000:
		gpio_set_value(GPIO_MCLK_SEL_18, 1);
		gpio_set_value(GPIO_MCLK_SEL48_18, 0);
		break;
	case 88200:
		gpio_set_value(GPIO_MCLK_SEL_18, 0);		// dummy
		gpio_set_value(GPIO_MCLK_SEL48_18, 1);		// dummy
		break;
	case 96000:
		gpio_set_value(GPIO_MCLK_SEL_18, 1);
		gpio_set_value(GPIO_MCLK_SEL48_18, 1);
		break;
	default:
		pr_err("[%s] unsupported sampling rate\n", __func__);
		return -EINVAL;
	}

	/* enable DAC data path */
	snd_soc_update_bits(codec, ES9017_SYSTEM_CONFIG,
			ES9017_SYSTEM_CONFIG_DAC_MODE_REG,
			ES9017_SYSTEM_CONFIG_DAC_MODE_REG);

	return 0;
}

static void es9017_dai_shutdown(
		struct snd_pcm_substream *substream, struct snd_soc_dai *dai)
{
	struct snd_soc_codec *codec = dai->codec;
	struct es9017_priv *es9017
					= snd_soc_codec_get_drvdata(codec);

	if (es9017 == NULL) {
		pr_err("[%s] invalid device private data\n", __func__);
		return;
	}
}

static int es9017_digital_mute(struct snd_soc_dai *dai, int mute)
{
	struct snd_soc_codec *codec = dai->codec;
	u8 mute_reg;

	/* MUTE/UNMUTE all channels */
	mute_reg = (mute) ? 0xFFU : 0x00U;
	return snd_soc_write(codec, ES9017_DAC_MUTE, mute_reg);
}

static const struct snd_soc_dai_ops es9017_dai_ops = {
	.startup = es9017_dai_startup,
	.shutdown = es9017_dai_shutdown,
	.hw_params = es9017_hw_params,
	.digital_mute = es9017_digital_mute,
};


#define ES9017_RATES (SNDRV_PCM_RATE_44100 | \
					  SNDRV_PCM_RATE_48000 | \
					  SNDRV_PCM_RATE_96000)

#define ES9017_FORMATS (SNDRV_PCM_FMTBIT_S16_LE | \
						SNDRV_PCM_FMTBIT_S24_LE | \
						SNDRV_PCM_FMTBIT_S32_LE)

static struct snd_soc_dai_driver es9017_dai = {
	.name = "es9017-dai",
	.playback = {
		.stream_name  = "Playback",
		.channels_min = 2,
		.channels_max = 2,
		.rates        = ES9017_RATES,
		.formats      = ES9017_FORMATS,
	},
	.ops = &es9017_dai_ops,
};

static bool es9017_init(struct es9017_priv *es9017)
{
	struct regmap *regmap = es9017->regmap;
	int ret;
	const unsigned int mask =
			  ES9017_TDM_CHn_CONFIG_LINE_SEL_MASK	/* LINE MASK */
			| ES9017_TDM_CHn_CONFIG_SLOT_SEL_MASK;	/* SLOT MASK */
	unsigned int val;

	/* CH[1,3,5,7] = LEFT */
	val = ES9017_TDM_CHn_CONFIG_LINE_SEL_DATA2		/* LINE = DATA 2 */
		| ES9017_TDM_CHn_CONFIG_SLOT_SEL_SLOT(0U);	/* SLOT = SLOT 1 (L DATA) */
	ret = regmap_update_bits(regmap, ES9017_TDM_CH1_CONFIG, mask, val);
	ret |= regmap_update_bits(regmap, ES9017_TDM_CH3_CONFIG, mask, val);
	ret |= regmap_update_bits(regmap, ES9017_TDM_CH5_CONFIG, mask, val);
	ret |= regmap_update_bits(regmap, ES9017_TDM_CH7_CONFIG, mask, val);
	if (ret) {
		return false;
	}

	/* CH[2,4,6,8] = RIGHT */
	val = ES9017_TDM_CHn_CONFIG_LINE_SEL_DATA2		/* LINE = DATA 2 */
		| ES9017_TDM_CHn_CONFIG_SLOT_SEL_SLOT(1U);	/* SLOT = SLOT 2 (R DATA) */
	ret = regmap_update_bits(regmap, ES9017_TDM_CH2_CONFIG, mask, val);
	ret |= regmap_update_bits(regmap, ES9017_TDM_CH4_CONFIG, mask, val);
	ret |= regmap_update_bits(regmap, ES9017_TDM_CH6_CONFIG, mask, val);
	ret |= regmap_update_bits(regmap, ES9017_TDM_CH8_CONFIG, mask, val);
	if (ret) {
		return false;
	}

	/* disable AUTOMUTE */
	ret = regmap_write(regmap, ES9017_AUTOMUTE_ENABLE, 0x00);
	if (ret) {
		return false;
	}

	/* set PCM filter */
	ret = regmap_update_bits(regmap, ES9017_FILTER_SHAPE, ES9017_FILTER_SHAPE_MASK,
			ES9017_FILTER_SHAPE_MINIMUM_PHASE_FAST_ROLLOFF);
	if (ret) {
		return false;
	}

	return true;
}

static bool es9017_check_chip_id(struct es9017_priv *es9017)
{
	unsigned int value = 0;
	int ret;

	/* ES9017_CHIP_ID_READはvolatile設定済みレジスタだけど、 */
	/* regcache_cache_bypass()を制御しておく                 */
	/* check ES9017 ID = 0x60 */
	regcache_cache_bypass(es9017->regmap, true);
	ret = regmap_read(es9017->regmap, ES9017_CHIP_ID_READ, &value);
	regcache_cache_bypass(es9017->regmap, false);
	if (ret != 0) {
		return false;
	}
	else if (value != ES9017_CHIP_ID) {
		return false;
	}

	return true;
}

static int es9017_probe(struct device *dev, struct regmap *regmap)
{
	struct es9017_priv *es9017;
	int ret;

	/* initialize gpio */
	gpio_direction_output(GPIO_XRST_18, 1);
	gpio_direction_output(GPIO_CLK_CLR_18, 1);
	gpio_direction_output(GPIO_MCLK_SEL_18, 1);
	gpio_direction_output(GPIO_MCLK_SEL48_18, 1);
	usleep_range(1000, 1000);

	es9017 = devm_kzalloc(dev, sizeof(*es9017), GFP_KERNEL);
	if (!es9017) {
		dev_err(dev, "devm_kzalloc");
		return (-ENOMEM);
	}

	es9017->regmap = regmap;
	es9017->fmt |= SND_SOC_DAIFMT_CBS_CFS;

	if (!es9017_check_chip_id(es9017)) {
		dev_err(dev, "Device not found.\n");
		return (-ENXIO);
	}

	if (!es9017_init(es9017)) {
		dev_err(dev, "Failed to setup.\n");
		return (-EIO);
	}
	dev_set_drvdata(dev, es9017);

	ret = snd_soc_register_codec(dev,
			&es9017_codec_driver, &es9017_dai, 1);
	if (ret != 0) {
		dev_err(dev, "Failed to register CODEC: %d\n", ret);
		return ret;
	}
	pr_info("%s(): \n" , __func__);

	return 0;
}

static void es9017_remove(struct device *dev)
{
	snd_soc_unregister_codec(dev);
}

static int es9017_i2c_probe(
		struct i2c_client *i2c, const struct i2c_device_id *id)
{
	struct regmap *regmap;

	pr_info("es9017: addr:0x%02X\n", i2c->addr);

	regmap = devm_regmap_init_i2c(i2c, &es9017_regmap);
	if (IS_ERR(regmap)) {
		return PTR_ERR(regmap);
	}
	pr_info("%s(): \n" , __func__);

	return es9017_probe(&i2c->dev, regmap);
}

static int es9017_i2c_remove(struct i2c_client *i2c)
{
	es9017_remove(&i2c->dev);

	return 0;
}

static const struct i2c_device_id es9017_i2c_id[] = {
	{ "es9017", },
	{ }
};
MODULE_DEVICE_TABLE(i2c, es9017_i2c_id);

static const struct of_device_id es9017_of_match[] = {
	{ .compatible = "ess,es9017", },
	{ }
};
MODULE_DEVICE_TABLE(of, es9017_of_match);

static struct i2c_driver es9017_i2c_driver = {
	.driver = {
		.name           = "es9017",
		.owner          = THIS_MODULE,
		.of_match_table = of_match_ptr(es9017_of_match),
	},
	.probe    = es9017_i2c_probe,
	.remove   = es9017_i2c_remove,
	.id_table = es9017_i2c_id,
};
module_i2c_driver(es9017_i2c_driver);


MODULE_DESCRIPTION("ES9017 codec driver");
MODULE_AUTHOR("AlphaTheta Corp.");
MODULE_LICENSE("GPL");
