/*
 * Driver for the ESS SABRE9016K2M
 *
 * Author: Satoru Kawase, Takahito Nishiara
 *      Copyright 2016
 * Modified by: AlphaTheta Corp.
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * version 2 as published by the Free Software Foundation.
 *
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 */

#include <linux/init.h>
#include <linux/module.h>
#include <linux/regmap.h>
#include <linux/i2c.h>
#include <linux/gpio.h>
#include <linux/delay.h>
#include <sound/soc.h>
#include <sound/pcm_params.h>
#include <sound/tlv.h>

#include "es9016k2m.h"

#define NEW_MCLK_SEL

#define GPIO2_A5_XRST_18	69	// 64+0+5
#define GPIO2_A3_CLK_CLR_18	67	// 64+0+3
#define GPIO4_D3_MCLK_SEL_18	155	// 128+24+3
#define GPIO2_B0_MCLK_SEL48_18	72	// 64+8+0

/* ES9016K2M Codec Private Data */
struct es9016k2m_priv {
	struct regmap *regmap;
	unsigned int fmt;
};


/* ES9016K2M Default Register Value */
static const struct reg_default es9016k2m_reg_defaults[] = {
	{ 0, 0x00 },
	{ 1, 0x8c },
	{ 4, 0x00 },
	{ 5, 0x68 },
	{ 6, 0x4a },
	{ 7, 0x80 },
	{ 8, 0x10 },
	{ 9, 0x00 },
	{ 10,0x05 },
	{ 11,0x02 },
	{ 12,0x5a },
	{ 13,0x40 },
	{ 14,0x8a },
	{ 15,0x00 },
	{ 16,0x00 },
	{ 17,0xff },
	{ 18,0xff },
	{ 19,0xff },
	{ 20,0x7f },
	{ 21,0x00 },
	{ 22,0x00 },
	{ 23,0x00 },
	{ 24,0x00 },
	{ 25,0x00 },
	{ 26,0x00 },
	{ 27,0x00 },
	{ 28,0x00 },
	{ 29,0x00 },
	{ 30,0x00 },
};


static bool es9016k2m_writeable(struct device *dev, unsigned int reg)
{
	if (reg > ES9016K2M_CACHEREGNUM)
		return false;
	else if (reg == 0x2 || reg == 0x3)
		return false;
	else
		return true;
}

static bool es9016k2m_readable(struct device *dev, unsigned int reg)
{
	if (reg <= ES9016K2M_CACHEREGNUM && reg != 2 && reg !=3)
		return true;
	else if (64 <= reg && reg <= 69)
		return true;
	else if (70 <= reg && reg <= 93)
		return true;
	else
		return false;
}

static bool es9016k2m_volatile(struct device *dev, unsigned int reg)
{
	if (reg == 64 || reg == 65)
		return true;
	else
		return false;
}


static const uint32_t es9016k2m_dai_rates_master[] = {
	44100, 48000, 88200, 96000, 176400, 192000
};

static const struct snd_pcm_hw_constraint_list constraints_master = {
	.list  = es9016k2m_dai_rates_master,
	.count = ARRAY_SIZE(es9016k2m_dai_rates_master),
};

static const uint32_t es9016k2m_dai_rates_slave[] = {
	8000, 11025, 16000, 22050, 32000,
	44100, 48000, 64000, 88200, 96000, 176400, 192000
};

static const struct snd_pcm_hw_constraint_list constraints_slave = {
	.list  = es9016k2m_dai_rates_slave,
	.count = ARRAY_SIZE(es9016k2m_dai_rates_slave),
};

static int es9016k2m_dai_startup_master(
		struct snd_pcm_substream *substream, struct snd_soc_dai *dai)
{
	struct snd_soc_codec *codec = dai->codec;
	int ret;

	ret = snd_pcm_hw_constraint_list(substream->runtime,
			0, SNDRV_PCM_HW_PARAM_RATE, &constraints_master);
	if (ret != 0) {
		dev_err(codec->dev, "Failed to setup rates constraints: %d\n", ret);
		return ret;
	}

	ret = snd_pcm_hw_constraint_mask64(substream->runtime,
			SNDRV_PCM_HW_PARAM_FORMAT, SNDRV_PCM_FMTBIT_S32_LE);
	if (ret != 0) {
		dev_err(codec->dev, "Failed to setup format constraints: %d\n", ret);
	}

	return ret;
}

static int es9016k2m_dai_startup_slave(
		struct snd_pcm_substream *substream, struct snd_soc_dai *dai)
{
	struct snd_soc_codec *codec = dai->codec;
	int ret;

	ret = snd_pcm_hw_constraint_list(substream->runtime,
			0, SNDRV_PCM_HW_PARAM_RATE, &constraints_slave);
	if (ret != 0) {
		dev_err(codec->dev, "Failed to setup rates constraints: %d\n", ret);
	}

	return ret;
}

static int es9016k2m_dai_startup(
		struct snd_pcm_substream *substream, struct snd_soc_dai *dai)
{
	struct snd_soc_codec     *codec = dai->codec;
	struct es9016k2m_priv *es9016k2m
					= snd_soc_codec_get_drvdata(codec);

	switch (es9016k2m->fmt & SND_SOC_DAIFMT_MASTER_MASK) {
	case SND_SOC_DAIFMT_CBM_CFM:
		return es9016k2m_dai_startup_master(substream, dai);

	case SND_SOC_DAIFMT_CBS_CFS:
		return es9016k2m_dai_startup_slave(substream, dai);

	default:
		return (-EINVAL);
	}
}

static int es9016k2m_hw_params(
	struct snd_pcm_substream *substream, struct snd_pcm_hw_params *params,
	struct snd_soc_dai *dai)
{
	uint8_t iface;
	struct snd_soc_codec *codec = dai->codec;

	pr_debug("%s(): rate=%d \n" , __func__, params_rate(params));

	iface = snd_soc_read(codec, ES9016K2M_CHANNEL_MAPPING);
	snd_soc_write(codec, ES9016K2M_CHANNEL_MAPPING, iface | 0x04);

	iface = snd_soc_read(codec, ES9016K2M_GENERAL_SETTINGS) & 0x9f;
	snd_soc_write(codec, ES9016K2M_GENERAL_SETTINGS, iface | 0x40);

	iface = snd_soc_read(codec, ES9016K2M_INPUT_CONFIGURATION) & 0x3f;
	switch (params_format(params)) {
	case SNDRV_PCM_FORMAT_S16_LE:
	case SNDRV_PCM_FORMAT_S24_LE:
	case SNDRV_PCM_FORMAT_S32_LE:
		iface |= 0x80;
		break;
	default:
		pr_err("[%s] invalid format\n", __func__);
		return -EINVAL;
	}
	snd_soc_write(codec, ES9016K2M_INPUT_CONFIGURATION, iface);
	pr_debug("%s(): rate=%d output \n" , __func__, params_rate(params));
	switch (params_rate(params)) {
	case 44100:
		gpio_direction_output(GPIO4_D3_MCLK_SEL_18, 0);
		gpio_direction_output(GPIO2_B0_MCLK_SEL48_18, 0);
		break;
	case 48000:
#ifdef NEW_MCLK_SEL
		gpio_direction_output(GPIO4_D3_MCLK_SEL_18, 1);
		gpio_direction_output(GPIO2_B0_MCLK_SEL48_18, 0);
#else
		gpio_direction_output(GPIO4_D3_MCLK_SEL_18, 0);
		gpio_direction_output(GPIO2_B0_MCLK_SEL48_18, 1);
#endif
		break;
	case 88200:
#ifdef NEW_MCLK_SEL
		gpio_direction_output(GPIO4_D3_MCLK_SEL_18, 0);		// dummy
		gpio_direction_output(GPIO2_B0_MCLK_SEL48_18, 1);	// dummy
#else
		gpio_direction_output(GPIO4_D3_MCLK_SEL_18, 1);		// dummy
		gpio_direction_output(GPIO2_B0_MCLK_SEL48_18, 1);	// dummy
#endif
		break;
	case 96000:
#ifdef NEW_MCLK_SEL
		gpio_direction_output(GPIO4_D3_MCLK_SEL_18, 1);
		gpio_direction_output(GPIO2_B0_MCLK_SEL48_18, 1);
#else
		gpio_direction_output(GPIO4_D3_MCLK_SEL_18, 1);
		gpio_direction_output(GPIO2_B0_MCLK_SEL48_18, 0);
#endif
		break;
	default:
		pr_err("[%s] unsupported sampling rate\n", __func__);
		return -EINVAL;
	}

	return 0;
}

static void es9016k2m_dai_shutdown(
		struct snd_pcm_substream *substream, struct snd_soc_dai *dai)
{
	struct snd_soc_codec *codec = dai->codec;
	struct es9016k2m_priv *es9016k2m
					= snd_soc_codec_get_drvdata(codec);

	if (es9016k2m == NULL) {
		pr_err("[%s] invalid device private data\n", __func__);
		return;
	}
}

static const struct snd_soc_dai_ops es9016k2m_dai_ops = {
	.startup = es9016k2m_dai_startup,
	.shutdown = es9016k2m_dai_shutdown,
	.hw_params = es9016k2m_hw_params,
};


#if 1
#define ES9016K2M_RATES (SNDRV_PCM_RATE_44100 | \
						SNDRV_PCM_RATE_48000 | \
						SNDRV_PCM_RATE_96000)

#define ES9016K2M_FORMATS (SNDRV_PCM_FMTBIT_S24_LE)
#else
#define ES9016K2M_RATES (SNDRV_PCM_RATE_8000_44100 | \
						SNDRV_PCM_RATE_48000 | \
						SNDRV_PCM_RATE_64000 | \
						SNDRV_PCM_RATE_88200 | \
						SNDRV_PCM_RATE_96000 | \
		                SNDRV_PCM_RATE_176400| \
						SNDRV_PCM_RATE_192000)

#define ES9016K2M_FORMATS (SNDRV_PCM_FMTBIT_S16_LE | \
						SNDRV_PCM_FMTBIT_S24_LE | \
		    			SNDRV_PCM_FMTBIT_S32_LE)
#endif

static struct snd_soc_dai_driver es9016k2m_dai = {
	.name = "es9016k2m-dai",
	.playback = {
		.stream_name  = "Playback",
		.channels_min = 2,
		.channels_max = 2,
		.rates        = ES9016K2M_RATES,
		.formats      = ES9016K2M_FORMATS,
	},
	.ops = &es9016k2m_dai_ops,
};

static unsigned int es9016k2m_soc_i2c_read(struct snd_soc_codec *codec, unsigned int reg)
{
	unsigned int val = 0;
	struct es9016k2m_priv *es9016k2m = snd_soc_codec_get_drvdata(codec);

	if (es9016k2m == NULL) {
		return -ENODEV;
	}

	regmap_read(es9016k2m->regmap, reg, &val);
	return val;
}

static int es9016k2m_soc_i2c_write(struct snd_soc_codec *codec, unsigned int reg, unsigned int value)
{
	struct es9016k2m_priv *es9016k2m = snd_soc_codec_get_drvdata(codec);

	if (es9016k2m == NULL) {
		return -ENODEV;
	}
	return(regmap_write(es9016k2m->regmap, reg, value));
}

static struct snd_soc_codec_driver es9016k2m_codec_driver = {
	.read = es9016k2m_soc_i2c_read,
	.write = es9016k2m_soc_i2c_write,
};


static const struct regmap_config es9016k2m_regmap = {
	.reg_bits         = 8,
	.val_bits         = 8,
	.max_register     = 93,

	.reg_defaults     = es9016k2m_reg_defaults,
	.num_reg_defaults = ARRAY_SIZE(es9016k2m_reg_defaults),

	.writeable_reg    = es9016k2m_writeable,
	.readable_reg     = es9016k2m_readable,
	.volatile_reg     = es9016k2m_volatile,

	.cache_type       = REGCACHE_RBTREE,
};


static bool es9016k2m_check_chip_id(struct es9016k2m_priv *es9016k2m)
{
	unsigned int value = 0;
	int ret;

	ret = regmap_read(es9016k2m->regmap, ES9016K2M_CHIP_STATUS, &value);
	if (ret != 0) {
		return false;
	}
	/* ES9016K2M ID = 6 */
	if (((value & 0x1C) >> 2) != 6) {
		return false;
	}

	return true;
}


static int es9016k2m_probe(struct device *dev, struct regmap *regmap)
{
	struct es9016k2m_priv *es9016k2m;
	int ret;

	/* initialize gpio */
	gpio_direction_output(GPIO2_A5_XRST_18, 1);
	gpio_direction_output(GPIO2_A3_CLK_CLR_18, 1);
#ifdef NEW_MCLK_SEL
	gpio_direction_output(GPIO4_D3_MCLK_SEL_18, 1);
	gpio_direction_output(GPIO2_B0_MCLK_SEL48_18, 1);
#else
	gpio_direction_output(GPIO4_D3_MCLK_SEL_18, 1);
	gpio_direction_output(GPIO2_B0_MCLK_SEL48_18, 0);
#endif
	usleep_range(1000, 1000);

	es9016k2m = devm_kzalloc(dev, sizeof(*es9016k2m), GFP_KERNEL);
	if (!es9016k2m) {
		dev_err(dev, "devm_kzalloc");
		return (-ENOMEM);
	}

	es9016k2m->regmap = regmap;
	es9016k2m->fmt |= SND_SOC_DAIFMT_CBS_CFS;

	if (!es9016k2m_check_chip_id(es9016k2m)) {
		dev_err(dev, "Device not found.\n");
		return (-EINVAL);
	}

	dev_set_drvdata(dev, es9016k2m);

	ret = snd_soc_register_codec(dev,
			&es9016k2m_codec_driver, &es9016k2m_dai, 1);
	if (ret != 0) {
		dev_err(dev, "Failed to register CODEC: %d\n", ret);
		return ret;
	}
	pr_info("%s(): \n" , __func__);

	return 0;
}

static void es9016k2m_remove(struct device *dev)
{
	snd_soc_unregister_codec(dev);
}


static int es9016k2m_i2c_probe(
		struct i2c_client *i2c, const struct i2c_device_id *id)
{
	struct regmap *regmap;

	regmap = devm_regmap_init_i2c(i2c, &es9016k2m_regmap);
	if (IS_ERR(regmap)) {
		return PTR_ERR(regmap);
	}
	pr_info("%s(): \n" , __func__);

	return es9016k2m_probe(&i2c->dev, regmap);
}

static int es9016k2m_i2c_remove(struct i2c_client *i2c)
{
	es9016k2m_remove(&i2c->dev);

	return 0;
}


static const struct i2c_device_id es9016k2m_i2c_id[] = {
	{ "es9016k2m", },
	{ }
};
MODULE_DEVICE_TABLE(i2c, es9016k2m_i2c_id);

static const struct of_device_id es9016k2m_of_match[] = {
	{ .compatible = "ess,es9016k2m", },
	{ }
};
MODULE_DEVICE_TABLE(of, es9016k2m_of_match);

static struct i2c_driver es9016k2m_i2c_driver = {
	.driver = {
		.name           = "es9016k2m-i2c",
		.owner          = THIS_MODULE,
		.of_match_table = of_match_ptr(es9016k2m_of_match),
	},
	.probe    = es9016k2m_i2c_probe,
	.remove   = es9016k2m_i2c_remove,
	.id_table = es9016k2m_i2c_id,
};
module_i2c_driver(es9016k2m_i2c_driver);


MODULE_DESCRIPTION("ES9016K2M codec driver");
MODULE_AUTHOR("AlphaTheta Corp.");
MODULE_LICENSE("GPL");
