/*
 * DIT4192 ASoC codec driver
 *
 * Based on dit4192.c Kevin Huang <ziqiangh@lab126.com>
 *
 * This program is free software; you can redistribute it and/or modify it
 * under the terms and conditions of the GNU General Public License,
 * version 2, as published by the Free Software Foundation.
 */

#include <linux/init.h>
#include <linux/module.h>
#include <linux/spi/spi.h>
#include <linux/gpio.h>
#include <linux/delay.h>
#include <linux/mutex.h>

#include <sound/pcm_params.h>
#include <sound/soc.h>

#include "dit4192.h"

#define NEW_MCLK_SEL

//#define DEBUG_CHANNEL_STATUS

#if 1
#define CATEGORY_CODE				(0x80)	/* cd player */

#define DIT4192_RATES (SNDRV_PCM_RATE_44100 | \
					SNDRV_PCM_RATE_48000 | \
					SNDRV_PCM_RATE_96000)

#define DIT4192_FORMATS (SNDRV_PCM_FMTBIT_S24_LE)
#else
#define CATEGORY_CODE				(0x80)	/* cd player */
//#define CATEGORY_CODE				(0x48)	/* digital signal mixer */
//#define CATEGORY_CODE				(0x44)	/* digital sound sampler */
//#define CATEGORY_CODE				(0x54)	/* digital sound processor */

#define DIT4192_RATES (SNDRV_PCM_RATE_44100 | \
					SNDRV_PCM_RATE_48000 | \
					SNDRV_PCM_RATE_88200 | \
					SNDRV_PCM_RATE_96000)

#define DIT4192_FORMATS (SNDRV_PCM_FMTBIT_S16_LE |\
					SNDRV_PCM_FMTBIT_S24_LE |\
					SNDRV_PCM_FMTBIT_S20_3LE)
#endif

#define GPIO4_D3_MCLK_SEL_18	155	// 128+24+3
#define GPIO2_B0_MCLK_SEL48_18	72	// 64+8+0


#define CATEGORY_LASEROPT_MASK		(0b11100000)
#define CATEGORY_LASEROPT_VALUE		(0b10000000)
#define CATEGORY_BROADCAST_MASK1	(0b11100000)
#define CATEGORY_BROADCAST_VALUE1	(0b00100000)
#define CATEGORY_BROADCAST_MASK2	(0b11110000)
#define CATEGORY_BROADCAST_VALUE2	(0b01110000)
#define COPYPARAM_FREE		(0)
#define COPYPARAM_ONCE		(1)
#define COPYPARAM_NEVER		(2)

struct dit4192_stream {
	struct snd_pcm_substream *substream;
	char hw_status[CHANNEL_STATUS_SIZE];		/* hardware status */
	char def_status[CHANNEL_STATUS_SIZE];		/* default status */
	char pcm_status[CHANNEL_STATUS_SIZE];		/* PCM private status */
	char hw_udata[32];
	struct snd_kcontrol *pcm_ctl;
};

struct dit4192 {
	struct mutex lock;
	struct spi_device *spi;
	struct dit4192_stream playback;
	unsigned int rate;
	int copyparam;
};

static void dit4192_spi_completion_cb(void *arg)
{
	complete(arg);
}

static int dit4192_spi_write_device(struct dit4192 *dit4192, u8 reg, int bytes, u8 *data)
{
	int ret;
	u8 spi_data[DIT4192_MAX_DATA_SIZE];
	struct spi_transfer spi_transfer;
	struct spi_message spi_message;
	DECLARE_COMPLETION_ONSTACK(context);

	pr_debug("%s(): [1]\n" , __func__);

	if (bytes > 0) {
		if (bytes <= (DIT4192_MAX_DATA_SIZE-2)) {
			memcpy(&spi_data[2], data, bytes);
		} else {
			/* This should never happen. */
			pr_err("%s: SPI transfer error. Bad data size\n", __func__);
			return -EINVAL;
		}
	}

	memset(&spi_transfer, 0, sizeof(struct spi_transfer));
	memset(&spi_message, 0, sizeof(struct spi_message));

	spi_data[DIT4192_HEADER_0] = DIT4192_CMD_W | DIT4192_IO_STEP_1 | reg;
	spi_data[DIT4192_HEADER_1] = 0;

	spi_transfer.tx_buf = spi_data;
	spi_transfer.len = bytes+2;

	spi_setup(dit4192->spi);
	spi_message_init(&spi_message);
	spi_message_add_tail(&spi_transfer, &spi_message);
	spi_message.complete = dit4192_spi_completion_cb;
	spi_message.context = &context;

	/* must use spi_async in a context that may sleep */
	ret = spi_async(dit4192->spi, &spi_message);
	if (ret == 0) {
		wait_for_completion(&context);
		/* update ret to contain the number of bytes actually written */
		if (spi_message.status == 0) {
			ret = spi_transfer.len;
		} else {
			pr_err("%s: SPI transfer error, spi_message.status = %d\n", 
				__func__, spi_message.status);
		}
	} else {
		pr_err("%s: Error calling spi_async, ret = %d\n", __func__, ret);
	}

	pr_debug("%s(): [2]\n" , __func__);

	return ret;
}

static int dit4192_spi_read_device(struct dit4192 *dit4192, u8 reg, int bytes, u8 *buf)
{
	int ret;
	unsigned char header[2];
	struct spi_transfer spi_transfer_w;
	struct spi_transfer spi_transfer_r;
	struct spi_message spi_message;
	DECLARE_COMPLETION_ONSTACK(context);

	memset(&spi_transfer_w, 0, sizeof(struct spi_transfer));
	memset(&spi_transfer_r, 0, sizeof(struct spi_transfer));
	memset(&spi_message, 0, sizeof(struct spi_message));

	spi_setup(dit4192->spi);
	spi_message_init(&spi_message);

	header[DIT4192_HEADER_0] = DIT4192_CMD_R | DIT4192_IO_STEP_1 | reg; //0x80
	header[DIT4192_HEADER_1] = 0;

	spi_transfer_w.tx_buf = header;
	spi_transfer_w.len =  2;
	spi_message_add_tail(&spi_transfer_w, &spi_message);

	spi_transfer_r.rx_buf = buf;
	spi_transfer_r.len =  bytes;
	spi_message_add_tail(&spi_transfer_r, &spi_message);

	spi_message.complete = dit4192_spi_completion_cb;
	spi_message.context = &context;

	/* must use spi_async in a context that may sleep */
	ret = spi_async(dit4192->spi, &spi_message);
	if (ret == 0) {
		wait_for_completion(&context);

		if (spi_message.status == 0) {
			/* spi_message.actual_length should contain the number
			* of bytes actually read and should update ret to be
			* the actual length, but since our driver doesn't
			* support this, assume all count bytes were read.
			*/
			ret = bytes;
		}

		if (ret > 0) {
			ret = -EFAULT;
		}
	} else {
		pr_err("%s: Error calling spi_async, ret = %d\n", __func__, ret);
	}

	return ret;
}

static int dit4192_startup(struct snd_pcm_substream *substream, struct snd_soc_dai *dai)
{
	struct dit4192 *chip = dev_get_drvdata(dai->codec->dev);

	if (chip == NULL) {
		pr_err("[%s] invalid device private data\n", __func__);
		return -ENODEV;
	}
	return 0;
}

static void dit4192_reg_dump(struct dit4192 *dit4192, u8 reg_start, int size);
static void dit4192_set_channel_status(struct dit4192 *chip)
{
	u8 spi_data[DIT4192_MAX_DATA_SIZE];

	/* Set BTD 1. */
	spi_data[0] = DIT4192_BIT_BTD_ON;
	dit4192_spi_write_device(chip, DIT4192_REG_7, 1, spi_data);

	/* Set channel status */
	spi_data[0] = 0; /* bit0-7 */
	spi_data[1] = 0; /* bit0-7 */
	spi_data[2] = CATEGORY_CODE; /* bit8-15 */
	spi_data[3] = CATEGORY_CODE; /* bit8-15 */
	spi_data[4] = 0; /* bit16-23 */
	spi_data[5] = 0; /* bit16-23 */

	if (((CATEGORY_CODE & CATEGORY_LASEROPT_MASK) == CATEGORY_LASEROPT_VALUE)
	||  ((CATEGORY_CODE & CATEGORY_BROADCAST_MASK1) == CATEGORY_BROADCAST_VALUE1)
	||  ((CATEGORY_CODE & CATEGORY_BROADCAST_MASK2) == CATEGORY_BROADCAST_VALUE2)) {
		switch (chip->copyparam) {
		case COPYPARAM_FREE:	/* copy free */
			spi_data[0] |= 0x20; /* bit0-7 */	/* COPY=1 */
			spi_data[1] |= 0x20; /* bit0-7 */	/* COPY=1 */
			spi_data[2] |= 0x01; /* bit8-15 */	/* L=1 */
			spi_data[3] |= 0x01; /* bit8-15 */	/* L=1 */
			break;
		case COPYPARAM_ONCE:	/* copy once */
			spi_data[0] &= 0xDF; /* bit0-7 */	/* COPY=0 */
			spi_data[1] &= 0xDF; /* bit0-7 */	/* COPY=0 */
			spi_data[2] &= 0xFE; /* bit8-15 */	/* L=0 */
			spi_data[3] &= 0xFE; /* bit8-15 */	/* L=0 */
			break;
		case COPYPARAM_NEVER:	/* copy never */
		default:				/* parameter NG */
			spi_data[0] &= 0xDF; /* bit0-7 */	/* COPY=0 */
			spi_data[1] &= 0xDF; /* bit0-7 */	/* COPY=0 */
			spi_data[2] |= 0x01; /* bit8-15 */	/* L=1 */
			spi_data[3] |= 0x01; /* bit8-15 */	/* L=1 */
			break;
		}
	}
	else {
		switch (chip->copyparam) {
		case COPYPARAM_FREE:	/* copy free */
			spi_data[0] |= 0x20; /* bit0-7 */	/* COPY=1 */
			spi_data[1] |= 0x20; /* bit0-7 */	/* COPY=1 */
			spi_data[2] &= 0xFE; /* bit8-15 */	/* L=0 */
			spi_data[3] &= 0xFE; /* bit8-15 */	/* L=0 */
			break;
		case COPYPARAM_ONCE:	/* copy once */
			spi_data[0] &= 0xDF; /* bit0-7 */	/* COPY=0 */
			spi_data[1] &= 0xDF; /* bit0-7 */	/* COPY=0 */
			spi_data[2] |= 0x01; /* bit8-15 */	/* L=1 */
			spi_data[3] |= 0x01; /* bit8-15 */	/* L=1 */
			break;
		case COPYPARAM_NEVER:	/* copy never */
		default:				/* parameter NG */
			spi_data[0] &= 0xDF; /* bit0-7 */	/* COPY=0 */
			spi_data[1] &= 0xDF; /* bit0-7 */	/* COPY=0 */
			spi_data[2] &= 0xFE; /* bit8-15 */	/* L=0 */
			spi_data[3] &= 0xFE; /* bit8-15 */	/* L=0 */
			break;
		}
	}

	switch (chip->rate) {
	case 44100:
		spi_data[6] = 0x00; /* bit24-31 */
		spi_data[7] = 0x00; /* bit24-31 */
		break;
	case 48000:
		spi_data[6] = 0x40; /* bit24-31 */
		spi_data[7] = 0x40; /* bit24-31 */ 
		break;
	case 88200:
		spi_data[6] = 0x10; /* bit24-31 */
		spi_data[7] = 0x10; /* bit24-31 */
		break;
	case 96000:
		spi_data[6] = 0x50; /* bit24-31 */
		spi_data[7] = 0x50; /* bit24-31 */
		break;
	case 192000:
		spi_data[6] = 0x70; /* bit24-31 */
		spi_data[7] = 0x70; /* bit24-31 */
		break;
	default:
		pr_err("[%s] unsupported sampling rate\n", __func__);
		return;
	}

	dit4192_spi_write_device(chip, DIT4192_CS_START, 8, spi_data);

	/* Set BTD 0. */
	spi_data[0] = DIT4192_BIT_BTD_OFF;
	dit4192_spi_write_device(chip, DIT4192_REG_7, 1, spi_data);

#ifdef DEBUG_CHANNEL_STATUS
	dit4192_reg_dump(chip, DIT4192_CS_START, 8);
#endif
}

static ssize_t dit4192_set_copyparam(struct device *dev,
				struct device_attribute *attr,
				const char *buf, size_t count)
{
	struct dit4192 *chip = dev_get_drvdata(dev);
	long copyparam = COPYPARAM_NEVER;
	int ret;

	ret = kstrtol(buf, 10, &copyparam);
	if (ret < 0) {
		return ret;
	}

	switch (copyparam) {
	case COPYPARAM_FREE:	/* parameter OK */
	case COPYPARAM_ONCE:	/* parameter OK */
	case COPYPARAM_NEVER:	/* parameter OK */
		break;
	default:				/* parameter NG */
		copyparam = COPYPARAM_NEVER;
		break;
	}

	mutex_lock(&(chip->lock));
	chip->copyparam = copyparam;
	dit4192_set_channel_status(chip);
	mutex_unlock(&(chip->lock));

	return count;
}
static DEVICE_ATTR(copyparam, 0200, NULL, dit4192_set_copyparam);

static int dit4192_hw_params(struct snd_pcm_substream *substream,
			    struct snd_pcm_hw_params *params,
			    struct snd_soc_dai *dai)
{
	struct snd_soc_codec *codec = dai->codec;
	struct dit4192 *chip = dev_get_drvdata(codec->dev);
	int ret = 0;

	if (chip == NULL) {
		pr_err("[%s] invalid device private data\n", __func__);
		return -ENODEV;
	}

	switch (params_rate(params)) {
	case 44100:
		ret = gpio_direction_output(GPIO4_D3_MCLK_SEL_18, 0);
		ret = gpio_direction_output(GPIO2_B0_MCLK_SEL48_18, 0);
		break;
	case 48000:
#ifdef NEW_MCLK_SEL
		ret = gpio_direction_output(GPIO4_D3_MCLK_SEL_18, 1);
		ret = gpio_direction_output(GPIO2_B0_MCLK_SEL48_18, 0);
#else
		ret = gpio_direction_output(GPIO4_D3_MCLK_SEL_18, 0);
		ret = gpio_direction_output(GPIO2_B0_MCLK_SEL48_18, 1);
#endif
		break;
	case 88200:
#ifdef NEW_MCLK_SEL
		ret = gpio_direction_output(GPIO4_D3_MCLK_SEL_18, 0);	// dummy
		ret = gpio_direction_output(GPIO2_B0_MCLK_SEL48_18, 1);	// dummy
#else
		ret = gpio_direction_output(GPIO4_D3_MCLK_SEL_18, 1);	// dummy
		ret = gpio_direction_output(GPIO2_B0_MCLK_SEL48_18, 1);	// dummy
#endif
		break;
	case 96000:
#ifdef NEW_MCLK_SEL
		ret = gpio_direction_output(GPIO4_D3_MCLK_SEL_18, 1);
		ret = gpio_direction_output(GPIO2_B0_MCLK_SEL48_18, 1);
#else
		ret = gpio_direction_output(GPIO4_D3_MCLK_SEL_18, 1);
		ret = gpio_direction_output(GPIO2_B0_MCLK_SEL48_18, 0);
#endif
		break;
	default:
		pr_err("[%s] unsupported sampling rate\n", __func__);
		ret = -EINVAL;
		break;
	}
	if (ret < 0) {
		return ret;
	}

	mutex_lock(&(chip->lock));
	chip->rate = params_rate(params);
	dit4192_set_channel_status(chip);
	mutex_unlock(&(chip->lock));

	switch (params_format(params)) {
	case SNDRV_PCM_FORMAT_S16_LE:
	case SNDRV_PCM_FORMAT_S20_3LE:
	case SNDRV_PCM_FORMAT_S24_LE:
		break;
	default:
		pr_err("[%s] invalid format\n", __func__);
		return -EINVAL;
	}
	dev_dbg(&chip->spi->dev,
		"%s(): substream = %s  stream = %d, fmt = %d\n" , __func__,
		 substream->name, substream->stream, params_format(params));
	return ret;
}

static void dit4192_shutdown(struct snd_pcm_substream *substream, struct snd_soc_dai *dai)
{
	struct dit4192 *chip = dev_get_drvdata(dai->codec->dev);

	if (chip == NULL) {
		pr_err("[%s] invalid device private data\n", __func__);
		return;
	}
	dev_dbg(&chip->spi->dev,
		"%s(): substream = %s  stream = %d\n" , __func__,
		 substream->name, substream->stream);
}

static struct snd_soc_dai_ops dit4192_dai_ops = {
	.startup = dit4192_startup,
	.shutdown = dit4192_shutdown,
	.hw_params = dit4192_hw_params,
};

static struct snd_soc_dai_driver dit4192_dai[] = {
	{
		.name = "dit4192-dai",
		.id = 1,
		.playback = {
			.stream_name = "Playback",
			.rates = DIT4192_RATES,
			.formats = DIT4192_FORMATS,
			.rate_max = 192000,
			.rate_min = 8000,
			.channels_min = 2,
			.channels_max = 2,
		},
		.ops = &dit4192_dai_ops,
	},
};

static int dit4192_soc_probe(struct snd_soc_codec *codec)
{
	struct dit4192 *chip;
	int ret;

	codec->control_data = dev_get_drvdata(codec->dev);
	chip = codec->control_data;

	if (chip == NULL) {
		pr_err("invalid device private data\n");
		return -ENODEV;
	}
	ret = device_create_file(codec->dev, &dev_attr_copyparam);
	if (ret < 0) {
		pr_err("Failed to create copyparam sysfs file: %d\n", ret);
		return ret;
	}
	dev_set_drvdata(codec->dev, chip);

	return 0;
}

static struct snd_soc_codec_driver soc_codec_dev_dit4192 = {
	.probe = dit4192_soc_probe,
};

static void dit4192_init(struct dit4192 *chip)
{
	u8 spi_data[1];

	/* initialize gpio */
#ifdef NEW_MCLK_SEL
	gpio_direction_output(GPIO4_D3_MCLK_SEL_18, 1);
	gpio_direction_output(GPIO2_B0_MCLK_SEL48_18, 1);
#else
	gpio_direction_output(GPIO4_D3_MCLK_SEL_18, 1);
	gpio_direction_output(GPIO2_B0_MCLK_SEL48_18, 0);
#endif

	/* Setup BLS as output. */
	spi_data[0] = DIT4192_BIT_BLSM_OUT;
	dit4192_spi_write_device(chip, DIT4192_REG_1, 1, spi_data);

	/* Setup OP mode and CLK. */
	spi_data[0] = DIT4192_BIT_PDN_OP | DIT4192_BIT_CLK_256_FS;
	dit4192_spi_write_device(chip, DIT4192_REG_2, 1, spi_data);

	msleep(10);

	/* Setup bit depth and other I2S parameters. */
	spi_data[0] = DIT4192_BIT_LEFT_LOW | DIT4192_BIT_WLEN_24BITS | DIT4192_BIT_DELAY | DIT4192_BIT_MS_SLAVE;
	dit4192_spi_write_device(chip, DIT4192_REG_3, 1, spi_data);

	/* Set channel status of 96kHz */
	mutex_lock(&(chip->lock));
	chip->rate = 96000;
	chip->copyparam = COPYPARAM_FREE;
	dit4192_set_channel_status(chip);
	mutex_unlock(&(chip->lock));
}

static void dit4192_reg_dump(struct dit4192 *dit4192, u8 reg_start, int size)
{
	int i;
	u8 data[DIT4192_CS_START+CHANNEL_STATUS_SIZE];

	memset(data, 0xA5, sizeof(data));
	dit4192_spi_read_device(dit4192, reg_start, size, data);
	for (i = 0; i < size; i++) {
		pr_info("%s(): dit4192_reg_dump-%i: 0x%X\n" , __func__, i, data[i]);
	}
}

static int dit4192_default_regs[DIT4192_CS_START] = {0, 0, 3, 0, 0, 0, 0, 0};
static int dit4192_check_device(struct dit4192 *chip)
{
	int i;
	u8 data[DIT4192_CS_START];
	pr_debug("%s()\n" , __func__);

	memset(data, 0xA5, sizeof(data));
	dit4192_spi_read_device(chip, 0, DIT4192_CS_START, data);
	pr_debug("%s(): after dit4192_spi_read_device()\n" , __func__);

	for (i = 0; i < DIT4192_CS_START; i++) {
		if (data[i] != dit4192_default_regs[i]) {
			pr_err("%s(): dit4192_reg: bad reg-%d value-0x%X\n" , __func__, i, data[i]);
			dit4192_reg_dump(chip, 0, DIT4192_CS_START);
			return -EINVAL;
		}
	}
	pr_info("%s(): found device.\n" , __func__);

	return 0;
}

static int dit4192_spi_probe(struct spi_device *spi)
{
	int ret;
	struct dit4192 *chip;

	chip = devm_kzalloc(&spi->dev, sizeof(struct dit4192), GFP_KERNEL);
	if (!chip)
		return -ENOMEM;

	mutex_init(&(chip->lock));

	spi->bits_per_word = 8;
	/* mode および chip_select は DeviceTreeで設定する */
//	spi->mode = SPI_MODE_0;
//	spi->chip_select = 2;

	pr_info("%s(): cs:%d mode:%04x bits_per_word:%d\n" , __func__, spi->chip_select, spi->mode, spi->bits_per_word);

	chip->spi = spi;

	spi_set_drvdata(spi, chip);
	pr_debug("%s(): after spi_set_drvdata()\n" , __func__);

	ret = dit4192_check_device(chip);
	if (ret < 0) {
		pr_err("%s(): failed to check device: %d\n" , __func__, ret);
		return ret;
	}
	pr_debug("%s(): after dit4192_check_device()\n" , __func__);

	/* send initial values */
	dit4192_init(chip);
	pr_debug("%s(): after dit4192_init()\n" , __func__);

	ret = snd_soc_register_codec(&spi->dev,
					&soc_codec_dev_dit4192, 
					dit4192_dai, 
					ARRAY_SIZE(dit4192_dai));
	if (ret < 0) {
		pr_err("%s(): failed to register codec: %d\n" , __func__, ret);
	}
	pr_debug("%s(): after snd_soc_register_codec()\n" , __func__);

	return ret;
}

static int dit4192_spi_remove(struct spi_device *spi)
{
	struct dit4192 *chip = spi_get_drvdata(spi);

	snd_soc_unregister_codec(&spi->dev);
	spi_set_drvdata(chip->spi, NULL);
	chip->spi = NULL;

	return 0;
}

#ifdef CONFIG_PM
static int dit4192_rt_resume(struct device *dev)
{
	return 0;
}

static int dit4192_rt_suspend(struct device *dev)
{
	return 0;
}
#endif

const struct dev_pm_ops dit4192_pm_ops = {
	SET_RUNTIME_PM_OPS(dit4192_rt_suspend, dit4192_rt_resume, NULL)
};
EXPORT_SYMBOL_GPL(dit4192_pm_ops);

static const struct spi_device_id dit4192_spi_id[] = {
	{ "dit4192", },
	{ },
};
MODULE_DEVICE_TABLE(spi, dit4192_spi_id);

static const struct of_device_id dit4192_of_match[] = {
	{ .compatible = "ti,dit4192", },
	{ }
};
MODULE_DEVICE_TABLE(of, dit4192_of_match);

static struct spi_driver dit4192_spi_driver = {
	.probe		= dit4192_spi_probe,
	.remove		= dit4192_spi_remove,
	.id_table	= dit4192_spi_id,
	.driver = {
		.name	= "dit4192",
		.of_match_table = dit4192_of_match,
		.pm		= &dit4192_pm_ops,
	},
};
module_spi_driver(dit4192_spi_driver);

MODULE_DESCRIPTION("DIT4192 codec driver");
MODULE_AUTHOR("AlphaTheta Corp.");
MODULE_LICENSE("GPL v2");
