/*
 * d810.c  --  D810K013DZKB400 ALSA SoC Audio driver
 *
 * Copyright (C) 2020 AlphaTheta Corp.
 * Copyright (C) 2015-2019 Pioneer DJ Corp.
 * Yasurnori Shibata <yasunori.shibata@pioneerdj.com>
 *
 * Based on ak4387.c by Pioneer DJ Corp.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation: either version 2 of the License, or
 * (at your option) any later version.
 */

#include <linux/init.h>
#include <linux/slab.h>
#include <linux/module.h>
#include <linux/device.h>
#include <sound/core.h>
#include <sound/pcm.h>
#include <sound/initval.h>
#include <sound/soc.h>
#include <linux/of.h>
#include <linux/platform_device.h>

#define DRV_NAME	"D810"

#define SOUND_DEBUG	0	/* for debug */

static int d810_hw_params(struct snd_pcm_substream *substream,
			  struct snd_pcm_hw_params *params,
			  struct snd_soc_dai *dai)
{
	struct snd_soc_pcm_runtime *rtd = substream->private_data;
	struct snd_soc_codec *codec = rtd->codec;

#if SOUND_DEBUG
	printk(KERN_INFO "@@@@@ %s() called.\n", __func__);
#endif

	switch (params_rate(params)) {
	case 44100:
		break;
	case 48000:
	case 96000:
	case 192000:
	default:
		dev_err(codec->dev, "unsupported sampling rate\n");
		return -EINVAL;
	}

	return 0;
}

static int d810_set_dai_fmt(struct snd_soc_dai *codec_dai,
			    unsigned int format)
{
	struct snd_soc_codec *codec = codec_dai->codec;

#if SOUND_DEBUG
	printk(KERN_INFO "@@@@@ %s() called.\n", __func__);
#endif

	/* set DAI format */
	switch (format & SND_SOC_DAIFMT_FORMAT_MASK) {
	case SND_SOC_DAIFMT_LEFT_J:
	case SND_SOC_DAIFMT_I2S:
		break;
	case SND_SOC_DAIFMT_RIGHT_J:
	default:
		dev_err(codec->dev, "invalid dai format\n");
		return -EINVAL;
	}

#if 0
	/* This device can only be slave */
	if ((format & SND_SOC_DAIFMT_MASTER_MASK) != SND_SOC_DAIFMT_CBS_CFS)
		return -EINVAL;
#endif

	return 0;
}

static struct snd_soc_dai_ops d810_dai_ops = {
	.hw_params	= d810_hw_params,
	.set_fmt	= d810_set_dai_fmt,
};

#if 1
#define D810_RATES		(SNDRV_PCM_RATE_44100)
#define D810_FORMATS		(SNDRV_PCM_FMTBIT_S32_LE)
#else
#define D810_RATES		(SNDRV_PCM_RATE_8000_192000)
#define D810_FORMATS		(SNDRV_PCM_FMTBIT_S16_LE  | \
				 SNDRV_PCM_FMTBIT_S20_3LE | \
				 SNDRV_PCM_FMTBIT_S24_LE)
#endif

static struct snd_soc_dai_driver d810_dai[] = {
{
	.name = "d810_inout",
	.id	= 0,
	.playback = {
		.stream_name = "Playback",
		.channels_min = 1,
		.channels_max = 8,
		.rates = D810_RATES,
		.formats = D810_FORMATS,
	},
	.capture = {
		.stream_name = "Capture",
		.channels_min = 1,
		.channels_max = 2,
		.rates = D810_RATES,
		.formats = D810_FORMATS,
	},
	.ops = &d810_dai_ops,
	.symmetric_rates = 1,
},
{
	.name = "d810_out",
	.id	= 1,
	.playback = {
		.stream_name = "Playback",
		.channels_min = 1,
		.channels_max = 2,
		.rates = D810_RATES,
		.formats = D810_FORMATS,
	},
	.ops = &d810_dai_ops,
	.symmetric_rates = 1,
}
#if 0
,{
	.name = "d810_in",
	.id	= 2,
	.capture = {
		.stream_name = "Capture",
		.channels_min = 1,
		.channels_max = 2,
		.rates = D810_RATES,
		.formats = D810_FORMATS,
	},
	.ops = &d810_dai_ops,
	.symmetric_rates = 1,
}
#endif
};

static int d810_codec_probe(struct snd_soc_codec *codec)
{
	return 0;
}

static int d810_codec_remove(struct snd_soc_codec *codec)
{
	return 0;
}

static struct snd_soc_codec_driver soc_codec_dev_d810 = {
	.probe = d810_codec_probe,
	.remove = d810_codec_remove,
};

static int d810_probe(struct platform_device *pdev)
{
	int ret;

	dev_info(&pdev->dev, "D810K013DZKB400 ALSA SoC Codec Driver\n");

	ret = snd_soc_register_codec(&pdev->dev,
			&soc_codec_dev_d810, d810_dai, ARRAY_SIZE(d810_dai));
	if (ret < 0)
		dev_err(&pdev->dev, "failed to register codec: %d\n", ret);

	return ret;
}

static int d810_remove(struct platform_device *pdev)
{
	snd_soc_unregister_codec(&pdev->dev);
	return 0;
}

static const struct of_device_id d810_of_match_table[] = {
	{ .compatible = "alphatheta,d810" },
	{ },
};
MODULE_DEVICE_TABLE(of, d810_of_match_table);

static struct platform_driver d810_codec_driver = {
	.driver  = {
		.name   = DRV_NAME,
		.owner  = THIS_MODULE,
		.of_match_table = d810_of_match_table,
	},
	.probe  = d810_probe,
	.remove = d810_remove,
};

module_platform_driver(d810_codec_driver);

MODULE_AUTHOR("AlphaTheta Corp.");
MODULE_DESCRIPTION("D810K013DZKB400 ALSA SoC Codec Driver");
MODULE_LICENSE("GPL");
