/*
 * subucom SPI user interface
 *
 * Copyright (C) 2020 AlphaTheta Corp.
 * Copyright (C) 2015-2017 Pioneer DJ Corp.
 * Copyright (C) 2011-2014 Pioneer Corp.
 * Yasurnori Shibata <yasunori.shibata@pioneerdj.com>
 *
 * Based on spidev.c by Andrea Paterniani
 *
 *  This program is free software; you can redistribute  it and/or modify it
 *  under the terms of  the GNU General  Public License as published by the
 *  Free Software Foundation;  either version 2 of the  License, or (at your
 *  option) any later version.
 */
/* #define SUBUCOM_SPI_AGEING_TEST */
#define FEATURE_SUBUCOM_JLCD_INIT
/* #define FEATURE_SUBUCOM_JLCD_INIT_DEBUG */
/* #define DEBUG_REQ_TIMEOUT */
/* #define DEBUG_PRINT_FAKE_REQ */

#include <linux/init.h>
#include <linux/module.h>
#include <linux/moduleparam.h>
#include <linux/list.h>
#include <linux/errno.h>
#include <linux/mutex.h>
#include <linux/slab.h>
#include <linux/poll.h>
#include <linux/timer.h>
#include <linux/kthread.h>
#include <linux/spi/spi.h>

#include <linux/bitmap.h>
#include <linux/clk.h>
#include <linux/delay.h>
#include <linux/of_gpio.h>

#include <linux/sched.h>

#include <linux/interrupt.h>
#include "subucom_spi.h"

	//"ver0.01 20181119a\n"	//xINT first ENABLE-->DISABLE"
	//"ver0.02 20181120a\n"	//SPI_MODE_3 | SPI_LSB_FIRST ... 1WORD 8bits
	//"ver0.03 20181120b\n"	//spi_transfer ... 1WORD 8bits
	//"ver0.05 20181120d\n"	/* 整理 Drivers(カーネル) → デバイスadapter（ユーザ）→アプリ */
	//"ver0.07 20181121a\n"	/* update時用 write対応 xINTを使うように */
	//"ver0.09a 20181122a\n"/* JOG LCD SPIによる初期設定 実装開始版 */
	//"ver0.10a 20181126 "	/* JOG LCD SPIによる初期設定 実装 */
	//"ver0.11 20181127 "	/* PANEL f/w update 対応初期版 */
	//"ver0.12 20181203 "	/* 余分なprintkコメント DEBUG_UPDATE_PRINT を未定義にする */
	//"ver0.13 20181204 "	/* SPI clock .dtsで 1MHzに統一 */	/* modeprobe時に GP_REQ=HIGHであれば CLOCKを下げること可能 */

#define VERSION_STRING	"ver0.14 2020xxxx "

#define SUBUCOM_SPI_DATA_SIZE_NORMAL	(128)	/* normal mode... bytes tx:128 rx:128 */
#define SUBUCOM_SPI_DATA_SIZE_UPDATE	(1036)	/* update mode... bytes tx:1036 (rx:16)*/

#define SUBUCOM_SPI_TIMER_INTERVAL	(1)			/* Kernel Features-->Timer frequency 1000HZ  1:2ms 2:3ms 3:4ms */

/* SPI modeはdevice treeで指定する。Documentation/devicetree/bindings/spi/spi-bus.txtを参照のこと */
/*#define SUBUCOM_SPI_BITS_PER_WORD	(32)*/	/* bits */
#define SUBUCOM_SPI_BITS_PER_WORD	(16)	/* bits */

#define SUBUCOM_SPI_TX_MULTIPLEX_NUM	(1)
#define SUBUCOM_SPI_TX_PARTITION_NUM	(1)

#define DRV_NAME			"subucom_spi"

/* #define N_SPI_MINORS			(3) */
#define N_SPI_MINORS			(5)

/* kthread bind cpu number */
#define SPI_THREAD_BIND_CPU_NO	(3)
/* kthread priority */
#define SPI_THREAD_PRIORITY		(98)	/* SCHED_FIFO,SCHED_RR: 0(low) to 99(high) */

#define SUBUCOM_TIMER_ON		(1)
#define SUBUCOM_TIMER_OFF		(0)

#ifdef FEATURE_SUBUCOM_JLCD_INIT
#define	JLCD_SEL_JOG			(1)
#define	JLCD_SEL_PANEL			(0)
#define JLCD_SPI_MODES			(SPI_MODE_3)
#define JLCD_SPI_BITS_PER_WORD	(8)		/* bits */

static const uint16_t joglcd_reg_wdata[][2] = {
	/* P/S Reg#                        Value */
	{ 0x01,   /* Driver output */ 0x2100 },/* dummy write for new kernel */
	{ 0xF100, /* Wait100ms     */ 0xF100 },/* wait for new kernel */
	{ 0x01,   /* Driver output */ 0x2100 },
	{ 0x02,   /* LCD driver AC */ 0x0200 },
	{ 0x03,   /* Power cont    */ 0x7184 },
	{ 0xF100, /* Wait100ms     */ 0xF100 },
	{ 0x04,   /* color         */ 0x0447 },
	{ 0x05,   /* Function ctl  */ 0xB484 },
	{ 0x0A,   /* cont bright   */ 0x4008 },
	{ 0xF040, /* Wait40ms      */ 0xF040 },
	{ 0x0B,   /* Frame cycle   */ 0xD400 },
	{ 0x0D,   /* Power cont 2  */ 0x0235 },
	{ 0xF100, /* Wait100ms     */ 0xF100 },
	{ 0x0E,   /* Power cont 3  */ 0x3000 },
	{ 0xF100, /* Wait100ms     */ 0xF100 },
	{ 0x0F,   /* Gate scan pos */ 0x0000 },
	{ 0x16,   /* H porch       */ 0x9f80 },
	{ 0x17,   /* V porch       */ 0x2212 },
	{ 0xF100, /* Wait100ms     */ 0xF100 },
	{ 0x1E,   /* Power cont 4  */ 0x005F },
	{ 0xF100, /* Wait100ms     */ 0xF100 },
	{ 0x30,   /* Gamma         */ 0x0000 },
	{ 0x31,   /* Gamma         */ 0x0407 },
	{ 0x32,   /* Gamma         */ 0x0000 },
	{ 0x33,   /* Gamma         */ 0x0000 },
	{ 0x34,   /* Gamma         */ 0x0505 },
	{ 0x35,   /* Gamma         */ 0x0003 },
	{ 0x36,   /* Gamma         */ 0x0707 },
	{ 0x37,   /* Gamma         */ 0x0000 },
	{ 0x3A,   /* Gamma         */ 0x0904 },
	{ 0x3B,   /* Gamma         */ 0x0904 },
};
#endif /* FEATURE_SUBUCOM_JLCD_INIT */

#ifdef DEBUG_REQ_TIMEOUT
#define SUBUCOM_SPI_TIMER_INTERVAL_REQ_TIMEOUT	(10 * HZ)	/* req handler timeout check interval: 10sec*/
#endif

/* driver private data */
struct subucom_spi_dev {
	dev_t				devt;
	spinlock_t			spi_lock;
	struct spi_device	*spi;

	struct list_head	device_entry;

	struct timer_list	spi_timer;
	wait_queue_head_t	spi_wait;
	uint8_t				timer_status;
	uint32_t			timer_interval;	/* unsigned */

	struct task_struct	*spi_thread;
	int32_t				event_flg;		/* int: dummy event flag */

	uint32_t			users;			/* unsigned */

	uint32_t			speed_hz;		/* unsigned */
	uint32_t			bytes_per_word;	/* unsigned */
	struct mutex		transfer_lock;

	int32_t				rx_detected;	/* int */
	uint32_t			rx_bytes;		/* unsigned */
	struct mutex		rx_buf_lock;
	uint8_t				*rx_buf;
	uint8_t				*rx_bufold;

	uint16_t			tx_multi_num;	/* number of multiplexes */
	uint16_t			tx_multi_id;	/* current multiplex */
	struct subucom_tx_part
						*tx;
	struct mutex		tx_buf_lock;
	uint8_t				*tx_buf;
	uint8_t				*tx_bufnew;

	struct mutex		buffer_lock;
	uint8_t				*buffer;

	struct subucom_ioc_transfer	ioc;

	uint16_t			max_cntbusy;

	int32_t				req_gpio;
	int					req_irq;
	int					req_value;
	wait_queue_head_t 	req_wq;
	int					fake_req_cnt;
	int32_t				cont_gpio;

#ifdef SUBUCOM_SPI_AGEING_TEST
	uint32_t			rx_count;
	uint32_t			rx_success_count;
	uint32_t			rx_fail_count;
	unsigned char		rx_fail_data[SUBUCOM_SPI_DATA_SIZE_NORMAL];
#endif

#ifdef FEATURE_SUBUCOM_JLCD_INIT
	uint16_t			subucom_mode;
	int32_t				jlcd_sel_gpio;
#endif /* FEATURE_SUBUCOM_JLCD_INIT */

#ifdef DEBUG_REQ_TIMEOUT
	uint8_t				timer_status_req_timeout;
	struct timer_list	req_timeout_timer;
	uint32_t			timer_interval_req_timeout;
#endif
};

struct subucom_tx_part {
	uint16_t	part_num;	/* number of partitions */
	uint16_t	part_id;	/* current partitions */

	int32_t		updated;
	uint32_t	bytes;
	uint32_t	transfer_bytes;
	int32_t		transfer_len;		/* size_t */

	uint8_t		*buf;
	uint8_t		*bufnew;

	struct subucom_ioc_transfer ioc;
};

#define SUBCPU_BUSYWAIT_USEC	(50)	/* udelay */
#define CNT_SUBCPU_BUSYWAIT		(15)	/* 750usec CNT(15) X USEC(50) */

#define GPIO_CONT_DISABLE		(0)
#define GPIO_CONT_ENABLE		(1)

static DECLARE_BITMAP(minors, N_SPI_MINORS);

static uint8_t			_debug = 0;
static int32_t			majorNumber = 0;
static struct class		*subucom_spi_class = NULL;

static LIST_HEAD(device_list);
static DEFINE_MUTEX(device_list_lock);

static uint32_t	DEFtmstat	= SUBUCOM_TIMER_OFF;	/* 2018.11.13 アプリがタイマ開始するはず */
module_param(DEFtmstat, uint, S_IRUGO);
MODULE_PARM_DESC(DEFtmstat, "timer status default setting");

static uint32_t	SZbufsize = SUBUCOM_SPI_DATA_SIZE_UPDATE;
module_param(SZbufsize, uint, S_IRUGO);
MODULE_PARM_DESC(SZbufsize, "maximum data size in SPI message");

static uint32_t	txmulti = SUBUCOM_SPI_TX_MULTIPLEX_NUM;
module_param(txmulti, uint, S_IRUGO);
MODULE_PARM_DESC(txmulti, "maximum multiplex number in SPI Tx data");

static uint32_t	txpart = SUBUCOM_SPI_TX_PARTITION_NUM;
module_param(txpart, uint, S_IRUGO);
MODULE_PARM_DESC(txpart, "maximum partition number in SPI Tx data");

#ifdef DEBUG_REQ_TIMEOUT
static void subucom_spi_timer_req_timeout_func(unsigned long data)
{
	struct subucom_spi_dev	*priv = (struct subucom_spi_dev *)data;
	struct timer_list *timer_req = &priv->req_timeout_timer;

	if (priv->timer_status_req_timeout == SUBUCOM_TIMER_ON) {
		dev_info(&priv->spi->dev, "[subucom_spi_timer_req_timeout_func] REQ timed out.\n");
		mod_timer(timer_req, jiffies + msecs_to_jiffies(priv->timer_interval_req_timeout));
	}
}

static inline void
subucom_spi_timer_req_timeout_init(struct subucom_spi_dev *priv)
{
	struct timer_list *timer_req = &priv->req_timeout_timer;

	priv->timer_status_req_timeout = SUBUCOM_TIMER_OFF;
	init_timer(timer_req);
	timer_req->data = (unsigned long)priv;
	timer_req->function = subucom_spi_timer_req_timeout_func;
}

static inline void
subucom_spi_timer_req_timeout_start(struct subucom_spi_dev *priv)
{
	struct timer_list *timer_req = &priv->req_timeout_timer;

	if (priv->timer_status_req_timeout != SUBUCOM_TIMER_ON) {
		timer_req->expires = jiffies + msecs_to_jiffies(priv->timer_interval_req_timeout);
		add_timer(timer_req);

		dev_info(&priv->spi->dev, "[subucom_spi_timer_req_timeout_start] REQ timeout timer started.\n");
		priv->timer_status_req_timeout = SUBUCOM_TIMER_ON;
	}
}

static inline void
subucom_spi_timer_req_timeout_restart(struct subucom_spi_dev *priv)
{
	struct timer_list *timer_req = &priv->req_timeout_timer;

	if (priv->timer_status_req_timeout == SUBUCOM_TIMER_ON) {
		mod_timer(timer_req, jiffies + msecs_to_jiffies(priv->timer_interval_req_timeout));
	}
}

static inline void
subucom_spi_timer_req_timeout_stop(struct subucom_spi_dev *priv)
{
	if (priv->timer_status_req_timeout == SUBUCOM_TIMER_ON) {
		del_timer_sync(&priv->req_timeout_timer);
		priv->timer_status_req_timeout = SUBUCOM_TIMER_OFF;
	}
}
#endif

/* top half */
static irqreturn_t req_hard_handler(int irq, void *dev_id)
{
	struct subucom_spi_dev *priv = dev_id;

	priv->req_value = gpio_get_value(priv->req_gpio);
	return IRQ_WAKE_THREAD;
}

/* bottom half */
static irqreturn_t req_handler(int irq, void *dev_id)
{
	struct subucom_spi_dev *priv = dev_id;
	irqreturn_t ret = IRQ_NONE;

	if (priv->req_value) {
		wake_up(&priv->req_wq);
#ifdef DEBUG_REQ_TIMEOUT
		subucom_spi_timer_req_timeout_restart(priv);
#endif
		ret = IRQ_HANDLED;
	}
	else {
		++priv->fake_req_cnt;
#ifdef DEBUG_PRINT_FAKE_REQ
		dev_info(&priv->spi->dev, "detect fake req %d\n", priv->fake_req_cnt);
#endif
	}
	return ret;
}

//******************************************************************************
//* from subucom_spi_sync_read_write()/subucom_spi_sync_rw_transfer()
//*    +---- this...
//*
//* name : subucom_spi_sync()
//*           SYNC SPI write/read
//******************************************************************************
static ssize_t
subucom_spi_sync(struct subucom_spi_dev *priv, struct spi_message *message)
{
	ssize_t status;
	struct spi_device	*spi;

	spin_lock_irq(&priv->spi_lock);
	spi = priv->spi;
	spin_unlock_irq(&priv->spi_lock);

	if (spi == NULL) {
		status = -ESHUTDOWN;
	}
	else {
		status = spi_sync(spi, message);
		if (status == 0) {
			return message->actual_length;
		}
	}
	return status;
}
//******************************************************************************
//* from application
//*    +-- read()
//*           +-- subucom_spi_read()
//*                  +-- this...
//*    +-- write()
//*           +-- subucom_spi_write()
//*                  +-- this...
//*
//* name : subucom_spi_sync_read_write()
//******************************************************************************
static inline ssize_t
subucom_spi_sync_read_write(struct subucom_spi_dev *priv, uint8_t *tx_buf, uint8_t *rx_buf, size_t len)
{
	struct spi_transfer t = {
		.tx_buf	= tx_buf,
		.rx_buf	= rx_buf,
		.len	= len,
		.speed_hz = priv->speed_hz,
	};
	struct spi_message	m;
	ssize_t status;

	spi_message_init(&m);
	spi_message_add_tail(&t, &m);

#ifdef FEATURE_SUBUCOM_JLCD_INIT
	gpio_set_value(priv->jlcd_sel_gpio, JLCD_SEL_PANEL);
#endif /* FEATURE_SUBUCOM_JLCD_INIT */

	status = subucom_spi_sync(priv, &m);

#ifdef FEATURE_SUBUCOM_JLCD_INIT
	gpio_set_value(priv->jlcd_sel_gpio, JLCD_SEL_JOG);
#endif /* FEATURE_SUBUCOM_JLCD_INIT */

	return status;
}
//******************************************************************************
//* from subucom_spi_thread()
//*	        for loop ...wait_event_interruptible()
//*            +-- subucom_spi_transfer
//*                   +-- this...
//*
//* name : subucom_spi_sync_rw_transfer()
//******************************************************************************
static inline int32_t
subucom_spi_sync_rw_transfer(struct subucom_spi_dev *priv, uint8_t *tx_buf, size_t len)
{
	struct spi_transfer t = {
		.tx_buf	= tx_buf,
		.rx_buf	= priv->rx_buf,
		.len	= len,
		.speed_hz = priv->speed_hz,
	};
	struct spi_message	m;
	int32_t	status;

	spi_message_init(&m);
	spi_message_add_tail(&t, &m);

#ifdef FEATURE_SUBUCOM_JLCD_INIT
	gpio_set_value(priv->jlcd_sel_gpio, JLCD_SEL_PANEL);
#endif /* FEATURE_SUBUCOM_JLCD_INIT */

	status = subucom_spi_sync(priv, &m);

#ifdef FEATURE_SUBUCOM_JLCD_INIT
	gpio_set_value(priv->jlcd_sel_gpio, JLCD_SEL_JOG);
#endif /* FEATURE_SUBUCOM_JLCD_INIT */

	if (status < 0)
		return status;
	return 0;
}

/*-------------------------------------------------------------------------*/

static inline void
subucom_spi_clear_buffers(struct subucom_spi_dev *priv)
{
	mutex_lock(&priv->transfer_lock);
	memset(priv->tx_buf, 0, SZbufsize * txmulti * txpart);
	memset(priv->rx_buf, 0, SZbufsize);
	mutex_unlock(&priv->transfer_lock);

	mutex_lock(&priv->tx_buf_lock);
	memset(priv->tx_bufnew, 0, SZbufsize * txmulti * txpart);
	mutex_unlock(&priv->tx_buf_lock);

	mutex_lock(&priv->rx_buf_lock);
	memset(priv->rx_bufold, 0, SZbufsize);
	mutex_unlock(&priv->rx_buf_lock);
}

static inline void
subucom_spi_init_ids(struct subucom_spi_dev *priv)
{
	struct subucom_tx_part	*tx;
	uint32_t	i;

	priv->tx_multi_id = 0;
	tx = priv->tx;
	for (i = 0; i < txmulti; ++i) {
		tx->part_id = 0;
		tx->updated = 0;
		++tx;
	}
}
//******************************************************************************
//* from APP
//*    +-- ioctl(SUBUCOM_IOC_WR_BITS_PER_WORD,,,)
//*           +-- subucom_spi_ioctl()               OR   subucom_spi_probe()
//*                  |
//*                  +-- this...
//*
//* name : subucom_spi_setup_bytes_per_word()
//******************************************************************************
static inline void 
subucom_spi_setup_bytes_per_word(struct subucom_spi_dev *priv)
{
	priv->bytes_per_word = DIV_ROUND_UP(priv->spi->bits_per_word, 8);
}
//******************************************************************************
//* from subucom_spi_setup_tx()
//*    +-- subucom_spi_setup_length()
//*           +-- this... subucom_spi_calc_length
//*
//* subucom_spi_write() or subucom_spi_read()
//*    +-- this... subucom_spi_calc_length
//*
//* name : subucom_spi_calc_length()
//******************************************************************************
static inline size_t
subucom_spi_calc_length(struct subucom_spi_dev *priv, size_t bytes)
{
	size_t	len = 0;

	if (priv->bytes_per_word) {
		len = DIV_ROUND_UP(bytes, priv->bytes_per_word) * priv->bytes_per_word;
	}
	return len;
}
//******************************************************************************
//* from subucom_spi_setup_tx  OR subucom_spi_probe()  OR ioctl(SUBUCOM_IOC_WR_RX_BYTES)
//*    +-- this... subucom_spi_setup_length()
//*
//* name : subucom_spi_setup_length()
//******************************************************************************
static inline void
subucom_spi_setup_length(struct subucom_spi_dev *priv, uint16_t multi_id)
{
	struct subucom_tx_part	*tx = priv->tx + multi_id;
	size_t	len;

	/*
	 * 送信,受信 同クロックで並列して実施の場合
	 * 長いレングス側で送受信transfer_lenを決める    EP122は 64 tx:64 rx 1036 tx:16 rx なのでこのままで問題なさそう...
	 */
	tx->transfer_bytes = (tx->bytes > priv->rx_bytes) ? tx->bytes : priv->rx_bytes;
	len	= subucom_spi_calc_length(priv, tx->transfer_bytes);

	mutex_lock(&priv->transfer_lock);
	tx->transfer_len = len;
	mutex_unlock(&priv->transfer_lock);
}
//******************************************************************************
//* from subucom_spi_thread()
//*	        for(;;) loop wait_event_interruptible()  <--- Cyclic TMOUT
//*            +-- subucom_spi_transfer
//*                   +-- this...
//*
//* name : subucom_spi_setup_tx
//******************************************************************************
static inline struct subucom_tx_part *
subucom_spi_setup_tx(struct subucom_spi_dev *priv)
{
	struct subucom_tx_part	*tx = priv->tx + priv->tx_multi_id;

	/* update parameters */
	if (tx->updated & 0x1) {
		tx->part_num = tx->ioc.part_num;
		tx->bytes	 = tx->ioc.bytes;
		subucom_spi_setup_length(priv, priv->tx_multi_id);

		tx->part_id	= tx->part_num;
	}

	/* copy from bufnew to buf */
	if (tx->updated) {
		int offset, shift = 0;

		mutex_lock(&priv->tx_buf_lock);

		do {
			if (tx->updated & 0x1) {
				offset = shift * SZbufsize;
				memcpy(tx->buf + offset, tx->bufnew + offset, SZbufsize);
			}
			tx->updated >>= 1;
			++shift;
		} while (tx->updated);

		mutex_unlock(&priv->tx_buf_lock);
	}

	/* update part_id */
	++tx->part_id;
	tx->part_id %= tx->part_num;

	/* update tx_multi_id */
	++priv->tx_multi_id;
	priv->tx_multi_id %= priv->tx_multi_num;

	return tx;
}

static inline void
subucom_spi_check_rxbuf(struct subucom_spi_dev *priv)
{
#ifdef SUBUCOM_SPI_AGEING_TEST
	int	i;

	priv->rx_count++;

	for (i = 0; i < priv->rx_bytes; i++) {
		if (priv->rx_buf[i] != (i % 256)) {
			break;
		}
	}
	if (i >= priv->rx_bytes) {
		priv->rx_success_count++;
	}
	else {
		priv->rx_fail_count++;
		memcpy(priv->rx_fail_data, priv->rx_buf, SUBUCOM_SPI_DATA_SIZE_NORMAL);
	}
#endif
	/* ここは MODE=0 の場合のチェックを入れたほうが良い？ */

	/* compare to old data */
	if (memcmp(priv->rx_bufold, priv->rx_buf, priv->rx_bytes)) {
		mutex_lock(&priv->rx_buf_lock);
		memcpy(priv->rx_bufold, priv->rx_buf, priv->rx_bytes);
		priv->rx_detected = 1;
		mutex_unlock(&priv->rx_buf_lock);
	}
}

//******************************************************************************
//* from subucom_spi_transfer()
//*         +-- this...
//*
//* subucom_check_ready
//******************************************************************************
static int32_t subucom_check_ready(struct subucom_spi_dev *priv)
{
	int32_t	subucom_status = 0;
	int32_t	retry_limit = priv->max_cntbusy;

	/* check req_gpio value */
	wait_event_interruptible_timeout(priv->req_wq,
									gpio_get_value(priv->req_gpio) != 0,
									retry_limit * SUBCPU_BUSYWAIT_USEC);
	subucom_status = gpio_get_value(priv->req_gpio);

	/* if req_gpio is high, set cont_gpio to high */
	if (gpio_is_valid(priv->cont_gpio) && (subucom_status)) {
		gpio_set_value(priv->cont_gpio, GPIO_CONT_ENABLE);
	}

	return subucom_status;
}
//******************************************************************************
//* from subucom_spi_transfer()
//*         +-- this...
//*
//* subucom_txrx_done
//******************************************************************************
static void subucom_txrx_done(struct subucom_spi_dev *priv)
{
	if (gpio_is_valid(priv->cont_gpio)) {
		gpio_set_value(priv->cont_gpio, GPIO_CONT_DISABLE);
	}
}
//******************************************************************************
//* from Cyclic TMOUT
//*         +-- subucom_spi_thread
//*                +-- this... subucom_spi_transfer()
//******************************************************************************
static int32_t subucom_spi_transfer(struct subucom_spi_dev *priv)
{
	struct subucom_tx_part	*tx;
	int32_t		status;
	uint8_t		*tx_buf;
	int32_t		ready;

	if (!priv) {
		return -EIO;
	}

	ready = subucom_check_ready(priv);
	if (ready) {

		/* setup tx data */
		tx = subucom_spi_setup_tx(priv);
		if (tx->transfer_bytes <= SZbufsize) {

			tx_buf = tx->buf + (SZbufsize * tx->part_id);

			/* transfer data */
			mutex_lock(&priv->transfer_lock);
			/* transfer_len ... calc&set from subucom_spi_setup_tx() --> subucom_spi_setup_length() */
			status = subucom_spi_sync_rw_transfer(priv, tx_buf, tx->transfer_len);
			if (status) {
				dev_err(&priv->spi->dev, "[subucom_spi_transfer] SPI r/w failed\n");
				status = -EIO;
			}
			mutex_unlock(&priv->transfer_lock);

			/* check rx data */
			if (status == 0) {
				subucom_spi_check_rxbuf(priv);
			}
		}
		else {
			return -EINVAL;
		}
	}
	else {
		status = -EBUSY;
	}
	subucom_txrx_done(priv);

	return status;
}

//******************************************************************************
//* from TIMER  ...expires = jiffies+msecs_to_jiffies(timeout)
//*
//* name: subucom_spi_timer_func()
//******************************************************************************
static void subucom_spi_timer_func(unsigned long data)
{
	struct subucom_spi_dev	*priv = (struct subucom_spi_dev *)data;

	priv->event_flg = 1;
	wake_up_interruptible(&priv->spi_wait);
}
//******************************************************************************
//* name: subucom_spi_timer_init()
//******************************************************************************
static inline void
subucom_spi_timer_init(struct subucom_spi_dev *priv)
{
	struct timer_list *timer = &priv->spi_timer;
	init_timer(timer);
	timer->data = (unsigned long)priv;
	timer->function = subucom_spi_timer_func;
}
//******************************************************************************
//* name: subucom_spi_timer_start()
//******************************************************************************
static inline void
subucom_spi_timer_start(struct subucom_spi_dev *priv)
{
	struct timer_list *timer = &priv->spi_timer;
	timer->expires = jiffies + msecs_to_jiffies(priv->timer_interval);
	add_timer(timer);

	if (!priv->timer_status) {
		subucom_spi_clear_buffers(priv);
		subucom_spi_init_ids(priv);
	}
	priv->timer_status = SUBUCOM_TIMER_ON;
}
//******************************************************************************
//* name: subucom_spi_timer_restart()
//******************************************************************************
static inline void
subucom_spi_timer_restart(struct subucom_spi_dev *priv)
{
	struct timer_list *timer = &priv->spi_timer;
	if (priv->timer_status) {
		mod_timer(timer, jiffies + msecs_to_jiffies(priv->timer_interval));
	}
}
//******************************************************************************
//* name: subucom_spi_timer_stop()
//******************************************************************************
static inline void
subucom_spi_timer_stop(struct subucom_spi_dev *priv)
{
	del_timer_sync(&priv->spi_timer);
	priv->timer_status = SUBUCOM_TIMER_OFF;
}

//******************************************************************************
//* from subucom_spi_probe()
//*         +---- subucom_spi_thread_setup()
//*                  +---- kthread_create(),kthread_bind(),wake_up_process()
//*                           +---- this...
//*
//* name: subucom_spi_thread()
//******************************************************************************
static bool  pm_ready = false;
static int32_t subucom_spi_thread(void *data)
{
	struct subucom_spi_dev	*priv = (struct subucom_spi_dev *)data;
	int32_t	status = 0;

	/* set kthread priority */
	struct sched_param param = { .sched_priority = SPI_THREAD_PRIORITY };

	status = sched_setscheduler(current, SCHED_RR, &param);
	dev_info(&priv->spi->dev, "[%s] sched_setscheduler:%d pid:%d pri:%d\n",
			__func__, status, current->pid, param.sched_priority);	/* debug print */

	priv->event_flg = 0;

	for (;;) {

		if (!pm_ready) {
			msleep(200);
			continue;
		}

		status = wait_event_interruptible_timeout(priv->spi_wait,
												(priv->event_flg != 0 || kthread_should_stop()),
												msecs_to_jiffies(1000));
		if (kthread_should_stop())
			break;

		if (status > 0) {/* normal */
			priv->event_flg = 0;
			subucom_spi_timer_restart(priv);
			subucom_spi_transfer(priv);
		}
		else if (status == 0) {	/* timed out */
			if (priv->timer_status == SUBUCOM_TIMER_ON) {
				subucom_spi_timer_stop(priv);
				dev_info(&priv->spi->dev, "[%s] wait_event_interruptible_timeout() Detected TIMEOUT!\n", __func__);
				msleep(10);
				priv->event_flg = 0;
				subucom_spi_timer_start(priv);
				subucom_spi_transfer(priv);
			}
		}
		else {					/* interrupt */
			dev_info(&priv->spi->dev, "[%s] wait_event_interruptible_timeout():%d\n", __func__, status);
		}
	}
	return 0;
}


//******************************************************************************
//* name : subucom_spi_thread_setup
//******************************************************************************
static int32_t subucom_spi_thread_setup(struct subucom_spi_dev *priv)
{
	struct task_struct	*thread;

	thread = kthread_create(subucom_spi_thread, priv, "subucom_spi%d.%d",
							priv->spi->master->bus_num, priv->spi->chip_select);
	if (IS_ERR(thread)) {
		dev_err(&priv->spi->dev, "[subucom_spi_thread_setup] creating SPI thread failed\n");
		return PTR_ERR(thread);
	}

	kthread_bind(thread, SPI_THREAD_BIND_CPU_NO);
	wake_up_process(thread);

	dev_info(&priv->spi->dev, "[subucom_spi_thread_setup] subucom_spi_thread created. pid:%d\n", thread->pid);

	priv->spi_thread = thread;
	return 0;
}

//******************************************************************************
//* name : subucom_spi_thread_kill
//******************************************************************************
static void subucom_spi_thread_kill(struct subucom_spi_dev *priv)
{
	kthread_stop(priv->spi_thread);
	priv->spi_thread = NULL;
}

/*-------------------------------------------------------------------------*/

//******************************************************************************
//* from APP
//*    +-- read()
//*           +-- this...
//*
//* name : subucom_spi_read
//******************************************************************************
static ssize_t
subucom_spi_read(struct file *filp, char __user *buf, size_t count, loff_t *f_pos)
{
	struct subucom_spi_dev	*priv;
	ssize_t			status = 0;
	unsigned long	missing;
	int32_t			ready;

	priv = filp->private_data;
	if (!priv) {
		return -EIO;
	}

	if (priv->timer_status == SUBUCOM_TIMER_ON) {	/* TIMER ON */
		if (count > SZbufsize) {
			return -EMSGSIZE;
		}
		if ((filp->f_flags & O_NONBLOCK) && (!(priv->rx_detected))) {
			return -EAGAIN;
		}
		mutex_lock(&priv->rx_buf_lock);
		missing = copy_to_user(buf, priv->rx_bufold, count);
		if (missing == count)
			status = -EFAULT;
		else
			status = count - missing;
		priv->rx_detected = 0;
		mutex_unlock(&priv->rx_buf_lock);
	}
	else {											/* TIMER OFF */
		size_t	len;
		uint8_t *tmp;

		mutex_lock(&priv->buffer_lock);
		priv->buffer = kzalloc(2 * count * sizeof(uint8_t), GFP_KERNEL);
		if (!priv->buffer) {
			mutex_unlock(&priv->buffer_lock);
			return -ENOMEM;
		}
		tmp = priv->buffer + (count * sizeof(uint8_t));
		len = subucom_spi_calc_length(priv, count);

		ready = subucom_check_ready(priv);
		if (ready) {
			status = subucom_spi_sync_read_write(priv, tmp, priv->buffer, len);
		}
		else {
			status = -EBUSY;
		}
		subucom_txrx_done(priv);

		if (status > 0) {
			missing = copy_to_user(buf, priv->buffer, count);
			if (missing == count) {
				status = -EFAULT;
				dev_err(&priv->spi->dev, "[subucom_spi_read] ERROR copy_to_user:%u subucom_spi_sync_read_write()\n", (uint32_t)missing);
			}
			else {
				status = count - missing;
			}
		}

		kfree(priv->buffer);
		mutex_unlock(&priv->buffer_lock);
	}
	return status;
}
//******************************************************************************
//* from APP
//*    +-- write()
//*           +-- this...
//*
//* name : subucom_spi_write()
//******************************************************************************
static ssize_t
subucom_spi_write(struct file *filp, const char __user *buf,
		  size_t count, loff_t *f_pos)
{
	struct subucom_spi_dev	*priv;
	ssize_t			status = 0;
	unsigned long	missing;
	int32_t			ready;

	priv = filp->private_data;
	if (!priv) {
		return -EIO;
	}

	if (priv->timer_status == SUBUCOM_TIMER_ON) {	/* TIMER ON */
		/* not supported. */
		return -EFAULT;
	}
	else {											/* TIMER OFF */
		size_t	len;
		uint8_t	*tmp;

		mutex_lock(&priv->buffer_lock);
		priv->buffer = kzalloc(2 * count * sizeof(uint8_t), GFP_KERNEL);
		if (!priv->buffer) {
			mutex_unlock(&priv->buffer_lock);
			return -ENOMEM;
		}
		tmp = priv->buffer + (count * sizeof(uint8_t));

		missing = copy_from_user(priv->buffer, buf, count);
		if (missing == 0) {
			len = subucom_spi_calc_length(priv, count);

			ready = subucom_check_ready(priv);
			if (ready) {
				status = subucom_spi_sync_read_write(priv, priv->buffer, tmp, len);
			}
			else {
				status = -EBUSY;
			}
			subucom_txrx_done(priv);
		}
		else {
			status = -EFAULT;
		}
		kfree(priv->buffer);
		mutex_unlock(&priv->buffer_lock);
	}
	return status;
}
//******************************************************************************
//* from APP
//*    poll()
//*       +-- this...
//*
//* name : subucom_spi_poll
//******************************************************************************
static unsigned int
subucom_spi_poll(struct file *filp, poll_table *wait)
{
	struct subucom_spi_dev	*priv;
	unsigned int	mask = 0;

	priv = filp->private_data;

	if (priv->timer_status == SUBUCOM_TIMER_ON) {	/* TIMER ON */
		poll_wait(filp, &priv->spi_wait, wait);
		if (priv->rx_detected) {
			mask |= (POLLIN | POLLRDNORM);
		}
	}

	return mask;
}
//******************************************************************************
//* from APP
//*    +-- ioctl(SUBUCOM_IOC_MESSAGE())
//*           +-- subucom_spi_ioctl()
//*                  +-- this...
//*
//* name : subucom_setup_txbufnew
//******************************************************************************
static int32_t subucom_setup_txbufnew(struct subucom_spi_dev *priv)
{
	uintptr_t	src;
	uint8_t		*dest;
	int32_t		status = -EFAULT;

	struct subucom_ioc_transfer	*ioc= &priv->ioc;
	struct subucom_tx_part		*tx	= priv->tx + ioc->multi_id;

	if ((ioc->multi_id >= priv->tx_multi_num)
	||  (ioc->part_num > txpart)
	||  (ioc->part_id >= ioc->part_num)
	||  (ioc->bytes > SZbufsize)) {
		return -EINVAL;
	}

	dest = tx->bufnew + (SZbufsize * ioc->part_id);
	src = (uintptr_t)ioc->tx_buf;
	mutex_lock(&priv->tx_buf_lock);
	if (!copy_from_user(dest, (const uint8_t __user *)src, ioc->bytes)) {
		memcpy(&tx->ioc, ioc, sizeof(struct subucom_ioc_transfer));
		tx->updated |= 0x1 << ioc->part_id;
		status = 0;
	}
	mutex_unlock(&priv->tx_buf_lock);

	return status;
}
//******************************************************************************
//* from APP
//*    +-- ioctl()
//*           +-- this...
//*
//* name : subucom_spi_ioctl
//******************************************************************************
static long
subucom_spi_ioctl(struct file *filp, unsigned int cmd, unsigned long arg)
{
	struct subucom_spi_dev	*priv;
	struct spi_device		*spi;
	int32_t		err = 0;
	int32_t		retval = 0;
	uint32_t	tmp;
	uint32_t	n_ioc;		//unsigned
	uint16_t	usec;
	int32_t		gp_write;

	/* Check type and command number */
	if (_IOC_TYPE(cmd) != SUBUCOM_IOC_MAGIC) {
		return -ENOTTY;
	}

	/* Check access direction once here; don't repeat below.
	 * IOC_DIR is from the user perspective, while access_ok is
	 * from the kernel perspective; so they look reversed.
	 */
	if (_IOC_DIR(cmd) & _IOC_READ) {
		err = !access_ok(VERIFY_WRITE, (void __user *)arg, _IOC_SIZE(cmd));
	}
	if (err == 0 && _IOC_DIR(cmd) & _IOC_WRITE) {
		err = !access_ok(VERIFY_READ, (void __user *)arg, _IOC_SIZE(cmd));
	}
	if (err) {
		return -EFAULT;
	}

	/* guard against device removal before, or while,
	 * we issue this ioctl.
	 */
	priv = filp->private_data;
	if (!priv) {
		return -ESHUTDOWN;
	}

	spin_lock_irq(&priv->spi_lock);
	spi = spi_dev_get(priv->spi);
	spin_unlock_irq(&priv->spi_lock);
	if (spi == NULL) {
		return -ESHUTDOWN;
	}

	switch (cmd) {
	/* read requests */
	case SUBUCOM_IOC_RD_TIMER_STATUS:
		retval = __put_user(priv->timer_status, (__u8 __user *)arg);
		break;
	case SUBUCOM_IOC_RD_TIMER_INTERVAL:
		retval = __put_user(priv->timer_interval, (__u32 __user *)arg);
		break;
	case SUBUCOM_IOC_RD_BITS_PER_WORD:
		retval = __put_user(spi->bits_per_word, (__u16 __user *)arg);
		break;
	case SUBUCOM_IOC_RD_RX_BYTES:
		retval = __put_user(priv->rx_bytes, (__u32 __user *)arg);
		break;
	case SUBUCOM_IOC_RD_BUSYWAIT_UTIME:
		usec	= (uint16_t)(priv->max_cntbusy * SUBCPU_BUSYWAIT_USEC);
		retval = __put_user(usec, (__u16 __user *)arg);
		break;
	case SUBUCOM_IOC_WR_DEBUG:
		retval = __get_user(tmp, (__u8 __user *)arg);
		if (retval == 0) {
			_debug = tmp;
			dev_info(&spi->dev, "[subucom_spi_ioctl](W) debug:%d\n", _debug);
		}
		break;
	/* write requests */
	case SUBUCOM_IOC_WR_TIMER_STATUS:
		retval = __get_user(tmp, (__u8 __user *)arg);
		if (retval == 0) {
			if (tmp & ~SUBUCOM_TIMER_ON) {
				retval = -EINVAL;
				break;
			}
			dev_info(&spi->dev, "[subucom_spi_ioctl](W) TIMER STATUS=%d\n", tmp);
#ifdef SUBUCOM_SPI_AGEING_TEST
			dev_info(&spi->dev, "[subucom_spi_ioctl](W) rx_count=%d, rx_success_count=%d, rx_fail_count=%d\n",
				priv->rx_count, priv->rx_success_count, priv->rx_fail_count);
			dev_info(&spi->dev, "[subucom_spi_ioctl](W) rx_fail_data=%02X,%02X,%02X,%02X,%02X,%02X,%02X,%02X\n",
				priv->rx_fail_data[0], priv->rx_fail_data[1], priv->rx_fail_data[2], priv->rx_fail_data[3],
				priv->rx_fail_data[4], priv->rx_fail_data[5], priv->rx_fail_data[6], priv->rx_fail_data[7]);
#endif

			if (tmp && !priv->timer_status) {
				dev_info(&spi->dev, "calling subucom_spi_timer_start()...\n"); /* debug print */
				subucom_spi_timer_start(priv);
#ifdef DEBUG_REQ_TIMEOUT
				subucom_spi_timer_req_timeout_start(priv);
#endif
			}
			else if (!tmp && priv->timer_status) {
				dev_info(&spi->dev, "calling subucom_spi_timer_stop()...\n"); /* debug print */
				subucom_spi_timer_stop(priv);
#ifdef DEBUG_REQ_TIMEOUT
				subucom_spi_timer_req_timeout_stop(priv);
#endif
			}
			else {
				dev_info(&spi->dev, "timer status will not changed. value:%d timer_status:%d\n", tmp, priv->timer_status);
			}
		}
		break;
	case SUBUCOM_IOC_WR_TIMER_INTERVAL:
		retval = __get_user(tmp, (__u32 __user *)arg);
		if (retval == 0) {
			priv->timer_interval = (unsigned long)tmp;
			dev_info(&spi->dev, "[subucom_spi_ioctl](W) TIMER INTERVAL=%u\n", priv->timer_interval);
		}
		break;
	case SUBUCOM_IOC_WR_BITS_PER_WORD:
		retval = __get_user(tmp, (__u16 __user *)arg);
		if (retval == 0) {
			uint16_t	save = spi->bits_per_word;
			if (priv->timer_status == SUBUCOM_TIMER_ON) {
				retval = -EBUSY;
				break;
			}
			spi->bits_per_word = tmp;
			dev_info(&spi->dev, "[subucom_spi_ioctl](W) BITS PER WORD=%u\n", spi->bits_per_word);
			retval = spi_setup(spi);
			if (retval < 0) {
				spi->bits_per_word = save;
				dev_err(&spi->dev, "[subucom_spi_ioctl] spi_setup() failed. %d\n", retval);
			}
			subucom_spi_setup_bytes_per_word(priv);
		}
		break;
	case SUBUCOM_IOC_WR_RX_BYTES:
		retval = __get_user(tmp, (__u32 __user *)arg);
		if (retval == 0) {
			int i;
			if (tmp > SZbufsize) {
				retval = -EINVAL;
				break;
			}
			if (priv->timer_status == SUBUCOM_TIMER_ON) {
				retval = -EBUSY;
				break;
			}

			priv->tx->bytes = (uint32_t)tmp;
			priv->rx_bytes = (uint32_t)tmp;
			dev_info(&spi->dev, "[subucom_spi_ioctl](W) RX BYTES=%u\n", priv->rx_bytes);
			for (i = 0; i < txmulti; ++i)
				subucom_spi_setup_length(priv, i);
		}
		break;
	case SUBUCOM_IOC_WR_BUSYWAIT_UTIME:
		retval = __get_user(tmp, (__u16 __user *)arg);
		if (retval == 0) {
			/* timer disable check */
			if (priv->timer_status == SUBUCOM_TIMER_OFF) {	/* TIMER OFF */
				usec = tmp;
				priv->max_cntbusy = usec / SUBCPU_BUSYWAIT_USEC;
				if (usec < SUBCPU_BUSYWAIT_USEC) {
					priv->max_cntbusy++;
				}
				dev_info(&spi->dev, "[subucom_spi_ioctl](W) BUSY WAIT=%d[usec] cnt=%d\n", usec, priv->max_cntbusy);
			}
			else {
				retval = -EBUSY;
			}
		}
		break;
	case SUBUCOM_IOC_WR_GPIO_CONT:
		retval = __get_user(tmp, (__u32 __user *)arg);
		if (retval == 0) {
			if (!gpio_is_valid(priv->cont_gpio)) {
				retval = -EINVAL;
			}
			else {
				gpio_set_value(priv->cont_gpio, tmp);
				dev_info(&spi->dev, "[subucom_spi_ioctl](W) CONT:%u\n", tmp);
			}
		}
		break;
	case SUBUCOM_IOC_RD_GPIO_REQ:
		gp_write = gpio_get_value(priv->req_gpio);
		retval = __put_user(gp_write, (__u32 __user *)arg);
		dev_info(&spi->dev, "[subucom_spi_ioctl](R) REQ:%d\n", gp_write);
		break;
	case SUBUCOM_IOC_RD_MAX_SPEED_HZ:
		retval = __put_user(priv->speed_hz, (__u32 __user *)arg);
		break;
	case SUBUCOM_IOC_WR_MAX_SPEED_HZ:
		retval = __get_user(tmp, (__u32 __user *)arg);
		if (retval == 0) {
			uint32_t save = spi->max_speed_hz;

			spi->max_speed_hz = tmp;
			dev_info(&spi->dev, "[subucom_spi_ioctl](W) MAX SPEED HZ=%u\n", spi->max_speed_hz);
			retval = spi_setup(spi);
			if (retval >= 0)
				priv->speed_hz = tmp;
			else
				dev_err(&spi->dev, "[subucom_spi_ioctl] spi_setup() failed. %d\n", retval);
			spi->max_speed_hz = save;
		}
		break;

	default:
		/* segmented and/or Tx message I/O request */
		if (_IOC_NR(cmd) != _IOC_NR(SUBUCOM_IOC_MESSAGE(0))
		||  _IOC_DIR(cmd) != _IOC_WRITE) {
			retval = -ENOTTY;
			break;
		}

		tmp = _IOC_SIZE(cmd);
		if ((tmp % sizeof(struct subucom_ioc_transfer)) != 0) {
			retval = -EINVAL;
			break;
		}
		n_ioc = tmp / sizeof(struct subucom_ioc_transfer);
		if (n_ioc == 0)
			break;

		if (__copy_from_user(&priv->ioc, (void __user *)arg, tmp)) {
			retval = -EFAULT;
			break;
		}

		retval = subucom_setup_txbufnew(priv);
		break;
	}
	spi_dev_put(spi);
	return retval;
}
//******************************************************************************
//* name : subucom_spi_open
//******************************************************************************
static int subucom_spi_open(struct inode *inode, struct file *filp)
{
	struct subucom_spi_dev	*priv;
	int			status = -ENXIO;

	mutex_lock(&device_list_lock);

	list_for_each_entry(priv, &device_list, device_entry) {
		if (priv->devt == inode->i_rdev) {
			status = 0;
			break;
		}
	}
	if (status == 0) {
		++priv->users;
		filp->private_data = priv;
		nonseekable_open(inode, filp);
	} else {
		pr_debug("subucom_spi: nothing for minor %d\n", iminor(inode));
	}

	mutex_unlock(&device_list_lock);

	mutex_lock(&priv->rx_buf_lock);
	memset(priv->rx_bufold, 0, SZbufsize);
	mutex_unlock(&priv->rx_buf_lock);

	return status;
}
//******************************************************************************
//* name : subucom_spi_release
//******************************************************************************
static int subucom_spi_release(struct inode *inode, struct file *filp)
{
	struct subucom_spi_dev	*priv;
	int			status = 0;

	mutex_lock(&device_list_lock);

	priv = filp->private_data;
	filp->private_data = NULL;

	--priv->users;

	mutex_unlock(&device_list_lock);

	return status;
}

static const struct file_operations subucom_spi_fops = {
	.owner		= THIS_MODULE,
	.read		= subucom_spi_read,
	.write		= subucom_spi_write,
	.poll		= subucom_spi_poll,
	.unlocked_ioctl = subucom_spi_ioctl,
	.open		= subucom_spi_open,
	.release	= subucom_spi_release,
};

#ifdef FEATURE_SUBUCOM_JLCD_INIT
//******************************************************************************
//* name : joglcd_spi_rw()
//******************************************************************************
static int32_t jlcd_spi_rw(struct subucom_spi_dev *priv, uint8_t *tx_buf, size_t len)
{
	struct spi_transfer t = {
		.tx_buf	= tx_buf,
		.rx_buf	= NULL,
		.len	= len,
	};
	struct spi_message	m;
	int32_t	status;

	if (priv == NULL) {
		return -EIO;
	}

	spi_message_init(&m);
	spi_message_add_tail(&t, &m);

	/* tcss min 10ns?? 20ns ?? */
	gpio_set_value(priv->jlcd_sel_gpio, JLCD_SEL_JOG);
	status = subucom_spi_sync(priv, &m);
	/* tcsh min 10nsec */
	gpio_set_value(priv->jlcd_sel_gpio, JLCD_SEL_PANEL);

#ifdef FEATURE_SUBUCOM_JLCD_INIT_DEBUG
	printk("[jlcd_spi_rw] bits_per:%d status=%d [%02x,%02x,%02x] len=%d\n",
	            priv->spi->bits_per_word, status, tx_buf[0],tx_buf[1],tx_buf[2],(int)len);
#endif /* FEATURE_SUBUCOM_JLCD_INIT_DEBUG */
	return status;
}

//******************************************************************************
//* name : jlcd_param_init_set
//******************************************************************************
static int jlcd_param_init_set(struct subucom_spi_dev *priv)
{
	int32_t	i, status;
	uint8_t	*sdata;

	if (priv == NULL) {
		return -ENOMEM;
	}
	sdata = priv->tx_buf;
	if (sdata == NULL) {
		return -ENOMEM;
	}

	for (i = 0; i < ARRAY_SIZE(joglcd_reg_wdata); i++) {
#ifdef FEATURE_SUBUCOM_JLCD_INIT_DEBUG
		printk("[jlcd_param_init_set] Reg#:%04X, Value:%04X\n",
			joglcd_reg_wdata[i][0],
			joglcd_reg_wdata[i][1]);
#endif /* FEATURE_SUBUCOM_JLCD_INIT_DEBUG */
		switch (joglcd_reg_wdata[i][0]) {
		case 0xF040:
			msleep(40);
			break;
		case 0xF100:
			msleep(100);
			break;
		default:
			sdata[0] = 0x70;	//01110000     Primary
								//      SW ... RS=0,RW=0
			sdata[1] = (uint8_t)((joglcd_reg_wdata[i][0] >> 8) & 0x00FF);
			sdata[2] = (uint8_t)(joglcd_reg_wdata[i][0] & 0x00FF);
			status = jlcd_spi_rw(priv, sdata, 3);
			msleep(1);

			sdata[0] = 0x72;	//01110010     Primary
								//      SW ... RS=1,RW=0
			sdata[1] = (uint8_t)((joglcd_reg_wdata[i][1] >> 8) & 0x00FF);
			sdata[2] = (uint8_t)(joglcd_reg_wdata[i][1] & 0x00FF);
			status = jlcd_spi_rw(priv, sdata, 3);
			msleep(1);
			break;
		}
	}
	return status;
}
#endif /* FEATURE_SUBUCOM_JLCD_INIT */

//******************************************************************************
//* name : subucom_spi_probe
//******************************************************************************
static int subucom_spi_probe(struct spi_device *spi)
{
	int32_t		status = 0;
	int32_t		i;
	uint32_t	minor;
	struct subucom_spi_dev	*priv = NULL;
	struct subucom_tx_part	*tx;
	int32_t		ret;

	dev_info(&spi->dev, "[subucom_spi_probe] %s\n", VERSION_STRING);

	//******************************************************************************
	//** ALLOCATE memory
	//******************************************************************************
	priv = devm_kzalloc(&spi->dev, sizeof(struct subucom_spi_dev), GFP_KERNEL);
	if (!priv) {
		return -ENOMEM;
	}

	priv->rx_buf = devm_kzalloc(&spi->dev, SZbufsize * sizeof(uint8_t) * 2, GFP_KERNEL);
	if (!priv->rx_buf) {
		return -ENOMEM;
	}
	priv->rx_bufold = priv->rx_buf + SZbufsize;

	priv->tx_buf = devm_kzalloc(&spi->dev, SZbufsize * txmulti * txpart * sizeof(uint8_t) * 2, GFP_KERNEL);
	if (!priv->tx_buf) {
		return -ENOMEM;
	}
	priv->tx_bufnew = priv->tx_buf + (SZbufsize * txmulti * txpart);

	priv->tx = devm_kzalloc(&spi->dev, txmulti * sizeof(struct subucom_tx_part), GFP_KERNEL);
	if (!priv->tx) {
		return -ENOMEM;
	}

	spin_lock_init(&priv->spi_lock);
	mutex_init(&priv->transfer_lock);
	mutex_init(&priv->tx_buf_lock);
	mutex_init(&priv->rx_buf_lock);
	mutex_init(&priv->buffer_lock);
	init_waitqueue_head(&priv->spi_wait);
	INIT_LIST_HEAD(&priv->device_entry);

	dev_info(&spi->dev, "[subucom_spi_probe] minors:0x%lx\n", minors[0]);
	mutex_lock(&device_list_lock);
	minor = find_first_zero_bit(minors, N_SPI_MINORS);
	if (minor < N_SPI_MINORS) {
		// create Device
		struct device *dev;
		priv->devt = MKDEV(majorNumber, minor);
		dev = device_create(subucom_spi_class, &spi->dev, priv->devt,
							priv, "subucom_spi%d.%d",
							spi->master->bus_num, spi->chip_select);

		status = PTR_ERR_OR_ZERO(dev);
	} else {
		dev_err(&spi->dev, "[subucom_spi_probe] no minor number available!\n");
		status = -ENODEV;
	}

	if (status == 0) {
		set_bit(minor, minors);
		list_add(&priv->device_entry, &device_list);
	}
	mutex_unlock(&device_list_lock);

	priv->speed_hz = spi->max_speed_hz;

	if (status) {
		goto err_probe_alloc;
	}

	dev_info(&spi->dev, "[subucom_spi_probe] major:%d minor:%d subucom_spi:bus=%d cs=%d\n",
			majorNumber, minor,
			spi->master->bus_num, spi->chip_select);

	//******************************************************************************
	//** GPIO initialize
	//******************************************************************************
	ret = of_get_named_gpio(spi->dev.of_node, "req-gpio", 0);
	if (!gpio_is_valid(ret)) {
		status = -EINVAL;
		goto err_probe;
	}
	priv->req_gpio = ret;
	ret = devm_gpio_request(&spi->dev, priv->req_gpio, "GPIO_REQ");
	if (ret) {
		status = -EINVAL;
		goto err_probe;
	}

	ret = of_get_named_gpio(spi->dev.of_node, "cont-gpio", 0);
	if (!gpio_is_valid(ret)) {
		priv->cont_gpio = -1;
		dev_info(&spi->dev, "[subucom_spi_probe] cont-gpio not used.\n");
	}
	else {
		priv->cont_gpio = ret;
		ret = devm_gpio_request(&spi->dev, priv->cont_gpio, "GPIO_CONT");
		if (ret) {
			status = -EINVAL;
			goto err_probe;
		}
	}

	if (gpio_is_valid(priv->cont_gpio)) {
		gpio_direction_output(priv->cont_gpio, GPIO_CONT_DISABLE);
	}
	gpio_direction_input(priv->req_gpio);

#ifdef FEATURE_SUBUCOM_JLCD_INIT
	//******************************************************************************
	//** JLCD GPIO initialize
	//******************************************************************************
	{
		ret = of_get_named_gpio(spi->dev.of_node, "jlcd-sel-gpio", 0);
		if (!gpio_is_valid(ret)) {
			status = -EINVAL;
			goto err_probe;
		}
		priv->jlcd_sel_gpio = ret;

		ret = devm_gpio_request(&spi->dev, priv->jlcd_sel_gpio, "GPIO_JLCD_SEL");
		if (ret) {
			status = -EINVAL;
			goto err_probe;
		}

		gpio_direction_output(priv->jlcd_sel_gpio, JLCD_SEL_PANEL);
	}
	//******************************************************************************
	//** JLCD identification
	//******************************************************************************
	{
		/* TBD */
	}
#endif /* FEATURE_SUBUCOM_JLCD_INIT */

	dev_info(&spi->dev, "[subucom_spi_probe] SPI clock %d[Hz]\n", spi->max_speed_hz);

#ifdef FEATURE_SUBUCOM_JLCD_INIT
	//******************************************************************************
	//** JLCD SPI initialize
	//******************************************************************************
	{
		priv->subucom_mode = spi->mode;		/* save mode */
		spi->mode = JLCD_SPI_MODES;
		spi->bits_per_word = JLCD_SPI_BITS_PER_WORD;
		status = spi_setup(spi);
		if (status < 0) {
			dev_err(&spi->dev, "[subucom_spi_probe] ERROR! MSBfirst 8bits SPI\n");
			goto err_probe;
		}
		priv->spi = spi;
		subucom_spi_setup_bytes_per_word(priv);
		spi_set_drvdata(spi, priv);
		dev_info(&spi->dev, "[subucom_spi_probe] JLCD SPI device initialized\n");

		status = jlcd_param_init_set(priv);

		spin_lock_irq(&priv->spi_lock);
		priv->spi = NULL;
		spi_set_drvdata(spi, NULL);
		spin_unlock_irq(&priv->spi_lock);
		spi->mode = priv->subucom_mode;		/* restore mode */
		gpio_set_value(priv->jlcd_sel_gpio, JLCD_SEL_PANEL);
		udelay(100);						/* 100usec delay */

		if (status < 0) {
			dev_err(&spi->dev, "[subucom_spi_probe] Failed to initialize JLCD. status = %d\n", status);
		}
		else {
			dev_info(&spi->dev, "[subucom_spi_probe] Succeeded to initialize JLCD.\n");
		}
		gpio_set_value(priv->jlcd_sel_gpio, JLCD_SEL_JOG);
	}
#endif /* FEATURE_SUBUCOM_JLCD_INIT */

	dev_info(&spi->dev, "[subucom_spi_probe] for PANEL-MICRO_COM RE-setup spi_setup MSBfirst 16bits SPI\n");
	init_waitqueue_head(&priv->req_wq);
	priv->req_irq = gpio_to_irq(priv->req_gpio);
	ret = devm_request_threaded_irq(&spi->dev,
						priv->req_irq,
						req_hard_handler,
						req_handler,
						IRQF_TRIGGER_RISING | IRQF_ONESHOT,
						"GPIO_REQ_IRQ",
						priv);
	if (ret) {
		status = -EINVAL;
		dev_err(&spi->dev, "[subucom_spi_probe] request_irq %d\n", priv->req_irq);
		goto err_probe;
	}

	spi->bits_per_word = SUBUCOM_SPI_BITS_PER_WORD;
	status	= spi_setup(spi);
	if (status < 0) {
		dev_err(&spi->dev, "[subucom_spi_probe] ERROR! MSBfirst 16bits SPI\n");
		goto err_probe;
	}
	priv->spi = spi;
	subucom_spi_setup_bytes_per_word(priv);
	spi_set_drvdata(spi, priv);
	dev_info(&spi->dev, "[subucom_spi_probe] PANEL SPI device initialized\n");
	priv->rx_bytes = SUBUCOM_SPI_DATA_SIZE_NORMAL;

	priv->tx_multi_num	= 1;
	priv->tx_multi_id	= 0;
	tx = priv->tx;
	for (i = 0; i < txmulti; ++i) {
		tx->part_num= 1;
		tx->part_id	= 0;
		tx->updated	= 0;
		tx->bytes	= SUBUCOM_SPI_DATA_SIZE_NORMAL;
		subucom_spi_setup_length(priv, i);
		tx->buf		= priv->tx_buf + (SZbufsize * i * txpart);
		tx->bufnew	= priv->tx_bufnew + (SZbufsize * i * txpart);
		++tx;
	}

	pm_ready = true;
	priv->timer_interval = SUBUCOM_SPI_TIMER_INTERVAL;
	status = subucom_spi_thread_setup(priv);
	if (status) {
		goto err_probe;
	}
	subucom_spi_timer_init(priv);
	if (DEFtmstat) {
		subucom_spi_timer_start(priv);
	}
#ifdef DEBUG_REQ_TIMEOUT
	priv->timer_interval_req_timeout = SUBUCOM_SPI_TIMER_INTERVAL_REQ_TIMEOUT;
	subucom_spi_timer_req_timeout_init(priv);
#endif
	priv->max_cntbusy = CNT_SUBCPU_BUSYWAIT;		//10cnt X 50usec = 500usec

#ifdef SUBUCOM_SPI_AGEING_TEST
	priv->rx_count = 0;
	priv->rx_success_count = 0;
	priv->rx_fail_count = 0;
#endif
	return 0;

err_probe:
	mutex_lock(&device_list_lock);
	list_del(&priv->device_entry);
	device_destroy(subucom_spi_class, priv->devt);
	clear_bit(minor, minors);
	mutex_unlock(&device_list_lock);

err_probe_alloc:
	return status;
}
//******************************************************************************
//* name : subucom_spi_remove
//******************************************************************************
static int subucom_spi_remove(struct spi_device *spi)
{
	struct subucom_spi_dev	*priv = spi_get_drvdata(spi);

	subucom_spi_thread_kill(priv);
	subucom_spi_timer_stop(priv);
#ifdef DEBUG_REQ_TIMEOUT
	subucom_spi_timer_req_timeout_stop(priv);
#endif

	spin_lock_irq(&priv->spi_lock);
	priv->spi = NULL;
	spi_set_drvdata(spi, NULL);
	spin_unlock_irq(&priv->spi_lock);

	mutex_lock(&device_list_lock);
	list_del(&priv->device_entry);
	device_destroy(subucom_spi_class, priv->devt);
	clear_bit(MINOR(priv->devt), minors);

	mutex_unlock(&device_list_lock);

	return 0;
}

static const struct of_device_id subucom_of_match[] = {
	{ .compatible = "alphatheta,subucom_spi" },
	{ }
};
MODULE_DEVICE_TABLE(of, subucom_of_match);

#ifdef CONFIG_PM
static int subucom_spi_suspend(struct device *dev)
{
	dev_info(dev, "[%s]\n", __func__);
	pm_ready = false;
	return 0;
}

static int subucom_spi_resume(struct device *dev)
{
	dev_info(dev, "[%s]\n", __func__);
	pm_ready = true;
	return 0;
}

static SIMPLE_DEV_PM_OPS(subucom_spi_pm_ops,
			subucom_spi_suspend, subucom_spi_resume);
#endif

static struct spi_driver subucom_spi_driver = {
	.driver  = {
		.name   = DRV_NAME,
		.of_match_table = of_match_ptr(subucom_of_match),

		.owner  = THIS_MODULE,
#ifdef CONFIG_PM
		.pm = &subucom_spi_pm_ops,
#endif
	},
	.probe  = subucom_spi_probe,
	.remove = subucom_spi_remove,
};

//******************************************************************************
//* name : subucom_spi_init
//******************************************************************************
static int __init subucom_spi_init(void)
{
	int32_t	status;

	pr_info("subucom SPI user interface\n");

	_debug = 0;

	majorNumber = register_chrdev(0, subucom_spi_driver.driver.name, &subucom_spi_fops);
	if (majorNumber < 0) {
		pr_err("[subucom_spi_init] ERROR=%d register_chrdev\n", majorNumber);
		return majorNumber;
	}
	pr_info("[subucom_spi_init] Major=%d register_chrdev\n", majorNumber);

	subucom_spi_class = class_create(THIS_MODULE, DRV_NAME"class");
	if (IS_ERR(subucom_spi_class)) {
		unregister_chrdev(majorNumber, subucom_spi_driver.driver.name);
		pr_err("[subucom_spi_init] ERROR class_create\n");
		return PTR_ERR(subucom_spi_class);
	}

	status = spi_register_driver(&subucom_spi_driver);
	if (status < 0) {
		class_destroy(subucom_spi_class);
		unregister_chrdev(majorNumber, subucom_spi_driver.driver.name);
		pr_err("[subucom_spi_init] ERROR=%d spi_register_driver\n", status);
	}
	return status;
}
module_init(subucom_spi_init);

//******************************************************************************
//* name : subucom_spi_exit
//******************************************************************************
static void __exit subucom_spi_exit(void)
{
	spi_unregister_driver(&subucom_spi_driver);
	device_destroy(subucom_spi_class, MKDEV(majorNumber, 0));
	class_destroy(subucom_spi_class);
	unregister_chrdev(majorNumber, subucom_spi_driver.driver.name);
}
module_exit(subucom_spi_exit);

MODULE_AUTHOR("Yasunori Shibata <yasunori.shibata@pioneerdj.com>");
MODULE_DESCRIPTION("subucom SPI user interface");
MODULE_LICENSE("GPL");
MODULE_ALIAS("spi:subucom_spi");
