/* SPDX-License-Identifier: GPL-2.0 */

#include <dt-bindings/gpio/gpio.h>
#include <linux/gpio.h>
#include <linux/of_gpio.h>
#include <linux/module.h>
#include <linux/kernel.h>
#include <linux/init.h>
#include <linux/platform_device.h>
#include <linux/fb.h>
#include <linux/backlight.h>
#include <linux/err.h>
#include <linux/pwm.h>
#include <linux/pwm_backlight.h>
#include <linux/slab.h>
#include <linux/delay.h>

#if 1
#define USB_5V_DBG(args...)   usb_5v_info(args)
#else
#define USB_5V_DBG(args...)
#endif

#define usb_5v_info(fmt, args...)   pr_info("usb_5v: "fmt, ##args)

struct usb_5v_gpio_ctrl {
	int host1_gpio;
	int typec1_gpio;

	u8 host1_en_level;
	u8 typec1_en_level;
};

static int usb_5v_ctrl_probe(struct platform_device *pdev)
{
	int ret = 0;
	enum of_gpio_flags flags;
	struct usb_5v_gpio_ctrl *u5c;
	struct device_node *np = pdev->dev.of_node;

	u5c = devm_kzalloc(&pdev->dev, sizeof(*u5c), GFP_KERNEL);
	if (!u5c) {
		ret = -ENOMEM;
		USB_5V_DBG("No memory space!\n");
		return ret;
	}

	platform_set_drvdata(pdev, u5c);

	u5c->host1_gpio =
		of_get_named_gpio_flags(np, "host1-gpio", 0, &flags);
	if (gpio_is_valid(u5c->host1_gpio)) {
		u5c->host1_en_level = (flags & OF_GPIO_ACTIVE_LOW) ? 0 : 1;
		ret = devm_gpio_request_one(&pdev->dev, u5c->host1_gpio,
					    GPIOF_DIR_OUT, NULL);
		if (ret) {
			dev_err(&pdev->dev, "failed to request host1-gpio!\n");
			return ret;
		}

		gpio_direction_output(u5c->host1_gpio, u5c->host1_en_level);
	}

	u5c->typec1_gpio =
		of_get_named_gpio_flags(np, "typec1-gpio", 0, &flags);
	if (gpio_is_valid(u5c->typec1_gpio)) {
		u5c->typec1_en_level = (flags & OF_GPIO_ACTIVE_LOW) ? 0 : 1;
		ret = devm_gpio_request_one(&pdev->dev, u5c->typec1_gpio,
					    GPIOF_DIR_OUT, NULL);
		if (ret) {
			dev_err(&pdev->dev, "failed to request typec1-gpio!\n");
			return ret;
		}

		gpio_direction_output(u5c->typec1_gpio, u5c->typec1_en_level);
	}

	USB_5V_DBG("%s successfully!\n", __func__);

	return ret;
}

static void usb_5v_ctrl_shutdown(struct platform_device *pdev)
{
	struct usb_5v_gpio_ctrl *u5c = platform_get_drvdata(pdev);

	USB_5V_DBG("%s enter\n", __func__);

	if (gpio_is_valid(u5c->host1_gpio))
		gpio_set_value(u5c->host1_gpio, !u5c->host1_en_level);
	if (gpio_is_valid(u5c->typec1_gpio))
		gpio_set_value(u5c->typec1_gpio, !u5c->typec1_en_level);

	return;
}

static struct of_device_id usb_5v_ctrl_of_match[] = {
	{ .compatible = "usb_5v_ctrl" },
	{ }
};
MODULE_DEVICE_TABLE(of, usb_5v_ctrl_of_match);

static struct platform_driver usb_5v_ctrl_driver = {
	.driver	= {
		.name = "usb_5v_ctrl",
		.owner = THIS_MODULE,
		.of_match_table = of_match_ptr(usb_5v_ctrl_of_match),
	},
	.probe    = usb_5v_ctrl_probe,
	.shutdown = usb_5v_ctrl_shutdown,
};
module_platform_driver(usb_5v_ctrl_driver);

MODULE_DESCRIPTION("USB host 5v power driver");
MODULE_LICENSE("GPL");
MODULE_ALIAS("platform: usb_5v_ctrl");
