// SPDX-License-Identifier: GPL-2.0
/*
 * xc7080 driver
 *
 * Copyright (C) 2017 Fuzhou Rockchip Electronics Co., Ltd.
 * V0.0X01.0X01 add enum_frame_interval function.
 */

#include <linux/clk.h>
#include <linux/device.h>
#include <linux/delay.h>
#include <linux/gpio/consumer.h>
#include <linux/i2c.h>
#include <linux/module.h>
#include <linux/pm_runtime.h>
#include <linux/regulator/consumer.h>
#include <linux/sysfs.h>
#include <linux/slab.h>
#include <linux/rk-camera-module.h>
#include <media/media-entity.h>
#include <media/v4l2-async.h>
#include <media/v4l2-ctrls.h>
#include <media/v4l2-subdev.h>
#include <linux/pinctrl/consumer.h>

#define DRIVER_VERSION			KERNEL_VERSION(0, 0x01, 0x1)

#define XC7080_LINK_FREQ_352MHZ		352000000
/* pixel rate = link frequency * 2 * lanes / BITS_PER_SAMPLE */
#define XC7080_PIXEL_RATE		(XC7080_LINK_FREQ_352MHZ * 2 * 2 / 8)
#define XC7080_XVCLK_FREQ		24000000

#define XC7080_CHIP_ID				0x71
#define XC7080_REG_CHIP_ID		0xfffb

#define XC530_CHIP_ID				0x20
#define XC530_REG_CHIP_ID		0x00

#define XC7080_VTS_MAX			0xFFFF

#define XC7080_REG_TEST_PATTERN		0x0090
#define XC7080_TEST_PATTERN_ENABLE	0x3a
#define XC7080_TEST_PATTERN_DISABLE	0x38

#define REG_NULL_XC7080			0xFFFF
#define REG_NULL_XC530			0XFF

#define XC7080_REG_VALUE_08BIT		1
#define XC7080_REG_VALUE_16BIT		2
#define XC7080_REG_VALUE_24BIT		3

#define XC7080_LANES			2
#define XC7080_PIX_FORMAT		MEDIA_BUS_FMT_YUYV8_2X8

#define OF_CAMERA_PINCTRL_STATE_DEFAULT	"rockchip,camera_default"
#define OF_CAMERA_PINCTRL_STATE_SLEEP	"rockchip,camera_sleep"

#define XC7080_NAME			"xc7080_external_isp"
#define EXTREANL_ISP		"xc7080"

static const char * const xc7080_supply_names[] = {
	"avdd",		/* Analog power */
	"dovdd",	/* Digital I/O power */
	"dvdd",		/* Digital core power */
};

#define XC7080_NUM_SUPPLIES ARRAY_SIZE(xc7080_supply_names)

struct regval {
	u16 addr;
	u8 val;
};

struct regval1 {
	u8 addr;
	u8 val;
};

struct xc7080_mode {
	u32 width;
	u32 height;
	struct v4l2_fract max_fps;
	u32 hts_def;
	u32 vts_def;
	u32 exp_def;
	const struct regval *xc7080_reg_list;
	const struct regval1 *xc530_reg_list;
};

struct xc7080 {
	struct i2c_client	*client;
	struct i2c_client   *sub_client;
	struct clk		*xvclk;
	struct gpio_desc	*reset_gpio;
	struct gpio_desc	*pwdn_gpio;
	struct regulator_bulk_data supplies[XC7080_NUM_SUPPLIES];

	struct pinctrl		*pinctrl;
	struct pinctrl_state	*pins_default;
	struct pinctrl_state	*pins_sleep;

	struct v4l2_subdev	subdev;
	struct media_pad	pad;
	struct v4l2_ctrl_handler ctrl_handler;
	struct v4l2_ctrl	*exposure;
	struct v4l2_ctrl	*anal_gain;
	struct v4l2_ctrl	*digi_gain;
	struct v4l2_ctrl	*hblank;
	struct v4l2_ctrl	*vblank;
	struct v4l2_ctrl	*test_pattern;
	struct mutex		mutex;
	bool			streaming;
	bool			power_on;
	const struct xc7080_mode *cur_mode;
	u32			module_index;
	const char		*module_facing;
	const char		*module_name;
	const char		*len_name;
};

#define to_xc7080(sd) container_of(sd, struct xc7080, subdev)

/*
 * XC7080 External ISP
 * Xclk 24Mhz
 * MIPI 2lane 1920*1080 704Mbps
 */
static const struct regval xc7080_1920x1080_regs[] = {
	// XC7080 setting
	//init
	{0xfffd, 0x80},
	{0xfffe, 0x50},
	{0x001c, 0xff},
	{0x001d, 0xff},
	{0x001e, 0xff},
	{0x001f, 0xff},
	{0x0018, 0x00},
	{0x0019, 0x00},
	{0x001a, 0x00},
	{0x001b, 0x00},
	{0xfffe, 0x50},
	{0x0050, 0x03},
	{0x0054, 0x03},
	{0x0058, 0x02},
	{0x0030, 0x44},//44,
	{0x0031, 0x91},//58,
	{0x0032, 0x32},//34,
	{0x0033, 0x70},//70,
	{0x0020, 0x02},    //0x05   02
	{0x0021, 0x0d},
	{0x0022, 0x01},
	{0x0023, 0x85},
	{0x0024, 0x0A},    //0x02 01
	{0x0025, 0x00},    //0x0d
	{0x0026, 0x01},
	{0x0027, 0x06},
	{0x0028, 0x01},
	{0x0029, 0x00},
	{0x002a, 0x02},
	{0x002b, 0x05},
	{0x002e, 0x05},
	{0x00bc, 0x19},
	{0x0090, 0x38},
	//datapath
	{0xfffe, 0x26},
	{0x8000, 0x1d},
	{0x8001, 0x00},
	{0x8002, 0x0A},
	{0x8003, 0xA0},
	{0x8004, 0x05},
	{0x8010, 0x04},
	{0x8012, 0x00},
	{0x8013, 0x0A},
	{0x8014, 0xA0},
	{0x8015, 0x05},
	{0x8016, 0x00},
	{0x8017, 0x00},
	{0x8018, 0x00},
	{0x8019, 0x00},
	{0xfffe, 0x30},
	{0x1900, 0x00},
	{0x1901, 0x00},
	{0x1902, 0x00},
	{0x1903, 0x00},
	{0x1904, 0x0A},
	{0x1905, 0x00},
	{0x1906, 0x05},
	{0x1907, 0xA0},
	{0x1908, 0x00},
	{0xfffe, 0x2c},
	{0x0000, 0x00},
	{0x0001, 0x07},
	{0x0002, 0x80},
	{0x0004, 0x04},
	{0x0005, 0x38},
	{0x0008, 0x10},
	{0x0044, 0x08},
	{0x0045, 0x04},
	{0x0048, 0x0E},
	{0x0049, 0xF0},
	{0xfffe, 0x26},
	{0x2019, 0x07},
	{0x201a, 0x80},
	{0x201b, 0x04},
	{0x201c, 0x38},
	{0x201d, 0x00},
	{0x201e, 0x00},
	{0x201f, 0x00},
	{0x2020, 0x00},
	{0x2015, 0x81},
	{0x2017, 0x1E},
	{0x2018, 0x00},
	{0x2023, 0x03},
	{0x0000, 0x20},
	{0x0009, 0xc4},
	{0xfffe, 0x30},
	{0x0000, 0x01},
	{0x0001, 0x10},
	{0x0002, 0x10},
	{0x0003, 0x20},
	{0x0004, 0x10},
	{0x0050, 0x20},
	{0x0019, 0x08},
	{0x005e, 0xFF},
	{0x005f, 0x09},
	{0x0060, 0x9F},
	{0x0061, 0x05},
	{0x0064, 0x80},
	{0x0065, 0x07},
	{0x0066, 0x38},
	{0x0067, 0x04},
	{0x0006, 0x0A},
	{0x0007, 0x00},
	{0x0008, 0x05},
	{0x0009, 0xA0},
	{0x000a, 0x07},
	{0x000b, 0x80},
	{0x000c, 0x04},
	{0x000d, 0x38},
	{0x0715, 0x04},
	{0xfffe, 0x50},
	{0x0226, 0x02},
	{0xfffe, 0x2e},
	{0x0000, 0x42},
	{0x0001, 0xee},
	{0x0003, 0x01},
	{0x0004, 0xa0},
	{0xfffe, 0x25},
	{0x0002, 0x80},
	{0xfffe, 0x30},
	{0x0004, 0x10},
	{0x2300, 0x0b},
	{0xfffe, 0x30},
	{0x1a0f, 0x5a},
	{0xfffd, 0x80},
	{0xfffe, 0x50},
	{0x000e, 0x54},
	{0xfffe, 0x14},
	{0x0006, 0x09},
	{0x0007, 0x44},
	{0x0014, 0x00},
	{0x0015, 0x14},
	{0x0016, 0x26},
	{0x0017, 0x94},
	{0x2114, 0x1b},
	{0x2115, 0x23},
	{0x2116, 0x2a},
	{0x2117, 0x2c},
	{0x2118, 0x37},
	{0x2119, 0x44},
	{0x211a, 0x3c},
	{0x211b, 0x4d},
	{0x211c, 0x5d},
	{0x211d, 0x4b},
	{0x211e, 0x5f},
	{0x211f, 0x74},
	{0x2120, 0x5a},
	{0x2121, 0x70},
	{0x2122, 0x87},
	{0x2123, 0x64},
	{0x2124, 0x7d},
	{0x2125, 0x96},
	{0x2126, 0x6c},
	{0x2127, 0x87},
	{0x2128, 0xa2},
	{0x2129, 0x71},
	{0x212a, 0x8e},
	{0x212b, 0xab},
	{0x212c, 0x75},
	{0x212d, 0x93},
	{0x212e, 0xb1},
	{0x212f, 0x78},
	{0x2130, 0x97},
	{0x2131, 0xb6},
	{0x2132, 0x7a},
	{0x2133, 0x9a},
	{0x2134, 0xb9},
	{0x2135, 0x7d},
	{0x2136, 0x9c},
	{0x2137, 0xbc},
	{0x2138, 0x7d},
	{0x2139, 0x9e},
	{0x213a, 0xbd},
	{0x213b, 0x7e},
	{0x213c, 0x9f},
	{0x213d, 0xbf},
	{0x213e, 0x7f},
	{0x213f, 0xa0},
	{0x2140, 0xc0},
	{0x2141, 0x7f},
	{0x2142, 0xa0},
	{0x2143, 0xc0},
	{0x2144, 0x00},
	{0x2145, 0x00},
	{0x2146, 0x04},
	{0x2147, 0x00},
	{0x2148, 0x00},
	{0x2149, 0x00},
	{0x214a, 0x00},
	{0x214b, 0x68},
	{0x214c, 0x00},
	{0x214d, 0x00},
	{0x214e, 0x20},
	{0x214f, 0x00},
	{0x2150, 0x00},
	{0x2151, 0x00},
	{0x2152, 0x00},
	{0x2153, 0x67},
	{0x2154, 0x00},
	{0x2155, 0x00},
	{0x2156, 0x30},
	{0x2157, 0x00},
	{0x2158, 0x00},
	{0x2159, 0x00},
	{0x215a, 0x00},
	{0x215b, 0x66},
	{0x215c, 0x00},
	{0x215d, 0x00},
	{0x215e, 0x37},
	{0x215f, 0xd0},
	{0x2160, 0x00},
	{0x2161, 0x00},
	{0x2162, 0x00},
	{0x2163, 0x65},
	{0x2164, 0x00},
	{0x2165, 0x00},
	{0x2166, 0x40},
	{0x2167, 0x00},
	{0x2168, 0x00},
	{0x2169, 0x00},
	{0x216a, 0x00},
	{0x216b, 0x55},
	{0x216c, 0x00},
	{0x216d, 0x00},
	{0x216e, 0x58},
	{0x216f, 0x00},
	{0x2170, 0x00},
	{0x2171, 0x00},
	{0x2172, 0x00},
	{0x2173, 0x54},
	{0x2174, 0x00},
	{0x2175, 0x00},
	{0x2176, 0x60},
	{0x2177, 0x00},
	{0x2178, 0x00},
	{0x2179, 0x00},
	{0x217a, 0x00},
	{0x217b, 0x44},
	{0x217c, 0x00},
	{0x217d, 0x00},
	{0x217e, 0x6c},
	{0x217f, 0x00},
	{0x2180, 0x00},
	{0x2181, 0x00},
	{0x2182, 0x00},
	{0x2183, 0x43},
	{0x2184, 0x00},
	{0x2185, 0x00},
	{0x2186, 0x73},
	{0x2187, 0x08},
	{0x2188, 0x00},
	{0x2189, 0x00},
	{0x218a, 0x00},
	{0x218b, 0x33},
	{0x218c, 0x00},
	{0x218d, 0x00},
	{0x218e, 0x7a},
	{0x218f, 0xd8},
	{0x2190, 0x00},
	{0x2191, 0x00},
	{0x2192, 0x00},
	{0x2193, 0x32},
	{0x2194, 0x00},
	{0x2195, 0x00},
	{0x2196, 0x80},
	{0x2197, 0x00},
	{0x2198, 0x00},
	{0x2199, 0x00},
	{0x219a, 0x00},
	{0x219b, 0x22},
	{0x219c, 0x9c},
	{0x219d, 0x21},
	{0x219e, 0xff},
	{0x219f, 0xf0},
	{0x21a0, 0xd4},
	{0x21a1, 0x01},
	{0x21a2, 0x48},
	{0x21a3, 0x00},
	{0x21a4, 0xd4},
	{0x21a5, 0x01},
	{0x21a6, 0x50},
	{0x21a7, 0x04},
	{0x21a8, 0xd4},
	{0x21a9, 0x01},
	{0x21aa, 0x60},
	{0x21ab, 0x08},
	{0x21ac, 0xd4},
	{0x21ad, 0x01},
	{0x21ae, 0x70},
	{0x21af, 0x0c},
	{0x21b0, 0x07},
	{0x21b1, 0xfb},
	{0x21b2, 0x07},
	{0x21b3, 0x82},
	{0x21b4, 0x15},
	{0x21b5, 0x00},
	{0x21b6, 0x00},
	{0x21b7, 0x00},
	{0x21b8, 0x85},
	{0x21b9, 0xcb},
	{0x21ba, 0x00},
	{0x21bb, 0x08},
	{0x21bc, 0x84},
	{0x21bd, 0x8e},
	{0x21be, 0x01},
	{0x21bf, 0x0c},
	{0x21c0, 0xb8},
	{0x21c1, 0x64},
	{0x21c2, 0x00},
	{0x21c3, 0x03},
	{0x21c4, 0xbc},
	{0x21c5, 0x43},
	{0x21c6, 0x02},
	{0x21c7, 0x00},
	{0x21c8, 0x10},
	{0x21c9, 0x00},
	{0x21ca, 0x00},
	{0x21cb, 0x0d},
	{0x21cc, 0xa9},
	{0x21cd, 0x8b},
	{0x21ce, 0x00},
	{0x21cf, 0x00},
	{0x21d0, 0xb8},
	{0x21d1, 0x64},
	{0x21d2, 0x00},
	{0x21d3, 0x07},
	{0x21d4, 0xb8},
	{0x21d5, 0x63},
	{0x21d6, 0x00},
	{0x21d7, 0x48},
	{0x21d8, 0x9c},
	{0x21d9, 0x63},
	{0x21da, 0xff},
	{0x21db, 0xf0},
	{0x21dc, 0xbc},
	{0x21dd, 0xa3},
	{0x21de, 0x00},
	{0x21df, 0x7},
	{0x21e0, 0x10},
	{0x21e1, 0x00},
	{0x21e2, 0x00},
	{0x21e3, 0x1c},
	{0x21e4, 0x9d},
	{0x21e5, 0x40},
	{0x21e6, 0x00},
	{0x21e7, 0x08},
	{0x21e8, 0xbc},
	{0x21e9, 0xa3},
	{0x21ea, 0x00},
	{0x21eb, 0x50},
	{0x21ec, 0x10},
	{0x21ed, 0x00},
	{0x21ee, 0x00},
	{0x21ef, 0x19},
	{0x21f0, 0xa9},
	{0x21f1, 0x43},
	{0x21f2, 0x00},
	{0x21f3, 0x00},
	{0x21f4, 0x00},
	{0x21f5, 0x00},
	{0x21f6, 0x00},
	{0x21f7, 0x17},
	{0x21f8, 0x9d},
	{0x21f9, 0x40},
	{0x21fa, 0x00},
	{0x21fb, 0x50},
	{0x21fc, 0xbc},
	{0x21fd, 0x43},
	{0x21fe, 0x04},
	{0x21ff, 0x00},
	{0x2200, 0x10},
	{0x2201, 0x00},
	{0x2202, 0x00},
	{0x2203, 0x05},
	{0x2204, 0xbc},
	{0x2205, 0x43},
	{0x2206, 0x08},
	{0x2207, 0x00},
	{0x2208, 0xb8},
	{0x2209, 0x64},
	{0x220a, 0x00},
	{0x220b, 0x06},
	{0x220c, 0x03},
	{0x220d, 0xff},
	{0x220e, 0xff},
	{0x220f, 0xf4},
	{0x2210, 0xb8},
	{0x2211, 0x63},
	{0x2212, 0x00},
	{0x2213, 0x48},
	{0x2214, 0x0c},
	{0x2215, 0x00},
	{0x2216, 0x00},
	{0x2217, 0x08},
	{0x2218, 0xbc},
	{0x2219, 0x43},
	{0x221a, 0x10},
	{0x221b, 0x00},
	{0x221c, 0x10},
	{0x221d, 0x00},
	{0x221e, 0x00},
	{0x221f, 0x0a},
	{0x2220, 0x15},
	{0x2221, 0x00},
	{0x2222, 0x00},
	{0x2223, 0x00},
	{0x2224, 0xe0},
	{0x2225, 0x63},
	{0x2226, 0x18},
	{0x2227, 0x00},
	{0x2228, 0xb8},
	{0x2229, 0x63},
	{0x222a, 0x00},
	{0x222b, 0x48},
	{0x222c, 0x03},
	{0x222d, 0xff},
	{0x222e, 0xff},
	{0x222f, 0xec},
	{0x2230, 0x9c},
	{0x2231, 0x63},
	{0x2232, 0x00},
	{0x2233, 0x20},
	{0x2234, 0xb8},
	{0x2235, 0x64},
	{0x2236, 0x00},
	{0x2237, 0x05},
	{0x2238, 0xb8},
	{0x2239, 0x63},
	{0x223a, 0x00},
	{0x223b, 0x48},
	{0x223c, 0x03},
	{0x223d, 0xff},
	{0x223e, 0xff},
	{0x223f, 0xe8},
	{0x2240, 0x9c},
	{0x2241, 0x63},
	{0x2242, 0x00},
	{0x2243, 0x10},
	{0x2244, 0xb8},
	{0x2245, 0x63},
	{0x2246, 0x00},
	{0x2247, 0x48},
	{0x2248, 0x03},
	{0x2249, 0xff},
	{0x224a, 0xff},
	{0x224b, 0xe5},
	{0x224c, 0x9c},
	{0x224d, 0x63},
	{0x224e, 0x00},
	{0x224f, 0x30},
	{0x2250, 0xa8},
	{0x2251, 0x6c},
	{0x2252, 0x00},
	{0x2253, 0x00},
	{0x2254, 0x9c},
	{0x2255, 0x80},
	{0x2256, 0x00},
	{0x2257, 0xef},
	{0x2258, 0x07},
	{0x2259, 0xfb},
	{0x225a, 0x06},
	{0x225b, 0xab},
	{0x225c, 0x9c},
	{0x225d, 0xa0},
	{0x225e, 0x00},
	{0x225f, 0x01},
	{0x2260, 0x94},
	{0x2261, 0x8e},
	{0x2262, 0x00},
	{0x2263, 0xe8},
	{0x2264, 0xa8},
	{0x2265, 0x6c},
	{0x2266, 0x00},
	{0x2267, 0x00},
	{0x2268, 0x07},
	{0x2269, 0xfb},
	{0x226a, 0x06},
	{0x226b, 0xa7},
	{0x226c, 0xa4},
	{0x226d, 0xaa},
	{0x226e, 0x00},
	{0x226f, 0xff},
	{0x2270, 0xa8},
	{0x2271, 0x6c},
	{0x2272, 0x00},
	{0x2273, 0x00},
	{0x2274, 0x9c},
	{0x2275, 0x80},
	{0x2276, 0x00},
	{0x2277, 0x09},
	{0x2278, 0x07},
	{0x2279, 0xfb},
	{0x227a, 0x06},
	{0x227b, 0xa3},
	{0x227c, 0x9c},
	{0x227d, 0xa0},
	{0x227e, 0x00},
	{0x227f, 0x01},
	{0x2280, 0x85},
	{0x2281, 0x21},
	{0x2282, 0x00},
	{0x2283, 0x00},
	{0x2284, 0x85},
	{0x2285, 0x41},
	{0x2286, 0x00},
	{0x2287, 0x04},
	{0x2288, 0x85},
	{0x2289, 0x81},
	{0x228a, 0x00},
	{0x228b, 0x08},
	{0x228c, 0x85},
	{0x228d, 0xc1},
	{0x228e, 0x00},
	{0x228f, 0x0c},
	{0x2290, 0x44},
	{0x2291, 0x00},
	{0x2292, 0x48},
	{0x2293, 0x00},
	{0x2294, 0x9c},
	{0x2295, 0x21},
	{0x2296, 0x00},
	{0x2297, 0x10},
	{0x2298, 0x9c},
	{0x2299, 0x21},
	{0x229a, 0xff},
	{0x229b, 0xf0},
	{0x229c, 0xd4},
	{0x229d, 0x01},
	{0x229e, 0x48},
	{0x229f, 0x00},
	{0x22a0, 0xd4},
	{0x22a1, 0x01},
	{0x22a2, 0x50},
	{0x22a3, 0x04},
	{0x22a4, 0xd4},
	{0x22a5, 0x01},
	{0x22a6, 0x60},
	{0x22a7, 0x08},
	{0x22a8, 0xd4},
	{0x22a9, 0x01},
	{0x22aa, 0x70},
	{0x22ab, 0x0c},
	{0x22ac, 0x07},
	{0x22ad, 0xfb},
	{0x22ae, 0x07},
	{0x22af, 0x43},
	{0x22b0, 0x15},
	{0x22b1, 0x00},
	{0x22b2, 0x00},
	{0x22b3, 0x00},
	{0x22b4, 0x85},
	{0x22b5, 0xcb},
	{0x22b6, 0x00},
	{0x22b7, 0x08},
	{0x22b8, 0x18},
	{0x22b9, 0xc0},
	{0x22ba, 0x00},
	{0x22bb, 0x14},
	{0x22bc, 0xa8},
	{0x22bd, 0xc6},
	{0x22be, 0x5d},
	{0x22bf, 0x00},
	{0x22c0, 0x84},
	{0x22c1, 0xee},
	{0x22c2, 0x01},
	{0x22c3, 0x08},
	{0x22c4, 0x94},
	{0x22c5, 0xc6},
	{0x22c6, 0x00},
	{0x22c7, 0x00},
	{0x22c8, 0xb8},
	{0x22c9, 0xe7},
	{0x22ca, 0x00},
	{0x22cb, 0x44},
	{0x22cc, 0xa9},
	{0x22cd, 0x8b},
	{0x22ce, 0x00},
	{0x22cf, 0x00},
	{0x22d0, 0x9d},
	{0x22d1, 0x06},
	{0x22d2, 0xff},
	{0x22d3, 0xfe},
	{0x22d4, 0xe0},
	{0x22d5, 0xe6},
	{0x22d6, 0x38},
	{0x22d7, 0x02},
	{0x22d8, 0x9c},
	{0x22d9, 0x80},
	{0x22da, 0x00},
	{0x22db, 0xef},
	{0x22dc, 0xa4},
	{0x22dd, 0xc6},
	{0x22de, 0xff},
	{0x22df, 0xff},
	{0x22e0, 0xa5},
	{0x22e1, 0x47},
	{0x22e2, 0xff},
	{0x22e3, 0xff},
	{0x22e4, 0xa8},
	{0x22e5, 0x6b},
	{0x22e6, 0x00},
	{0x22e7, 0x00},
	{0x22e8, 0x9c},
	{0x22e9, 0xa0},
	{0x22ea, 0x00},
	{0x22eb, 0x01},
	{0x22ec, 0xbc},
	{0x22ed, 0x4a},
	{0x22ee, 0x00},
	{0x22ef, 0x03},
	{0x22f0, 0x10},
	{0x22f1, 0x00},
	{0x22f2, 0x00},
	{0x22f3, 0x03},
	{0x22f4, 0x9c},
	{0x22f5, 0xc6},
	{0x22f6, 0xff},
	{0x22f7, 0xfe},
	{0x22f8, 0x9d},
	{0x22f9, 0x40},
	{0x22fa, 0x00},
	{0x22fb, 0x04},
	{0x22fc, 0xe5},
	{0x22fd, 0xaa},
	{0x22fe, 0x30},
	{0x22ff, 0x00},
	{0x2300, 0x10},
	{0x2301, 0x00},
	{0x2302, 0x00},
	{0x2303, 0x03},
	{0x2304, 0x15},
	{0x2305, 0x00},
	{0x2306, 0x00},
	{0x2307, 0x00},
	{0x2308, 0xa5},
	{0x2309, 0x48},
	{0x230a, 0xff},
	{0x230b, 0xff},
	{0x230c, 0x07},
	{0x230d, 0xfb},
	{0x230e, 0x06},
	{0x230f, 0x7e},
	{0x2310, 0x15},
	{0x2311, 0x00},
	{0x2312, 0x00},
	{0x2313, 0x00},
	{0x2314, 0x94},
	{0x2315, 0x8e},
	{0x2316, 0x00},
	{0x2317, 0xc8},
	{0x2318, 0xb8},
	{0x2319, 0xaa},
	{0x231a, 0x00},
	{0x231b, 0x48},
	{0x231c, 0x07},
	{0x231d, 0xfb},
	{0x231e, 0x06},
	{0x231f, 0x7a},
	{0x2320, 0xa8},
	{0x2321, 0x6c},
	{0x2322, 0x00},
	{0x2323, 0x00},
	{0x2324, 0x94},
	{0x2325, 0x8e},
	{0x2326, 0x00},
	{0x2327, 0xca},
	{0x2328, 0xa8},
	{0x2329, 0x6c},
	{0x232a, 0x00},
	{0x232b, 0x00},
	{0x232c, 0x07},
	{0x232d, 0xfb},
	{0x232e, 0x06},
	{0x232f, 0x76},
	{0x2330, 0xa4},
	{0x2331, 0xaa},
	{0x2332, 0x00},
	{0x2333, 0xff},
	{0x2334, 0xa8},
	{0x2335, 0x6c},
	{0x2336, 0x00},
	{0x2337, 0x00},
	{0x2338, 0x9c},
	{0x2339, 0x80},
	{0x233a, 0x00},
	{0x233b, 0x09},
	{0x233c, 0x07},
	{0x233d, 0xfb},
	{0x233e, 0x06},
	{0x233f, 0x72},
	{0x2340, 0x9c},
	{0x2341, 0xa0},
	{0x2342, 0x00},
	{0x2343, 0x01},
	{0x2344, 0x85},
	{0x2345, 0x21},
	{0x2346, 0x00},
	{0x2347, 0x00},
	{0x2348, 0x85},
	{0x2349, 0x41},
	{0x234a, 0x00},
	{0x234b, 0x04},
	{0x234c, 0x85},
	{0x234d, 0x81},
	{0x234e, 0x00},
	{0x234f, 0x08},
	{0x2350, 0x85},
	{0x2351, 0xc1},
	{0x2352, 0x00},
	{0x2353, 0x0c},
	{0x2354, 0x44},
	{0x2355, 0x00},
	{0x2356, 0x48},
	{0x2357, 0x00},
	{0x2358, 0x9c},
	{0x2359, 0x21},
	{0x235a, 0x00},
	{0x235b, 0x10},
	{0x235c, 0x9c},
	{0x235d, 0x21},
	{0x235e, 0xff},
	{0x235f, 0x58},
	{0x2360, 0xd4},
	{0x2361, 0x01},
	{0x2362, 0x48},
	{0x2363, 0x00},
	{0x2364, 0xd4},
	{0x2365, 0x01},
	{0x2366, 0x50},
	{0x2367, 0x04},
	{0x2368, 0xd4},
	{0x2369, 0x01},
	{0x236a, 0x60},
	{0x236b, 0x08},
	{0x236c, 0xd4},
	{0x236d, 0x01},
	{0x236e, 0x70},
	{0x236f, 0x0c},
	{0x2370, 0xd4},
	{0x2371, 0x01},
	{0x2372, 0x80},
	{0x2373, 0x10},
	{0x2374, 0xd4},
	{0x2375, 0x01},
	{0x2376, 0x90},
	{0x2377, 0x14},
	{0x2378, 0xd4},
	{0x2379, 0x01},
	{0x237a, 0xa0},
	{0x237b, 0x18},
	{0x237c, 0xd4},
	{0x237d, 0x01},
	{0x237e, 0xb0},
	{0x237f, 0x1c},
	{0x2380, 0x07},
	{0x2381, 0xfb},
	{0x2382, 0x07},
	{0x2383, 0x0e},
	{0x2384, 0x9e},
	{0x2385, 0x80},
	{0x2386, 0x00},
	{0x2387, 0x00},
	{0x2388, 0xa9},
	{0x2389, 0xcb},
	{0x238a, 0x00},
	{0x238b, 0x00},
	{0x238c, 0x85},
	{0x238d, 0x8b},
	{0x238e, 0x00},
	{0x238f, 0x08},
	{0x2390, 0x9c},
	{0x2391, 0x61},
	{0x2392, 0x00},
	{0x2393, 0x78},
	{0x2394, 0x18},
	{0x2395, 0x80},
	{0x2396, 0x00},
	{0x2397, 0x14},
	{0x2398, 0xa8},
	{0x2399, 0x84},
	{0x239a, 0x21},
	{0x239b, 0x14},
	{0x239c, 0x07},
	{0x239d, 0xfb},
	{0x239e, 0x1e},
	{0x239f, 0x1f},
	{0x23a0, 0x9c},
	{0x23a1, 0xa0},
	{0x23a2, 0x00},
	{0x23a3, 0x30},
	{0x23a4, 0x9c},
	{0x23a5, 0xa0},
	{0x23a6, 0x00},
	{0x23a7, 0x58},
	{0x23a8, 0x9c},
	{0x23a9, 0x61},
	{0x23aa, 0x00},
	{0x23ab, 0x20},
	{0x23ac, 0x18},
	{0x23ad, 0x80},
	{0x23ae, 0x00},
	{0x23af, 0x14},
	{0x23b0, 0xa8},
	{0x23b1, 0x84},
	{0x23b2, 0x21},
	{0x23b3, 0x44},
	{0x23b4, 0x07},
	{0x23b5, 0xfb},
	{0x23b6, 0x1e},
	{0x23b7, 0x19},
	{0x23b8, 0xaa},
	{0x23b9, 0xd4},
	{0x23ba, 0x00},
	{0x23bb, 0x00},
	{0x23bc, 0xa8},
	{0x23bd, 0x6e},
	{0x23be, 0x00},
	{0x23bf, 0x00},
	{0x23c0, 0x9c},
	{0x23c1, 0x80},
	{0x23c2, 0x00},
	{0x23c3, 0x1b},
	{0x23c4, 0x07},
	{0x23c5, 0xfb},
	{0x23c6, 0x06},
	{0x23c7, 0x80},
	{0x23c8, 0xaa},
	{0x23c9, 0x54},
	{0x23ca, 0x00},
	{0x23cb, 0x00},
	{0x23cc, 0xa5},
	{0x23cd, 0x6b},
	{0x23ce, 0x00},
	{0x23cf, 0x0f},
	{0x23d0, 0xa8},
	{0x23d1, 0x6e},
	{0x23d2, 0x00},
	{0x23d3, 0x00},
	{0x23d4, 0xb9},
	{0x23d5, 0x4b},
	{0x23d6, 0x00},
	{0x23d7, 0x08},
	{0x23d8, 0x07},
	{0x23d9, 0xfb},
	{0x23da, 0x06},
	{0x23db, 0x7b},
	{0x23dc, 0x9c},
	{0x23dd, 0x80},
	{0x23de, 0x00},
	{0x23df, 0x1c},
	{0x23e0, 0x84},
	{0x23e1, 0x6c},
	{0x23e2, 0x01},
	{0x23e3, 0x0c},
	{0x23e4, 0xe2},
	{0x23e5, 0x0a},
	{0x23e6, 0x58},
	{0x23e7, 0x04},
	{0x23e8, 0xb9},
	{0x23e9, 0x83},
	{0x23ea, 0x00},
	{0x23eb, 0x05},
	{0x23ec, 0x9c},
	{0x23ed, 0xa0},
	{0x23ee, 0x00},
	{0x23ef, 0x01},
	{0x23f0, 0x9c},
	{0x23f1, 0x80},
	{0x23f2, 0x00},
	{0x23f3, 0xef},
	{0x23f4, 0x07},
	{0x23f5, 0xfb},
	{0x23f6, 0x06},
	{0x23f7, 0x44},
	{0x23f8, 0xa8},
	{0x23f9, 0x6e},
	{0x23fa, 0x00},
	{0x23fb, 0x00},
	{0x23fc, 0xa8},
	{0x23fd, 0x6e},
	{0x23fe, 0x00},
	{0x23ff, 0x00},
	{0x2400, 0x07},
	{0x2401, 0xfb},
	{0x2402, 0x06},
	{0x2403, 0x71},
	{0x2404, 0x9c},
	{0x2405, 0x80},
	{0x2406, 0x00},
	{0x2407, 0x1b},
	{0x2408, 0x18},
	{0x2409, 0xa0},
	{0x240a, 0x00},
	{0x240b, 0x14},
	{0x240c, 0xa8},
	{0x240d, 0xa5},
	{0x240e, 0x5d},
	{0x240f, 0x08},
	{0x2410, 0xa5},
	{0x2411, 0x4b},
	{0x2412, 0xff},
	{0x2413, 0xff},
	{0x2414, 0x94},
	{0x2415, 0x65},
	{0x2416, 0x00},
	{0x2417, 0x00},
	{0x2418, 0xe4},
	{0x2419, 0x03},
	{0x241a, 0x50},
	{0x241b, 0x00},
	{0x241c, 0x10},
	{0x241d, 0x00},
	{0x241e, 0x00},
	{0x241f, 0x15},
	{0x2420, 0xb8},
	{0x2421, 0x6a},
	{0x2422, 0x00},
	{0x2423, 0x47},
	{0x2424, 0xa4},
	{0x2425, 0x8a},
	{0x2426, 0x00},
	{0x2427, 0xf0},
	{0x2428, 0xdc},
	{0x2429, 0x05},
	{0x242a, 0x50},
	{0x242b, 0x00},
	{0x242c, 0xa5},
	{0x242d, 0x63},
	{0x242e, 0x00},
	{0x242f, 0x01},
	{0x2430, 0xb8},
	{0x2431, 0x84},
	{0x2432, 0x00},
	{0x2433, 0x08},
	{0x2434, 0xe1},
	{0x2435, 0x70},
	{0x2436, 0x58},
	{0x2437, 0x02},
	{0x2438, 0xa8},
	{0x2439, 0x6e},
	{0x243a, 0x00},
	{0x243b, 0x00},
	{0x243c, 0xe0},
	{0x243d, 0xab},
	{0x243e, 0x20},
	{0x243f, 0x04},
	{0x2440, 0xa5},
	{0x2441, 0x45},
	{0x2442, 0xff},
	{0x2443, 0xff},
	{0x2444, 0x9c},
	{0x2445, 0x80},
	{0x2446, 0x00},
	{0x2447, 0x1b},
	{0x2448, 0x07},
	{0x2449, 0xfb},
	{0x244a, 0x06},
	{0x244b, 0x2f},
	{0x244c, 0xb8},
	{0x244d, 0xaa},
	{0x244e, 0x00},
	{0x244f, 0x48},
	{0x2450, 0xa8},
	{0x2451, 0x6e},
	{0x2452, 0x00},
	{0x2453, 0x00},
	{0x2454, 0xa4},
	{0x2455, 0xaa},
	{0x2456, 0x00},
	{0x2457, 0xff},
	{0x2458, 0x07},
	{0x2459, 0xfb},
	{0x245a, 0x06},
	{0x245b, 0x2b},
	{0x245c, 0x9c},
	{0x245d, 0x80},
	{0x245e, 0x00},
	{0x245f, 0x1c},
	{0x2460, 0xa8},
	{0x2461, 0x6e},
	{0x2462, 0x00},
	{0x2463, 0x00},
	{0x2464, 0x9c},
	{0x2465, 0x80},
	{0x2466, 0x00},
	{0x2467, 0x09},
	{0x2468, 0x07},
	{0x2469, 0xfb},
	{0x246a, 0x06},
	{0x246b, 0x27},
	{0x246c, 0x9c},
	{0x246d, 0xa0},
	{0x246e, 0x00},
	{0x246f, 0x01},
	{0x2470, 0x18},
	{0x2471, 0xe0},
	{0x2472, 0x00},
	{0x2473, 0x14},
	{0x2474, 0xa8},
	{0x2475, 0xe7},
	{0x2476, 0x5d},
	{0x2477, 0x04},
	{0x2478, 0x84},
	{0x2479, 0x67},
	{0x247a, 0x00},
	{0x247b, 0x00},
	{0x247c, 0xe4},
	{0x247d, 0x03},
	{0x247e, 0x60},
	{0x247f, 0x00},
	{0x2480, 0x10},
	{0x2481, 0x00},
	{0x2482, 0x00},
	{0x2483, 0x3e},
	{0x2484, 0x84},
	{0x2485, 0x61},
	{0x2486, 0x00},
	{0x2487, 0x24},
	{0x2488, 0x18},
	{0x2489, 0xa0},
	{0x248a, 0x00},
	{0x248b, 0x14},
	{0x248c, 0xa8},
	{0x248d, 0xa5},
	{0x248e, 0x5d},
	{0x248f, 0x02},
	{0x2490, 0xd8},
	{0x2491, 0x05},
	{0x2492, 0xa0},
	{0x2493, 0x00},
	{0x2494, 0xa8},
	{0x2495, 0xc5},
	{0x2496, 0x00},
	{0x2497, 0x00},
	{0x2498, 0x8c},
	{0x2499, 0x66},
	{0x249a, 0x00},
	{0x249b, 0x00},
	{0x249c, 0x9d},
	{0x249d, 0x01},
	{0x249e, 0x00},
	{0x249f, 0xa8},
	{0x24a0, 0x9c},
	{0x24a1, 0x83},
	{0x24a2, 0x00},
	{0x24a3, 0x01},
	{0x24a4, 0xb8},
	{0x24a5, 0x63},
	{0x24a6, 0x00},
	{0x24a7, 0x03},
	{0x24a8, 0xe0},
	{0x24a9, 0x63},
	{0x24aa, 0x40},
	{0x24ab, 0x00},
	{0x24ac, 0x84},
	{0x24ad, 0x63},
	{0x24ae, 0xff},
	{0x24af, 0x78},
	{0x24b0, 0xe4},
	{0x24b1, 0x83},
	{0x24b2, 0x60},
	{0x24b3, 0x00},
	{0x24b4, 0x0c},
	{0x24b5, 0x00},
	{0x24b6, 0x00},
	{0x24b7, 0x07},
	{0x24b8, 0x15},
	{0x24b9, 0x00},
	{0x24ba, 0x00},
	{0x24bb, 0x00},
	{0x24bc, 0xd8},
	{0x24bd, 0x06},
	{0x24be, 0x20},
	{0x24bf, 0x00},
	{0x24c0, 0x8c},
	{0x24c1, 0x65},
	{0x24c2, 0x00},
	{0x24c3, 0x00},
	{0x24c4, 0xbc},
	{0x24c5, 0x63},
	{0x24c6, 0x00},
	{0x24c7, 0x0b},
	{0x24c8, 0x0f},
	{0x24c9, 0xff},
	{0x24ca, 0xff},
	{0x24cb, 0xf4},
	{0x24cc, 0x15},
	{0x24cd, 0x00},
	{0x24ce, 0x00},
	{0x24cf, 0x00},
	{0x24d0, 0xd4},
	{0x24d1, 0x07},
	{0x24d2, 0x60},
	{0x24d3, 0x00},
	{0x24d4, 0x18},
	{0x24d5, 0xe0},
	{0x24d6, 0x00},
	{0x24d7, 0x14},
	{0x24d8, 0xa8},
	{0x24d9, 0xe7},
	{0x24da, 0x5d},
	{0x24db, 0x03},
	{0x24dc, 0x8c},
	{0x24dd, 0x67},
	{0x24de, 0x00},
	{0x24df, 0x00},
	{0x24e0, 0xe4},
	{0x24e1, 0x03},
	{0x24e2, 0x90},
	{0x24e3, 0x00},
	{0x24e4, 0x10},
	{0x24e5, 0x00},
	{0x24e6, 0x00},
	{0x24e7, 0x11},
	{0x24e8, 0xa8},
	{0x24e9, 0x6e},
	{0x24ea, 0x00},
	{0x24eb, 0x00},
	{0x24ec, 0x9c},
	{0x24ed, 0x80},
	{0x24ee, 0x00},
	{0x24ef, 0xef},
	{0x24f0, 0x9c},
	{0x24f1, 0xa0},
	{0x24f2, 0x00},
	{0x24f3, 0x06},
	{0x24f4, 0x07},
	{0x24f5, 0xfb},
	{0x24f6, 0x06},
	{0x24f7, 0x04},
	{0x24f8, 0xd8},
	{0x24f9, 0x07},
	{0x24fa, 0x90},
	{0x24fb, 0x00},
	{0x24fc, 0xa8},
	{0x24fd, 0x6e},
	{0x24fe, 0x00},
	{0x24ff, 0x00},
	{0x2500, 0xa8},
	{0x2501, 0xb2},
	{0x2502, 0x00},
	{0x2503, 0x00},
	{0x2504, 0x9c},
	{0x2505, 0x80},
	{0x2506, 0x00},
	{0x2507, 0x99},
	{0x2508, 0x07},
	{0x2509, 0xfb},
	{0x250a, 0x05},
	{0x250b, 0xff},
	{0x250c, 0x15},
	{0x250d, 0x00},
	{0x250e, 0x00},
	{0x250f, 0x00},
	{0x2510, 0xa8},
	{0x2511, 0x6e},
	{0x2512, 0x00},
	{0x2513, 0x00},
	{0x2514, 0x9c},
	{0x2515, 0x80},
	{0x2516, 0x00},
	{0x2517, 0xf1},
	{0x2518, 0x07},
	{0x2519, 0xfb},
	{0x251a, 0x05},
	{0x251b, 0xfb},
	{0x251c, 0x9c},
	{0x251d, 0xa0},
	{0x251e, 0x00},
	{0x251f, 0x01},
	{0x2520, 0x00},
	{0x2521, 0x00},
	{0x2522, 0x00},
	{0x2523, 0x28},
	{0x2524, 0x15},
	{0x2525, 0x00},
	{0x2526, 0x00},
	{0x2527, 0x00},
	{0x2528, 0x18},
	{0x2529, 0x60},
	{0x252a, 0x00},
	{0x252b, 0x14},
	{0x252c, 0xa8},
	{0x252d, 0x63},
	{0x252e, 0x5d},
	{0x252f, 0x02},
	{0x2530, 0x8c},
	{0x2531, 0x63},
	{0x2532, 0x00},
	{0x2533, 0x00},
	{0x2534, 0xbc},
	{0x2535, 0xa3},
	{0x2536, 0x00},
	{0x2537, 0x00},
	{0x2538, 0x10},
	{0x2539, 0x00},
	{0x253a, 0x00},
	{0x253b, 0x04},
	{0x253c, 0xe4},
	{0x253d, 0x16},
	{0x253e, 0xa0},
	{0x253f, 0x00},
	{0x2540, 0x9e},
	{0x2541, 0x80},
	{0x2542, 0x00},
	{0x2543, 0x01},
	{0x2544, 0xe4},
	{0x2545, 0x16},
	{0x2546, 0xa0},
	{0x2547, 0x00},
	{0x2548, 0x10},
	{0x2549, 0x00},
	{0x254a, 0x00},
	{0x254b, 0x1e},
	{0x254c, 0xa8},
	{0x254d, 0x6e},
	{0x254e, 0x00},
	{0x254f, 0x00},
	{0x2550, 0x9c},
	{0x2551, 0x80},
	{0x2552, 0x00},
	{0x2553, 0xef},
	{0x2554, 0x07},
	{0x2555, 0xfb},
	{0x2556, 0x05},
	{0x2557, 0xec},
	{0x2558, 0x9c},
	{0x2559, 0xa0},
	{0x255a, 0x00},
	{0x255b, 0x06},
	{0x255c, 0xb8},
	{0x255d, 0xb4},
	{0x255e, 0x00},
	{0x255f, 0x03},
	{0x2560, 0x9c},
	{0x2561, 0xc1},
	{0x2562, 0x00},
	{0x2563, 0xa8},
	{0x2564, 0xa8},
	{0x2565, 0x6e},
	{0x2566, 0x00},
	{0x2567, 0x00},
	{0x2568, 0xe0},
	{0x2569, 0xa5},
	{0x256a, 0x30},
	{0x256b, 0x00},
	{0x256c, 0x9c},
	{0x256d, 0x80},
	{0x256e, 0x00},
	{0x256f, 0x9a},
	{0x2570, 0x03},
	{0x2571, 0xff},
	{0x2572, 0xff},
	{0x2573, 0xe6},
	{0x2574, 0x84},
	{0x2575, 0xa5},
	{0x2576, 0xff},
	{0x2577, 0x7c},
	{0x2578, 0x18},
	{0x2579, 0xe0},
	{0x257a, 0x00},
	{0x257b, 0x14},
	{0x257c, 0xa8},
	{0x257d, 0xe7},
	{0x257e, 0x5d},
	{0x257f, 0x03},
	{0x2580, 0xa4},
	{0x2581, 0x83},
	{0x2582, 0x00},
	{0x2583, 0x0f},
	{0x2584, 0x8c},
	{0x2585, 0xc7},
	{0x2586, 0x00},
	{0x2587, 0x00},
	{0x2588, 0xb8},
	{0x2589, 0x64},
	{0x258a, 0x00},
	{0x258b, 0x01},
	{0x258c, 0xe0},
	{0x258d, 0x63},
	{0x258e, 0x20},
	{0x258f, 0x00},
	{0x2590, 0x9c},
	{0x2591, 0x81},
	{0x2592, 0x00},
	{0x2593, 0xa8},
	{0x2594, 0xe0},
	{0x2595, 0xa3},
	{0x2596, 0x20},
	{0x2597, 0x00},
	{0x2598, 0x8e},
	{0x2599, 0x45},
	{0x259a, 0xff},
	{0x259b, 0xd1},
	{0x259c, 0xe4},
	{0x259d, 0xa6},
	{0x259e, 0x90},
	{0x259f, 0x00},
	{0x25a0, 0x0c},
	{0x25a1, 0x00},
	{0x25a2, 0x00},
	{0x25a3, 0x06},
	{0x25a4, 0xe4},
	{0x25a5, 0x66},
	{0x25a6, 0x90},
	{0x25a7, 0x00},
	{0x25a8, 0x13},
	{0x25a9, 0xff},
	{0x25aa, 0xff},
	{0x25ab, 0xcd},
	{0x25ac, 0x15},
	{0x25ad, 0x00},
	{0x25ae, 0x00},
	{0x25af, 0x00},
	{0x25b0, 0x03},
	{0x25b1, 0xff},
	{0x25b2, 0xff},
	{0x25b3, 0xcb},
	{0x25b4, 0x9e},
	{0x25b5, 0x52},
	{0x25b6, 0x00},
	{0x25b7, 0x01},
	{0x25b8, 0x03},
	{0x25b9, 0xff},
	{0x25ba, 0xff},
	{0x25bb, 0xc9},
	{0x25bc, 0x9e},
	{0x25bd, 0x52},
	{0x25be, 0xff},
	{0x25bf, 0xff},
	{0x25c0, 0x85},
	{0x25c1, 0x21},
	{0x25c2, 0x00},
	{0x25c3, 0x00},
	{0x25c4, 0x85},
	{0x25c5, 0x41},
	{0x25c6, 0x00},
	{0x25c7, 0x04},
	{0x25c8, 0x85},
	{0x25c9, 0x81},
	{0x25ca, 0x00},
	{0x25cb, 0x08},
	{0x25cc, 0x85},
	{0x25cd, 0xc1},
	{0x25ce, 0x00},
	{0x25cf, 0x0c},
	{0x25d0, 0x86},
	{0x25d1, 0x01},
	{0x25d2, 0x00},
	{0x25d3, 0x10},
	{0x25d4, 0x86},
	{0x25d5, 0x41},
	{0x25d6, 0x00},
	{0x25d7, 0x14},
	{0x25d8, 0x86},
	{0x25d9, 0x81},
	{0x25da, 0x00},
	{0x25db, 0x18},
	{0x25dc, 0x86},
	{0x25dd, 0xc1},
	{0x25de, 0x00},
	{0x25df, 0x1c},
	{0x25e0, 0x44},
	{0x25e1, 0x00},
	{0x25e2, 0x48},
	{0x25e3, 0x00},
	{0x25e4, 0x9c},
	{0x25e5, 0x21},
	{0x25e6, 0x00},
	{0x25e7, 0xa8},
	{0x25e8, 0x9c},
	{0x25e9, 0x21},
	{0x25ea, 0xff},
	{0x25eb, 0xf8},
	{0x25ec, 0xd4},
	{0x25ed, 0x01},
	{0x25ee, 0x48},
	{0x25ef, 0x00},
	{0x25f0, 0xd4},
	{0x25f1, 0x01},
	{0x25f2, 0x50},
	{0x25f3, 0x04},
	{0x25f4, 0x18},
	{0x25f5, 0x80},
	{0x25f6, 0x00},
	{0x25f7, 0x14},
	{0x25f8, 0xa8},
	{0x25f9, 0x84},
	{0x25fa, 0x20},
	{0x25fb, 0x1c},
	{0x25fc, 0x19},
	{0x25fd, 0x40},
	{0x25fe, 0x00},
	{0x25ff, 0x14},
	{0x2600, 0xa9},
	{0x2601, 0x4a},
	{0x2602, 0x5d},
	{0x2603, 0x0c},
	{0x2604, 0x84},
	{0x2605, 0x84},
	{0x2606, 0x00},
	{0x2607, 0x00},
	{0x2608, 0xbc},
	{0x2609, 0x03},
	{0x260a, 0x04},
	{0x260b, 0x0b},
	{0x260c, 0x84},
	{0x260d, 0x84},
	{0x260e, 0x00},
	{0x260f, 0x14},
	{0x2610, 0x10},
	{0x2611, 0x00},
	{0x2612, 0x00},
	{0x2613, 0x07},
	{0x2614, 0xd4},
	{0x2615, 0x0a},
	{0x2616, 0x20},
	{0x2617, 0x00},
	{0x2618, 0xbc},
	{0x2619, 0x03},
	{0x261a, 0x04},
	{0x261b, 0x0c},
	{0x261c, 0x0c},
	{0x261d, 0x00},
	{0x261e, 0x00},
	{0x261f, 0x1a},
	{0x2620, 0x15},
	{0x2621, 0x00},
	{0x2622, 0x00},
	{0x2623, 0x00},
	{0x2624, 0x00},
	{0x2625, 0x00},
	{0x2626, 0x00},
	{0x2627, 0x06},
	{0x2628, 0x15},
	{0x2629, 0x00},
	{0x262a, 0x00},
	{0x262b, 0x00},
	{0x262c, 0x07},
	{0x262d, 0xff},
	{0x262e, 0xff},
	{0x262f, 0x1b},
	{0x2630, 0x15},
	{0x2631, 0x00},
	{0x2632, 0x00},
	{0x2633, 0x00},
	{0x2634, 0x00},
	{0x2635, 0x00},
	{0x2636, 0x00},
	{0x2637, 0x14},
	{0x2638, 0x15},
	{0x2639, 0x00},
	{0x263a, 0x00},
	{0x263b, 0x00},
	{0x263c, 0x07},
	{0x263d, 0xff},
	{0x263e, 0xfe},
	{0x263f, 0xd8},
	{0x2640, 0x15},
	{0x2641, 0x00},
	{0x2642, 0x00},
	{0x2643, 0x00},
	{0x2644, 0x18},
	{0x2645, 0x60},
	{0x2646, 0x00},
	{0x2647, 0x14},
	{0x2648, 0xa8},
	{0x2649, 0x63},
	{0x264a, 0x5d},
	{0x264b, 0x14},
	{0x264c, 0x8c},
	{0x264d, 0x63},
	{0x264e, 0x00},
	{0x264f, 0x00},
	{0x2650, 0xbc},
	{0x2651, 0x23},
	{0x2652, 0x00},
	{0x2653, 0x01},
	{0x2654, 0x10},
	{0x2655, 0x00},
	{0x2656, 0x00},
	{0x2657, 0x0c},
	{0x2658, 0x15},
	{0x2659, 0x00},
	{0x265a, 0x00},
	{0x265b, 0x00},
	{0x265c, 0x18},
	{0x265d, 0x80},
	{0x265e, 0x00},
	{0x265f, 0x14},
	{0x2660, 0xa8},
	{0x2661, 0x84},
	{0x2662, 0x5d},
	{0x2663, 0x10},
	{0x2664, 0x84},
	{0x2665, 0x6a},
	{0x2666, 0x00},
	{0x2667, 0x00},
	{0x2668, 0x07},
	{0x2669, 0xfb},
	{0x266a, 0x77},
	{0x266b, 0x5b},
	{0x266c, 0x84},
	{0x266d, 0x84},
	{0x266e, 0x00},
	{0x266f, 0x00},
	{0x2670, 0xbc},
	{0x2671, 0x2b},
	{0x2672, 0x00},
	{0x2673, 0x00},
	{0x2674, 0x10},
	{0x2675, 0x00},
	{0x2676, 0x00},
	{0x2677, 0x04},
	{0x2678, 0x15},
	{0x2679, 0x00},
	{0x267a, 0x00},
	{0x267b, 0x00},
	{0x267c, 0x07},
	{0x267d, 0xff},
	{0x267e, 0xff},
	{0x267f, 0x38},
	{0x2680, 0x15},
	{0x2681, 0x00},
	{0x2682, 0x00},
	{0x2683, 0x00},
	{0x2684, 0x85},
	{0x2685, 0x21},
	{0x2686, 0x00},
	{0x2687, 0x00},
	{0x2688, 0x85},
	{0x2689, 0x41},
	{0x268a, 0x00},
	{0x268b, 0x04},
	{0x268c, 0x44},
	{0x268d, 0x00},
	{0x268e, 0x48},
	{0x268f, 0x00},
	{0x2690, 0x9c},
	{0x2691, 0x21},
	{0x2692, 0x00},
	{0x2693, 0x08},
	{0x2694, 0x9c},
	{0x2695, 0x21},
	{0x2696, 0xff},
	{0x2697, 0xfc},
	{0x2698, 0xd4},
	{0x2699, 0x01},
	{0x269a, 0x48},
	{0x269b, 0x00},
	{0x269c, 0x07},
	{0x269d, 0xff},
	{0x269e, 0xff},
	{0x269f, 0xd3},
	{0x26a0, 0x15},
	{0x26a1, 0x00},
	{0x26a2, 0x00},
	{0x26a3, 0x00},
	{0x26a4, 0x9d},
	{0x26a5, 0x60},
	{0x26a6, 0x00},
	{0x26a7, 0x00},
	{0x26a8, 0x85},
	{0x26a9, 0x21},
	{0x26aa, 0x00},
	{0x26ab, 0x00},
	{0x26ac, 0x44},
	{0x26ad, 0x00},
	{0x26ae, 0x48},
	{0x26af, 0x00},
	{0x26b0, 0x9c},
	{0x26b1, 0x21},
	{0x26b2, 0x00},
	{0x26b3, 0x04},
	{0x26b4, 0x00},
	{0x26b5, 0x00},
	{0x26b6, 0x00},
	{0x26b7, 0x00},
	{0x26b8, 0x00},
	{0x26b9, 0x00},
	{0x26ba, 0x00},
	{0x26bb, 0x00},
	{0x26bc, 0x00},
	{0x26bd, 0x00},
	{0x26be, 0x00},
	{0x26bf, 0x00},
	{0x26c0, 0x00},
	{0x26c1, 0x00},
	{0x26c2, 0x00},
	{0x26c3, 0x00},
	{0x26c4, 0x00},
	{0x26c5, 0x00},
	{0x26c6, 0x00},
	{0x26c7, 0x00},
	{0x26c8, 0x00},
	{0x26c9, 0x00},
	{0x26ca, 0x00},
	{0x26cb, 0x00},
	{0x5d00, 0x07},
	{0x5d01, 0xbb},
	{0x5d02, 0x00},//u8ltmtarget
	{0x5d03, 0x00},//u32pretb1ltm_work
	{0x5d04, 0x00},
	{0x5d05, 0x00},
	{0x5d06, 0x00},
	{0x5d07, 0x01},//u32again
	{0x5d08, 0x00},//su16hsize
	{0x5d10, 0x00},
	{0x5d11, 0x00},
	{0x5d12, 0x00},
	{0x5d13, 0x02},//FC
	{0x5d14, 0x00},
	{0xfffe, 0x50},
	{0x0137, 0x99},
	{0xfffe, 0x14},
	{0x002b, 0x01},
	{0xfffe, 0x30},
	{0x1f00, 0x00},// WIN start X  100
	{0x1f01, 0x00},
	{0x1f02, 0x00},// WIN stat Y
	{0x1f03, 0x00},
	{0x1f04, 0x0a},// WIN width
	{0x1f05, 0x20},
	{0x1f06, 0x05},// WIN height
	{0x1f07, 0xa0},
	{0x1f08, 0x03},
	{0x0051, 0x01},//gamma
	{0xfffe, 0x25},
	{0x0002, 0xf0},
	{0xfffe, 0x14},
	{0x0022, 0x1E},//fps
	{0xfffe, 0x14},
	{0x000e, 0x00},
	{0x010e, 0x90},
	{0x010f, 0x00},
	{0x0110, 0x05},
	{0x0111, 0x05},
	{0x0114, 0x00},
	{0x0115, 0x0c},
	{0x0116, 0x00},
	{0x0117, 0x0d},
	{0x011c, 0x00},
	{0x011d, 0xff},
	{0x011e, 0x00},
	{0x011f, 0xff},
	{0x0134, 0x00},
	{0x0135, 0x83},
	{0x0136, 0x00},
	{0x0137, 0x00},
	{0x013c, 0x00},
	{0x013d, 0xff},
	{0x013e, 0x00},
	{0x013f, 0x00},
	{0x004c, 0x00},
	{0x002b, 0x01},
	{0x004d, 0x01},
	{0x00fa, 0x02},
	{0x00fb, 0x00},
	{0x00fc, 0x00},
	{0x00fd, 0x30},
	{0x00e2, 0x70},
	{0x00e3, 0x00},
	{0x00de, 0x00},
	{0x00df, 0x10},
	{0x00a0, 0x01},
	{0x00a1, 0x40},
	{0x0104, 0x00},
	{0x0105, 0x00},
	{0x0106, 0x11},
	{0x0107, 0x90},
	{0x0108, 0x15},
	{0x0109, 0x10},
	{0x0145, 0x00},
	{0x0055, 0x10},
	{0x0056, 0x10},
	{0x0057, 0x10},
	{0x0058, 0x10},
	{0x0059, 0x10},
	{0x005a, 0x10},
	{0x005b, 0x40},
	{0x005c, 0x40},
	{0x005d, 0x40},
	{0x005e, 0x10},
	{0x005f, 0x10},
	{0x0060, 0x40},
	{0x0061, 0x80},
	{0x0062, 0x40},
	{0x0063, 0x10},
	{0x0064, 0x10},
	{0x0065, 0x40},
	{0x0066, 0x40},
	{0x0067, 0x40},
	{0x0068, 0x10},
	{0x0069, 0x10},
	{0x006a, 0x10},
	{0x006b, 0x10},
	{0x006c, 0x10},
	{0x006d, 0x10},
	{0x0088, 0x00},
	{0x0089, 0x07},
	{0x008a, 0x39},
	{0x008b, 0xc0},
	{0x0050, 0x01},
	{0x00c6, 0x01},
	{0x01bc, 0x00},
	{0x01bd, 0x60},
	{0x01be, 0x00},
	{0x01bf, 0x60},
	{0x00ca, 0x00},
	{0x00cb, 0x60},
	{0x00cc, 0x00},
	{0x00cd, 0x80},
	{0x00c7, 0x18},
	{0x00d8, 0x40},
	{0x0212, 0x00},
	{0x0215, 0x50},
	{0x0217, 0x50},
	{0x0219, 0x50},
	{0x021b, 0xff},
	{0x01b6, 0x50},
	{0x00c8, 0x01},
	{0x0208, 0x02},
	{0x020b, 0x40},
	{0x020d, 0x04},
	{0x00da, 0x00},
	{0x00db, 0x00},
	{0x00dc, 0x03},
	{0x00dd, 0x40},
	{0x00ce, 0x00},
	{0x00cf, 0x40},
	{0x00d0, 0x00},
	{0x00d1, 0x80},
	{0x00d4, 0x00},
	{0x00d5, 0x80},
	{0x00d6, 0x00},
	{0x00d7, 0x80},
	{0x00d9, 0x00},
	{0xfffe, 0x50},
	{0x004d, 0x00},
	{0xfffd, 0x80},
	{0xfffe, 0x30},
	{0x0013, 0x00},
	{0x071b, 0x00},
	{0xfffe, 0x30},
	{0x0000, 0x01},
	{0x0002, 0x96},
	{0xfffe, 0x30},
	{0x03ca, 0x09},
	{0x03cb, 0x7B},
	{0x03cc, 0x10},
	{0x03cd, 0xDB},
	{0x03ce, 0x09},
	{0x03cf, 0x7B},
	{0x03d0, 0x08},
	{0x03d1, 0x6D},
	{0xfffe, 0x14},
	{0x002c, 0x01},
	{0x0030, 0x01},
	{0x0620, 0x01},
	{0x0621, 0x01},
	{0xfffe, 0x14},
	{0x0928, 0x00},
	{0x0929, 0x61},
	{0x092a, 0x00},
	{0x092b, 0xb8},
	{0x092c, 0x01},
	{0x092d, 0x60},
	{0x06e5, 0x1c},
	{0x06e6, 0x12},
	{0x06e7, 0x0f},
	{0x06e8, 0x0d},
	{0x06e9, 0x0e},
	{0x06ea, 0x10},
	{0x06eb, 0x1a},
	{0x06ec, 0x0c},
	{0x06ed, 0x06},
	{0x06ee, 0x03},
	{0x06ef, 0x02},
	{0x06f0, 0x02},
	{0x06f1, 0x02},
	{0x06f2, 0x03},
	{0x06f3, 0x04},
	{0x06f4, 0x0d},
	{0x06f5, 0x02},
	{0x06f6, 0x01},
	{0x06f7, 0x01},
	{0x06f8, 0x01},
	{0x06f9, 0x01},
	{0x06fa, 0x01},
	{0x06fb, 0x01},
	{0x06fc, 0x01},
	{0x06fd, 0x01},
	{0x06fe, 0x00},
	{0x06ff, 0x00},
	{0x0700, 0x00},
	{0x0701, 0x00},
	{0x0702, 0x00},
	{0x0703, 0x01},
	{0x0704, 0x02},
	{0x0705, 0x01},
	{0x0706, 0x00},
	{0x0707, 0x01},
	{0x0708, 0x00},
	{0x0709, 0x00},
	{0x070a, 0x00},
	{0x070b, 0x00},
	{0x070c, 0x01},
	{0x070d, 0x02},
	{0x070e, 0x01},
	{0x070f, 0x01},
	{0x0710, 0x00},
	{0x0711, 0x00},
	{0x0712, 0x00},
	{0x0713, 0x01},
	{0x0714, 0x03},
	{0x0715, 0x06},
	{0x0716, 0x03},
	{0x0717, 0x02},
	{0x0718, 0x02},
	{0x0719, 0x02},
	{0x071a, 0x02},
	{0x071b, 0x04},
	{0x071c, 0x07},
	{0x071d, 0x19},
	{0x071e, 0x10},
	{0x071f, 0x0c},
	{0x0720, 0x0a},
	{0x0721, 0x0b},
	{0x0722, 0x0d},
	{0x0723, 0x14},
	{0x0724, 0x23},
	{0x0725, 0x16},
	{0x0726, 0x18},
	{0x0727, 0x19},
	{0x0728, 0x19},
	{0x0729, 0x19},
	{0x072a, 0x19},
	{0x072b, 0x17},
	{0x072c, 0x1a},
	{0x072d, 0x1c},
	{0x072e, 0x1d},
	{0x072f, 0x1e},
	{0x0730, 0x1e},
	{0x0731, 0x1e},
	{0x0732, 0x1e},
	{0x0733, 0x1d},
	{0x0734, 0x1b},
	{0x0735, 0x1d},
	{0x0736, 0x1e},
	{0x0737, 0x1f},
	{0x0738, 0x1f},
	{0x0739, 0x20},
	{0x073a, 0x1f},
	{0x073b, 0x1f},
	{0x073c, 0x1e},
	{0x073d, 0x1e},
	{0x073e, 0x1f},
	{0x073f, 0x20},
	{0x0740, 0x20},
	{0x0741, 0x20},
	{0x0742, 0x20},
	{0x0743, 0x20},
	{0x0744, 0x1f},
	{0x0745, 0x1e},
	{0x0746, 0x1f},
	{0x0747, 0x20},
	{0x0748, 0x21},
	{0x0749, 0x21},
	{0x074a, 0x20},
	{0x074b, 0x20},
	{0x074c, 0x1f},
	{0x074d, 0x1e},
	{0x074e, 0x20},
	{0x074f, 0x20},
	{0x0750, 0x20},
	{0x0751, 0x21},
	{0x0752, 0x20},
	{0x0753, 0x1f},
	{0x0754, 0x1f},
	{0x0755, 0x1e},
	{0x0756, 0x1e},
	{0x0757, 0x20},
	{0x0758, 0x20},
	{0x0759, 0x1f},
	{0x075a, 0x1e},
	{0x075b, 0x1d},
	{0x075c, 0x1d},
	{0x075d, 0x17},
	{0x075e, 0x1c},
	{0x075f, 0x1b},
	{0x0760, 0x1b},
	{0x0761, 0x1c},
	{0x0762, 0x1c},
	{0x0763, 0x1a},
	{0x0764, 0x15},
	{0x0765, 0x21},
	{0x0766, 0x21},
	{0x0767, 0x21},
	{0x0768, 0x22},
	{0x0769, 0x22},
	{0x076a, 0x21},
	{0x076b, 0x21},
	{0x076c, 0x24},
	{0x076d, 0x21},
	{0x076e, 0x21},
	{0x076f, 0x21},
	{0x0770, 0x21},
	{0x0771, 0x21},
	{0x0772, 0x21},
	{0x0773, 0x22},
	{0x0774, 0x21},
	{0x0775, 0x20},
	{0x0776, 0x21},
	{0x0777, 0x21},
	{0x0778, 0x21},
	{0x0779, 0x21},
	{0x077a, 0x21},
	{0x077b, 0x21},
	{0x077c, 0x21},
	{0x077d, 0x20},
	{0x077e, 0x20},
	{0x077f, 0x20},
	{0x0780, 0x20},
	{0x0781, 0x20},
	{0x0782, 0x20},
	{0x0783, 0x20},
	{0x0784, 0x21},
	{0x0785, 0x20},
	{0x0786, 0x20},
	{0x0787, 0x20},
	{0x0788, 0x20},
	{0x0789, 0x20},
	{0x078a, 0x20},
	{0x078b, 0x20},
	{0x078c, 0x21},
	{0x078d, 0x20},
	{0x078e, 0x21},
	{0x078f, 0x21},
	{0x0790, 0x21},
	{0x0791, 0x21},
	{0x0792, 0x21},
	{0x0793, 0x21},
	{0x0794, 0x22},
	{0x0795, 0x22},
	{0x0796, 0x22},
	{0x0797, 0x22},
	{0x0798, 0x22},
	{0x0799, 0x22},
	{0x079a, 0x22},
	{0x079b, 0x22},
	{0x079c, 0x22},
	{0x079d, 0x21},
	{0x079e, 0x23},
	{0x079f, 0x23},
	{0x07a0, 0x24},
	{0x07a1, 0x23},
	{0x07a2, 0x23},
	{0x07a3, 0x23},
	{0x07a4, 0x23},
	{0x07a5, 0x1c},
	{0x07a6, 0x13},
	{0x07a7, 0x0f},
	{0x07a8, 0x0d},
	{0x07a9, 0x0e},
	{0x07aa, 0x10},
	{0x07ab, 0x19},
	{0x07ac, 0x18},
	{0x07ad, 0x06},
	{0x07ae, 0x03},
	{0x07af, 0x02},
	{0x07b0, 0x02},
	{0x07b1, 0x02},
	{0x07b2, 0x03},
	{0x07b3, 0x04},
	{0x07b4, 0x0b},
	{0x07b5, 0x01},
	{0x07b6, 0x01},
	{0x07b7, 0x01},
	{0x07b8, 0x01},
	{0x07b9, 0x01},
	{0x07ba, 0x01},
	{0x07bb, 0x01},
	{0x07bc, 0x01},
	{0x07bd, 0x01},
	{0x07be, 0x00},
	{0x07bf, 0x00},
	{0x07c0, 0x00},
	{0x07c1, 0x00},
	{0x07c2, 0x00},
	{0x07c3, 0x01},
	{0x07c4, 0x01},
	{0x07c5, 0x01},
	{0x07c6, 0x00},
	{0x07c7, 0x01},
	{0x07c8, 0x00},
	{0x07c9, 0x00},
	{0x07ca, 0x00},
	{0x07cb, 0x00},
	{0x07cc, 0x01},
	{0x07cd, 0x01},
	{0x07ce, 0x01},
	{0x07cf, 0x00},
	{0x07d0, 0x00},
	{0x07d1, 0x00},
	{0x07d2, 0x00},
	{0x07d3, 0x01},
	{0x07d4, 0x02},
	{0x07d5, 0x05},
	{0x07d6, 0x03},
	{0x07d7, 0x02},
	{0x07d8, 0x02},
	{0x07d9, 0x01},
	{0x07da, 0x02},
	{0x07db, 0x04},
	{0x07dc, 0x07},
	{0x07dd, 0x18},
	{0x07de, 0x0f},
	{0x07df, 0x0b},
	{0x07e0, 0x09},
	{0x07e1, 0x0a},
	{0x07e2, 0x0d},
	{0x07e3, 0x13},
	{0x07e4, 0x21},
	{0x07e5, 0x18},
	{0x07e6, 0x18},
	{0x07e7, 0x18},
	{0x07e8, 0x1a},
	{0x07e9, 0x18},
	{0x07ea, 0x19},
	{0x07eb, 0x18},
	{0x07ec, 0x18},
	{0x07ed, 0x1b},
	{0x07ee, 0x1d},
	{0x07ef, 0x1d},
	{0x07f0, 0x1e},
	{0x07f1, 0x1e},
	{0x07f2, 0x1d},
	{0x07f3, 0x1c},
	{0x07f4, 0x1b},
	{0x07f5, 0x1d},
	{0x07f6, 0x1e},
	{0x07f7, 0x1f},
	{0x07f8, 0x1f},
	{0x07f9, 0x20},
	{0x07fa, 0x1f},
	{0x07fb, 0x1e},
	{0x07fc, 0x1d},
	{0x07fd, 0x1e},
	{0x07fe, 0x1f},
	{0x07ff, 0x20},
	{0x0800, 0x21},
	{0x0801, 0x20},
	{0x0802, 0x20},
	{0x0803, 0x20},
	{0x0804, 0x1f},
	{0x0805, 0x1e},
	{0x0806, 0x20},
	{0x0807, 0x21},
	{0x0808, 0x21},
	{0x0809, 0x21},
	{0x080a, 0x20},
	{0x080b, 0x20},
	{0x080c, 0x1e},
	{0x080d, 0x1e},
	{0x080e, 0x1f},
	{0x080f, 0x20},
	{0x0810, 0x20},
	{0x0811, 0x20},
	{0x0812, 0x20},
	{0x0813, 0x1e},
	{0x0814, 0x1f},
	{0x0815, 0x1c},
	{0x0816, 0x1e},
	{0x0817, 0x1f},
	{0x0818, 0x1f},
	{0x0819, 0x1f},
	{0x081a, 0x1e},
	{0x081b, 0x1d},
	{0x081c, 0x1b},
	{0x081d, 0x19},
	{0x081e, 0x1a},
	{0x081f, 0x1c},
	{0x0820, 0x1a},
	{0x0821, 0x1b},
	{0x0822, 0x1b},
	{0x0823, 0x18},
	{0x0824, 0x18},
	{0x0825, 0x1d},
	{0x0826, 0x1d},
	{0x0827, 0x1d},
	{0x0828, 0x1e},
	{0x0829, 0x1e},
	{0x082a, 0x1e},
	{0x082b, 0x1d},
	{0x082c, 0x1e},
	{0x082d, 0x1e},
	{0x082e, 0x1f},
	{0x082f, 0x1f},
	{0x0830, 0x20},
	{0x0831, 0x1f},
	{0x0832, 0x1f},
	{0x0833, 0x1f},
	{0x0834, 0x1f},
	{0x0835, 0x1f},
	{0x0836, 0x20},
	{0x0837, 0x20},
	{0x0838, 0x20},
	{0x0839, 0x20},
	{0x083a, 0x20},
	{0x083b, 0x20},
	{0x083c, 0x20},
	{0x083d, 0x1f},
	{0x083e, 0x1f},
	{0x083f, 0x20},
	{0x0840, 0x20},
	{0x0841, 0x20},
	{0x0842, 0x20},
	{0x0843, 0x20},
	{0x0844, 0x20},
	{0x0845, 0x1f},
	{0x0846, 0x20},
	{0x0847, 0x20},
	{0x0848, 0x20},
	{0x0849, 0x20},
	{0x084a, 0x20},
	{0x084b, 0x20},
	{0x084c, 0x20},
	{0x084d, 0x20},
	{0x084e, 0x20},
	{0x084f, 0x20},
	{0x0850, 0x20},
	{0x0851, 0x20},
	{0x0852, 0x20},
	{0x0853, 0x20},
	{0x0854, 0x21},
	{0x0855, 0x1f},
	{0x0856, 0x20},
	{0x0857, 0x20},
	{0x0858, 0x21},
	{0x0859, 0x20},
	{0x085a, 0x21},
	{0x085b, 0x20},
	{0x085c, 0x20},
	{0x085d, 0x1f},
	{0x085e, 0x1f},
	{0x085f, 0x21},
	{0x0860, 0x20},
	{0x0861, 0x21},
	{0x0862, 0x20},
	{0x0863, 0x1f},
	{0x0864, 0x1f},
	{0x0865, 0x1c},
	{0x0866, 0x13},
	{0x0867, 0x0f},
	{0x0868, 0x0d},
	{0x0869, 0x0e},
	{0x086a, 0x10},
	{0x086b, 0x1a},
	{0x086c, 0x16},
	{0x086d, 0x06},
	{0x086e, 0x03},
	{0x086f, 0x02},
	{0x0870, 0x02},
	{0x0871, 0x02},
	{0x0872, 0x03},
	{0x0873, 0x04},
	{0x0874, 0x0c},
	{0x0875, 0x01},
	{0x0876, 0x01},
	{0x0877, 0x01},
	{0x0878, 0x01},
	{0x0879, 0x01},
	{0x087a, 0x01},
	{0x087b, 0x01},
	{0x087c, 0x01},
	{0x087d, 0x01},
	{0x087e, 0x00},
	{0x087f, 0x00},
	{0x0880, 0x00},
	{0x0881, 0x00},
	{0x0882, 0x00},
	{0x0883, 0x01},
	{0x0884, 0x01},
	{0x0885, 0x01},
	{0x0886, 0x00},
	{0x0887, 0x01},
	{0x0888, 0x00},
	{0x0889, 0x00},
	{0x088a, 0x00},
	{0x088b, 0x00},
	{0x088c, 0x00},
	{0x088d, 0x02},
	{0x088e, 0x01},
	{0x088f, 0x00},
	{0x0890, 0x00},
	{0x0891, 0x00},
	{0x0892, 0x00},
	{0x0893, 0x01},
	{0x0894, 0x02},
	{0x0895, 0x05},
	{0x0896, 0x03},
	{0x0897, 0x02},
	{0x0898, 0x01},
	{0x0899, 0x01},
	{0x089a, 0x02},
	{0x089b, 0x04},
	{0x089c, 0x06},
	{0x089d, 0x17},
	{0x089e, 0x0f},
	{0x089f, 0x0b},
	{0x08a0, 0x09},
	{0x08a1, 0x0a},
	{0x08a2, 0x0d},
	{0x08a3, 0x13},
	{0x08a4, 0x21},
	{0x08a5, 0x19},
	{0x08a6, 0x1a},
	{0x08a7, 0x1b},
	{0x08a8, 0x1b},
	{0x08a9, 0x1b},
	{0x08aa, 0x1b},
	{0x08ab, 0x1a},
	{0x08ac, 0x1b},
	{0x08ad, 0x1c},
	{0x08ae, 0x1d},
	{0x08af, 0x1e},
	{0x08b0, 0x1e},
	{0x08b1, 0x1e},
	{0x08b2, 0x1d},
	{0x08b3, 0x1d},
	{0x08b4, 0x1c},
	{0x08b5, 0x1d},
	{0x08b6, 0x1f},
	{0x08b7, 0x1f},
	{0x08b8, 0x20},
	{0x08b9, 0x20},
	{0x08ba, 0x20},
	{0x08bb, 0x1f},
	{0x08bc, 0x1d},
	{0x08bd, 0x1f},
	{0x08be, 0x1f},
	{0x08bf, 0x20},
	{0x08c0, 0x20},
	{0x08c1, 0x20},
	{0x08c2, 0x20},
	{0x08c3, 0x20},
	{0x08c4, 0x1f},
	{0x08c5, 0x1e},
	{0x08c6, 0x1f},
	{0x08c7, 0x20},
	{0x08c8, 0x21},
	{0x08c9, 0x21},
	{0x08ca, 0x20},
	{0x08cb, 0x20},
	{0x08cc, 0x1e},
	{0x08cd, 0x1e},
	{0x08ce, 0x1f},
	{0x08cf, 0x20},
	{0x08d0, 0x20},
	{0x08d1, 0x20},
	{0x08d2, 0x20},
	{0x08d3, 0x1f},
	{0x08d4, 0x1e},
	{0x08d5, 0x1e},
	{0x08d6, 0x1e},
	{0x08d7, 0x1f},
	{0x08d8, 0x1f},
	{0x08d9, 0x1f},
	{0x08da, 0x1e},
	{0x08db, 0x1d},
	{0x08dc, 0x1d},
	{0x08dd, 0x19},
	{0x08de, 0x1c},
	{0x08df, 0x1c},
	{0x08e0, 0x1b},
	{0x08e1, 0x1c},
	{0x08e2, 0x1c},
	{0x08e3, 0x1b},
	{0x08e4, 0x18},
	{0x08e5, 0x1c},
	{0x08e6, 0x1e},
	{0x08e7, 0x1e},
	{0x08e8, 0x1f},
	{0x08e9, 0x1f},
	{0x08ea, 0x1f},
	{0x08eb, 0x1d},
	{0x08ec, 0x20},
	{0x08ed, 0x1f},
	{0x08ee, 0x1f},
	{0x08ef, 0x21},
	{0x08f0, 0x21},
	{0x08f1, 0x21},
	{0x08f2, 0x21},
	{0x08f3, 0x20},
	{0x08f4, 0x20},
	{0x08f5, 0x20},
	{0x08f6, 0x20},
	{0x08f7, 0x21},
	{0x08f8, 0x21},
	{0x08f9, 0x21},
	{0x08fa, 0x20},
	{0x08fb, 0x21},
	{0x08fc, 0x21},
	{0x08fd, 0x1f},
	{0x08fe, 0x20},
	{0x08ff, 0x20},
	{0x0900, 0x20},
	{0x0901, 0x20},
	{0x0902, 0x20},
	{0x0903, 0x20},
	{0x0904, 0x21},
	{0x0905, 0x20},
	{0x0906, 0x20},
	{0x0907, 0x20},
	{0x0908, 0x21},
	{0x0909, 0x20},
	{0x090a, 0x20},
	{0x090b, 0x21},
	{0x090c, 0x21},
	{0x090d, 0x20},
	{0x090e, 0x21},
	{0x090f, 0x21},
	{0x0910, 0x21},
	{0x0911, 0x21},
	{0x0912, 0x21},
	{0x0913, 0x21},
	{0x0914, 0x22},
	{0x0915, 0x20},
	{0x0916, 0x21},
	{0x0917, 0x21},
	{0x0918, 0x22},
	{0x0919, 0x21},
	{0x091a, 0x22},
	{0x091b, 0x22},
	{0x091c, 0x22},
	{0x091d, 0x21},
	{0x091e, 0x21},
	{0x091f, 0x22},
	{0x0920, 0x22},
	{0x0921, 0x22},
	{0x0922, 0x21},
	{0x0923, 0x21},
	{0x0924, 0x1d},
	//blc
	{0xfffe, 0x30},
	{0x0013, 0x00},  //bias
	{0x0014, 0x00},
	{0x071b, 0x00},  //blc
	{0x0000, 0x01},
	{0x0002, 0x90},
	{0x0003, 0x31},
	{0x006C, 0x40},
	{0x0715, 0x04},
	//AWB
	{0xfffe, 0x14},
	{0x0248, 0x02},
	{0x0282, 0x06},   //int B gain
	{0x0283, 0x00},
	{0x0286, 0x04},   //int Gb gain  G
	{0x0287, 0x00},
	{0x028a, 0x04},  //int Gr gain   G
	{0x028b, 0x00},
	{0x028e, 0x04},   //int R gain
	{0x028f, 0x04},
	{0x02b6, 0x06},
	{0x02b7, 0x48},
	{0x02ba, 0x04},
	{0x02bb, 0x00},
	{0x02be, 0x07},
	{0x02bf, 0x17},
	{0xfffe, 0x14},
	{0x0248, 0x01},
	{0x0249, 0x01},//AWBFlexiMap_en
	{0x024a, 0x00},//AWBMove_en
	{0x027a, 0x08},//nCTBasedMinNum    //20  ////01
	{0x027b, 0x00},
	{0x027c, 0x0f},
	{0x027d, 0xff},//nMaxAWBGain
	{0x002c, 0x01}, ////
	////0x027e,0x0
	{0xfffe, 0x30},
	{0x0708, 0x03}, //pixel max value[9:8]  //03 ////
	{0x0709, 0xf0}, //pixel max value[7:0]  //f0 ////
	{0x070a, 0x00}, //pixel min value[9:8]
	{0x070b, 0xc0}, //pixel min value[7:0]  ////
	{0x071c, 0x0a}, //simple awb ////
	{0xfffd, 0x80},
	{0xfffe, 0x30},
	{0x0730, 0x90},    // win1 startx
	{0x0731, 0xc7},    // win1 endx
	{0x0732, 0x3a},    // win1 starty
	{0x0733, 0x55},    // win1 endy
	{0x0734, 0x80},    // win2 startx
	{0x0735, 0xa8},    // win2 endx
	{0x0736, 0x50},    // win2 starty
	{0x0737, 0x70},    // win2 endy
	{0x0738, 0x50},    // win3 startx
	{0x0739, 0x77},    // win3 endx
	{0x073a, 0x60},    // win3 starty
	{0x073b, 0x8b},    // win3 endy
	{0x073c, 0x77},    // win4 startx
	{0x073d, 0x9e},    // win4 endx
	{0x073e, 0x49},    // win4 starty
	{0x073f, 0x60},    // win4 endy
	{0x0740, 0x75},    // win5 startx
	{0x0741, 0x83},    // win5 endx
	{0x0742, 0x62},    // win5 starty
	{0x0743, 0x70},    // win5 endy
	{0x0744, 0x00},    // win6 startx
	{0x0745, 0x00},    // win6 endx
	{0x0746, 0x00},    // win6 starty
	{0x0747, 0x00},    // win6 endy
	{0x0748, 0x00},    // win7 startx
	{0x0749, 0x00},    // win7 endx
	{0x074a, 0x00},    // win7 starty
	{0x074b, 0x00},    // win7 endy
	{0x074c, 0x00},    // win8 startx
	{0x074d, 0x00},    // win8 endx
	{0x074e, 0x00},    // win8 starty
	{0x074f, 0x00},    // win8 endy
	{0x0750, 0x00},    // win9 startx
	{0x0751, 0x00},    // win9 endx
	{0x0752, 0x00},    // win9 starty
	{0x0753, 0x00},    // win9 endy
	{0x0754, 0x00},    // win10 startx
	{0x0755, 0x00},    // win10 endx
	{0x0756, 0x00},    // win10 starty
	{0x0757, 0x00},    // win10 endy
	{0x0758, 0x00},    // win11 startx
	{0x0759, 0x00},    // win11 endx
	{0x075a, 0x00},    // win11 starty
	{0x075b, 0x00},    // win11 endy
	{0x075c, 0x00},    // win12 startx
	{0x075d, 0x00},    // win12 endx
	{0x075e, 0x00},    // win12 starty
	{0x075f, 0x00},    // win12 endy
	{0x0760, 0x00},    // win13 startx
	{0x0761, 0x00},    // win13 endx
	{0x0762, 0x00},    // win13 starty
	{0x0763, 0x00},    // win13 endy
	{0x0764, 0x00},    // win14 startx
	{0x0765, 0x00},    // win14 endx
	{0x0766, 0x00},    // win14 starty
	{0x0767, 0x00},    // win14 endy
	{0x0768, 0x00},    // win15 startx
	{0x0769, 0x00},    // win15 endx
	{0x076a, 0x00},    // win15 starty
	{0x076b, 0x00},    // win15 endy
	{0x076c, 0x00},    // win16 startx
	{0x076d, 0x00},    // win16 endx
	{0x076e, 0x00},    // win16 starty
	{0x076f, 0x00},    // win16 endy
	{0x0770, 0x22},    // wt1 wt2
	{0x0771, 0x21},    // wt3 wt4
	{0x0772, 0x10},    // wt5 wt6
	{0x0773, 0x00},    // wt7 wt8
	{0x0774, 0x00},    // wt9 wt10
	{0x0775, 0x00},    // wt11 wt12
	{0x0776, 0x00},    // wt13 wt14
	{0x0777, 0x00},    // wt15 wt16
	{0xfffd, 0x80},
	{0xfffe, 0x30},
	{0x1400, 0x00},
	{0x1401, 0x09},
	{0x1402, 0x11},
	{0x1403, 0x19},
	{0x1404, 0x21},
	{0x1405, 0x29},
	{0x1406, 0x31},
	{0x1407, 0x39},
	{0x1408, 0x40},
	{0x1409, 0x46},
	{0x140a, 0x4c},
	{0x140b, 0x52},
	{0x140c, 0x58},
	{0x140d, 0x5d},
	{0x140e, 0x61},
	{0x140f, 0x66},
	{0x1410, 0x6a},
	{0x1411, 0x6e},
	{0x1412, 0x71},
	{0x1413, 0x75},
	{0x1414, 0x78},
	{0x1415, 0x7b},
	{0x1416, 0x7e},
	{0x1417, 0x81},
	{0x1418, 0x83},
	{0x1419, 0x86},
	{0x141a, 0x88},
	{0x141b, 0x8b},
	{0x141c, 0x8d},
	{0x141d, 0x8f},
	{0x141e, 0x92},
	{0x141f, 0x94},
	{0x1420, 0x96},
	{0x1421, 0x99},
	{0x1422, 0x9d},
	{0x1423, 0xa0},
	{0x1424, 0xa4},
	{0x1425, 0xa6},
	{0x1426, 0xa9},
	{0x1427, 0xac},
	{0x1428, 0xae},
	{0x1429, 0xb0},
	{0x142a, 0xb2},
	{0x142b, 0xb4},
	{0x142c, 0xb6},
	{0x142d, 0xb8},
	{0x142e, 0xba},
	{0x142f, 0xbc},
	{0x1430, 0xbd},
	{0x1431, 0xc1},
	{0x1432, 0xc4},
	{0x1433, 0xc7},
	{0x1434, 0xca},
	{0x1435, 0xcd},
	{0x1436, 0xd0},
	{0x1437, 0xd4},
	{0x1438, 0xd7},
	{0x1439, 0xdc},
	{0x143a, 0xe0},
	{0x143b, 0xe5},
	{0x143c, 0xea},
	{0x143d, 0xef},
	{0x143e, 0xf5},
	{0x143f, 0xfa},
	{0x1440, 0xff},
	{0xfffe, 0x30},
	{0x1200, 0x05},
	{0x1201, 0xC5},
	{0x1202, 0x00},
	{0x1203, 0xe0},
	{0x1204, 0x01},
	{0x1205, 0x1A},
	{0x1206, 0x01},
	{0x1207, 0x35},
	{0x1208, 0x00},
	{0x1209, 0x44},
	{0x120A, 0x04},
	{0x120B, 0xE6},
	{0x120C, 0x0C},
	{0x120D, 0x6A},
	{0x120E, 0x03},
	{0x120F, 0x0D},
	{0x1210, 0x00},
	{0x1211, 0xF2},
	{0x1212, 0x03},
	{0x1213, 0x50},
	{0x1214, 0x00},
	{0x1215, 0xF8},
	{0x1216, 0x02},
	{0x1217, 0x3F},
	{0x1218, 0x07},
	{0x1219, 0xE6},
	{0x121A, 0x00},
	{0x121B, 0x87},
	{0x121C, 0x00},
	{0x121D, 0x3C},
	{0x121E, 0x01},
	{0x121F, 0xea},
	{0x1220, 0x00},
	{0x1221, 0x4E},
	{0x1222, 0x04},
	{0x1223, 0xCF},
	{0x122e, 0x00},
	{0x122F, 0x00},
	{0x1230, 0x02},
	{0x1228, 0x00},
	{0x1229, 0x6D},
	{0x122A, 0x00},
	{0x122B, 0xAE},
	{0x122C, 0x01},
	{0x122D, 0x37},
	{0x1231, 0x02},
	{0xfffe, 0x30},
	{0x2000, 0x0f},
	{0x2001, 0x16},
	{0x2002, 0x1a},
	{0x2003, 0x1f},
	{0x2004, 0x20},
	{0x2005, 0x24},
	{0x2006, 0x18},
	{0x2007, 0x12},
	{0x2008, 0x10},
	{0x2009, 0x06},
	{0x200a, 0x01},
	{0x200b, 0x01},
	{0x200c, 0x10},
	{0x200d, 0x0e},
	{0x200e, 0x0a},
	{0x200f, 0x06},
	{0x2010, 0x01},
	{0x2011, 0x01},
	{0x2012, 0x28},
	{0x2013, 0x2a},
	{0x2014, 0x2e},
	{0x2015, 0x20},
	{0x2016, 0x24},
	{0x2017, 0x28},
	{0x2018, 0x20},
	{0x2019, 0x24},
	{0x201a, 0x28},
	{0x201b, 0x2c},
	{0x201c, 0x30},
	{0x201d, 0x34},
	{0x201e, 0x2c},
	{0x201f, 0x28},
	{0x2020, 0x30},
	{0x2021, 0x20},
	{0x2022, 0x30},
	{0x2023, 0x34},
	{0x2024, 0x10},
	{0x2025, 0x2c},
	{0x2026, 0x30},
	{0x2027, 0x24},
	{0x2028, 0x30},
	{0x2029, 0x34},
	{0x202a, 0x20},
	{0x202b, 0x20},
	{0x202c, 0x20},
	{0x202d, 0x20},
	{0x202e, 0x20},
	{0x202f, 0x20},
	{0x2030, 0x20},
	{0x2031, 0x20},
	{0x2032, 0x20},
	{0x2033, 0x20},
	{0x2034, 0x20},
	{0x2035, 0x20},
	{0x2036, 0x20},
	{0x2037, 0x20},
	{0x2038, 0x20},
	{0x2039, 0x20},
	{0x203a, 0x20},
	{0x203b, 0x20},
	{0x203c, 0x20},
	{0x203d, 0x20},
	{0x203e, 0x20},
	{0x203f, 0x20},
	{0x2040, 0x20},
	{0x2041, 0x20},
	{0x2044, 0x10},
	{0xfffe, 0x30},
	{0x2100, 0x01},
	{0x2101, 0x01},
	{0x2102, 0x01},
	{0x2103, 0x01},
	{0x2104, 0x3f},
	{0x2105, 0x4f},
	{0x2106, 0x5f},
	{0x2107, 0x08},
	{0xfffe, 0x30},
	{0x0f00, 0x65},
	{0x0f02, 0x00},
	{0x0f03, 0x00},
	{0x0f04, 0x00},
	{0x0f05, 0x00},
	{0x0f06, 0x00},
	{0x0f07, 0x00},
	{0x0f08, 0x00},
	{0x0f09, 0x00},
	{0x0f0a, 0x02},
	{0x0f0b, 0x2f},
	{0x0f0c, 0x01},
	{0x0f0d, 0x03},
	{0x0f0e, 0x06},
	{0x0f0f, 0x0c},
	{0x0f10, 0x30},
	{0xfffe, 0x30},
	{0x2b04, 0x08},
	{0x2b16, 0x05},
	{0xfffe, 0x14},
	{0x002f, 0x01},
	{0x09a8, 0x01},
	{0x09ab, 0x02},
	{0x09ac, 0x08},
	{0x09ad, 0xb4},
	{0x09ae, 0x41},
	{0x09af, 0xFF},
	{0x09b2, 0x02},
	{0x09b3, 0x00},
	{0x09b4, 0x10},
	{0x09b5, 0x14},
	{0x09b6, 0x00},
	{0x09bb, 0x30},
	{0x09bc, 0x31},
	{0x09bd, 0x67},
	{0x09be, 0x68},
	{0x09bf, 0xf7},
	{0x09c0, 0xf8},
	{0x09c1, 0xff},
	{0x09c2, 0x0c},
	{0x09c3, 0x65},
	{0x09c4, 0x66},
	{0x09c5, 0x89},
	{0x09c6, 0x8a},
	{0x09c7, 0xf8},
	{0x09c8, 0xfa},
	{0x09c9, 0xff},
	{0xfffe, 0x30},
	{0x2b9a, 0x00},
	{0x2b9b, 0x00},
	{0x2b9c, 0x10},
	{0x2b9d, 0x00},
	{0x2b9e, 0x20},
	{0x2b9f, 0x00},
	{0x2ba0, 0x40},
	{0x2ba1, 0x00},
	{0x2ba2, 0x70},
	{0x2ba3, 0x00},
	{0x2ba4, 0xa0},
	{0x2ba5, 0x00},
	{0x2ba6, 0xd0},
	{0x2ba7, 0x00},
	{0x2ba8, 0x00},
	{0x2ba9, 0x01},
	{0x2bac, 0x18},
	{0x2bad, 0x18},
	{0x2bae, 0x1a},
	{0x2baf, 0x1a},
	{0x2bb0, 0x1b},
	{0x2bb1, 0x1b},
	{0x2bb2, 0x1d},
	{0x2bb3, 0x20},
	{0x2bb4, 0x7f},
	{0x2bb5, 0x7f},
	{0x2bb6, 0x7f},
	{0x2bb7, 0x7f},
	{0x2bb8, 0x7f},
	{0x2bb9, 0x7f},
	{0x2bba, 0x7f},
	{0x2bbb, 0x7f},
	{0x2bbc, 0x24},
	{0x2bbd, 0x24},
	{0x2bbe, 0x20},
	{0x2bbf, 0x12},
	{0x2bc0, 0x0e},
	{0x2bc1, 0x0e},
	{0x2bc2, 0x0c},
	{0x2bc3, 0x0c},
	{0xfffe, 0x14},
	{0x026c, 0x00},
	{0x026e, 0x50},
	{0x026f, 0x40},
	{0x0270, 0x40},
	{0x0271, 0x30},
	{0x0272, 0x30},
	{0x0273, 0x30},
	{0x0274, 0x50},
	{0x0275, 0x3A},
	{0x0276, 0x3A},
	{0x0277, 0x2A},
	{0x0278, 0x2A},
	{0x0279, 0x2A},
	{0xfffe, 0x30},
	{0x0000, 0x47},
	{0x0001, 0x92},
	{0x0002, 0x96},
	{0x0003, 0x35},
	{0x2b16, 0x04},
	{0x2b04, 0x08},
	{0x1700, 0x09},
	{0x1701, 0x32},
	{0x1702, 0x43},
	{0x1704, 0x22},
	{0x071b, 0x00},
	{0x071c, 0x0a},
	{0xfffe, 0x14},
	{0x024a, 0x02},
	{0x026a, 0x84},
	{0x026b, 0x80},
	{REG_NULL_XC7080, 0x00},
};

/*
 * Xclk 24Mhz
 * max_framerate 120fps
 * mipi_datarate per lane 800Mbps
 */
static const struct regval1 xc7080_xc530_2560x1440_regs[] = {
	{0xEF, 0x05},
	{0x0F, 0x00},
	{0x43, 0x02},
	{0x44, 0x00},//T_pll_enh=0
	{0xED, 0x01},
	{0xEF, 0x01},
	{0xF5, 0x01},
	{0x09, 0x01},
	{0xEF, 0x00},
	{0x06, 0x02},
	{0x0B, 0x00},
	{0x0C, 0xA0},
	{0x10, 0x80},//Cmd_BYTECLK_InvSel=0 for PS308
	{0x11, 0x80},//GatedAllClk enable
	{0x12, 0x80},
	{0x13, 0x00},
	{0x14, 0xBD},
	{0x15, 0x1F},
	{0x16, 0xBD},
	{0x17, 0xBD},
	{0x18, 0xBD},
	{0x19, 0x64},
	{0x1B, 0x64},
	{0x1C, 0x64},
	{0x1E, 0x64},
	{0x1F, 0x64},
	{0x20, 0x64},
	{0x21, 0x00},
	{0x23, 0x00},
	{0x24, 0x00},
	{0x26, 0x00},
	{0x27, 0x00},
	{0x28, 0x00},
	{0x29, 0x64},
	{0x2B, 0x64},
	{0x2C, 0x64},
	{0x2E, 0x64},
	{0x2F, 0x64},
	{0x30, 0x64},
	{0x31, 0x0F},
	{0x32, 0x00},
	{0x33, 0x64},
	{0x34, 0x64},
	{0x35, 0x01},
	{0x36, 0x0F},
	{0x37, 0x0F},
	{0x38, 0xE0},
	{0x39, 0x00},
	{0x3A, 0x00},
	{0x3B, 0x00},
	{0x3C, 0x00},
	{0x45, 0x00},
	{0x46, 0x00},
	{0x48, 0x00},
	{0x49, 0xD2},
	{0x4A, 0x0A},
	{0x4B, 0x6E},
	{0x4C, 0x01},
	{0x4D, 0xD2},
	{0x4E, 0xC4},
	{0x4F, 0x28},
	{0x53, 0x00},
	{0x57, 0x00},
	{0x58, 0x06},
	{0x59, 0x05},
	{0x5A, 0x04},
	{0x5B, 0x03},
	{0x5C, 0x02},
	{0x5D, 0x01},
	{0x5F, 0xC2},
	{0x60, 0x44},
	{0x61, 0x54},
	{0x62, 0x29},
	{0x69, 0x10},
	{0x6A, 0x40},
	{0x6B, 0x00},
	{0x6C, 0x00},
	{0x6D, 0x00},
	{0x6E, 0x00},
	{0x6F, 0x32},
	{0x74, 0x00},
	{0x75, 0x78},
	{0x76, 0x00},
	{0x77, 0x20},
	{0x7E, 0x32},
	{0x7F, 0x90},
	{0x80, 0x02},
	{0x81, 0x80},
	{0x84, 0xFE},
	{0x85, 0x22},
	{0x86, 0x09},
	{0x87, 0x60},
	{0x88, 0x0B},
	{0x89, 0x01},
	{0x8B, 0x76},
	{0x8C, 0x00},
	{0x8D, 0x00},
	{0x8E, 0x00},
	{0x8F, 0x00},
	{0x90, 0x00},
	{0x91, 0x00},
	{0x92, 0x11},
	{0x93, 0x10},
	{0x94, 0x00},
	{0x95, 0x00},
	{0x96, 0x00},
	{0x97, 0x00},
	{0x98, 0x02},
	{0x99, 0x00},
	{0x9A, 0x00},
	{0x9B, 0x09},
	{0x9C, 0x00},
	{0x9D, 0x00},
	{0x9E, 0x00},
	{0x9F, 0x00},
	{0xA0, 0x02},
	{0xA1, 0x00},
	{0xA2, 0x0A},
	{0xA3, 0x07},
	{0xA4, 0xFF},
	{0xA5, 0x03},
	{0xA6, 0xFF},
	{0xA7, 0x00},
	{0xA8, 0x00},
	{0xA9, 0x11},
	{0xAA, 0x54},
	{0xAB, 0x10},
	{0xAC, 0x11},
	{0xAD, 0x00},
	{0xAE, 0x00},
	{0xAF, 0x00},
	{0xB0, 0x02},
	{0xB1, 0x00},
	{0xB2, 0x13},
	{0xBE, 0x05},
	{0xBF, 0x00},
	{0xC0, 0x10},
	{0xC7, 0x10},
	{0xC8, 0x01},
	{0xC9, 0x00},
	{0xCA, 0x55},
	{0xCB, 0x06},
	{0xCC, 0x09},
	{0xCD, 0x00},
	{0xCE, 0xA2},
	{0xCF, 0x00},
	{0xD0, 0x02},
	{0xD1, 0x10},
	{0xD2, 0x1E},
	{0xD3, 0x19},
	{0xD4, 0x04},
	{0xD5, 0x18},
	{0xD6, 0xC8},
	{0xD8, 0x10},
	{0xD9, 0x30},
	{0xDA, 0x40},
	{0xDE, 0x00},
	{0xDF, 0x24},
	{0xE0, 0x0D},
	{0xE1, 0x05},
	{0xE2, 0x05},
	{0xE3, 0x24},
	{0xE4, 0x0D},
	{0xE5, 0x05},
	{0xE6, 0x05},
	{0xF0, 0x80},
	{0xF1, 0x00},
	{0xF2, 0x00},
	{0xF3, 0xB0},//B02A: Version
	{0xF4, 0x00},
	{0xF5, 0x48},
	{0xF6, 0x00},
	{0xF7, 0x00},
	{0xF8, 0x2A},//B02A: Version
	{0xF9, 0x00},
	{0xFA, 0x00},
	{0xFB, 0x00},
	{0xFC, 0x00},
	{0xFD, 0x00},
	{0xFE, 0x00},
	{0xED, 0x01},
	{0xEF, 0x01},
	{0x02, 0xFF},
	{0x03, 0x03},
	{0x04, 0x00},
	{0x05, 0x0B},
	{0x06, 0xFF},
	{0x07, 0x04},
	{0x08, 0x00},
	{0x09, 0x00},
	{0x0A, 0x07},
	{0x0B, 0xBB},
	{0x0C, 0x00},
	{0x0D, 0x03},
	{0x0E, 0x0A},
	{0x0F, 0x12},
	{0x10, 0x00},
	{0x11, 0x78},
	{0x12, 0x78},
	{0x13, 0x78},
	{0x14, 0x01},
	{0x15, 0x00},
	{0x16, 0x00},
	{0x17, 0x00},
	{0x1A, 0x00},
	{0x1B, 0x0A},
	{0x1C, 0x30},
	{0x1D, 0x07},
	{0x1E, 0xAE},
	{0x1F, 0x00},
	{0x20, 0x02},
	{0x21, 0x00},
	{0x22, 0x00},
	{0x23, 0x00},
	{0x24, 0x00},
	{0x25, 0x00},
	{0x26, 0x08},
	{0x27, 0x0C},//A8C->CA8 for 25fps
	{0x28, 0xA8},
	{0x29, 0x1E},
	{0x2A, 0x56},
	{0x2C, 0x00},
	{0x2D, 0x06},
	{0x2E, 0x78},
	{0x2F, 0x00},
	{0x34, 0x01},
	{0x35, 0x01},
	{0x37, 0x2C},
	{0x39, 0x36},
	{0x3D, 0x00},
	{0x3E, 0x10},
	{0x3F, 0xA6},
	{0x40, 0x96},
	{0x41, 0x11},
	{0x42, 0xF4},
	{0x43, 0xD6},
	{0x47, 0x00},
	{0x48, 0x08},
	{0x49, 0x00},
	{0x4A, 0x0E},
	{0x4B, 0x07},
	{0x4C, 0x98},
	{0x4D, 0x00},
	{0x4E, 0x02},
	{0x4F, 0x08},
	{0x50, 0x08},
	{0x51, 0x0A},
	{0x52, 0x08},
	{0x53, 0x00},
	{0x54, 0x02},
	{0x55, 0x00},
	{0x56, 0x08},
	{0x57, 0x00},
	{0x58, 0x00},
	{0x59, 0x0A},
	{0x5A, 0x20},
	{0x5B, 0x20},
	{0x5C, 0x1E},
	{0x5D, 0x0A},
	{0x5E, 0xB2},
	{0x5F, 0x00},
	{0x60, 0xC8},
	{0x61, 0x05},
	{0x62, 0xDC},
	{0x63, 0x07},
	{0x64, 0x98},
	{0x65, 0x03},
	{0x66, 0xCC},
	{0x67, 0x00},
	{0x68, 0xFA},
	{0x69, 0xC8},
	{0x6B, 0x08},
	{0x6C, 0x00},
	{0x6D, 0xC8},
	{0x6E, 0x00},
	{0x6F, 0x08},
	{0x70, 0x00},
	{0x71, 0xC8},
	{0x72, 0x00},
	{0x73, 0x98},
	{0x74, 0x00},
	{0x75, 0x56},
	{0x76, 0x00},
	{0x77, 0x00},
	{0x78, 0x00},
	{0x79, 0x0F},
	{0x7A, 0x02},
	{0x7B, 0xC8},
	{0x7C, 0x02},
	{0x7D, 0x06},
	{0x7E, 0x04},
	{0x7F, 0x00},
	{0x80, 0x00},
	{0x81, 0x00},
	{0x82, 0x00},
	{0x83, 0x00},
	{0x84, 0x12},
	{0x85, 0xC0},
	{0x86, 0x00},
	{0x87, 0x00},
	{0x88, 0x0C},
	{0x89, 0x00},
	{0x8A, 0x07},
	{0x8B, 0xAE},
	{0x8C, 0x00},
	{0x8D, 0x02},
	{0x8E, 0x10},
	{0x8F, 0x0F},
	{0x90, 0x01},
	{0x91, 0x01},
	{0x92, 0x80},
	{0x93, 0x00},
	{0x94, 0xFF},
	{0x96, 0x80},
	{0x97, 0x01},
	{0x99, 0x12},
	{0x9A, 0x14},
	{0x9B, 0x0E},
	{0x9C, 0x14},
	{0x9E, 0x14},
	{0x9F, 0x00},
	{0xA0, 0x00},
	{0xA1, 0x00},
	{0xA2, 0x00},
	{0xA3, 0x01},
	{0xA4, 0x13},
	{0xA5, 0x05},
	{0xA6, 0xA0},
	{0xA7, 0x00},
	{0xA8, 0x19},
	{0xA9, 0x0A},
	{0xAA, 0x00},
	{0xAB, 0x01},
	{0xAD, 0x00},
	{0xAE, 0x50},
	{0xAF, 0x00},
	{0xB0, 0x50},
	{0xB1, 0x02},
	{0xB2, 0x01},
	{0xB3, 0x03},
	{0xB5, 0x04},
	{0xB6, 0x0A},
	{0xB7, 0x14},
	{0xB8, 0x01},
	{0xB9, 0x2C},
	{0xBA, 0xA0},
	{0xBB, 0xA0},
	{0xBC, 0x00},
	{0xBD, 0x01},
	{0xBE, 0x07},
	{0xBF, 0xFF},
	{0xC0, 0x07},
	{0xC1, 0xFF},
	{0xC2, 0x00},
	{0xC3, 0x30},
	{0xC4, 0x54},
	{0xC6, 0x10},
	{0xC7, 0x10},
	{0xC8, 0x60},
	{0xC9, 0x55},
	{0xCB, 0x32},
	{0xCC, 0x96},
	{0xCD, 0x64},
	{0xCE, 0x30},
	{0xCF, 0x82},
	{0xD0, 0x02},
	{0xD1, 0x60},
	{0xD3, 0x01},
	{0xD4, 0x04},
	{0xD5, 0x61},
	{0xD6, 0x00},
	{0xD7, 0x0A},
	{0xD8, 0xA0},
	{0xDA, 0x70},
	{0xDC, 0x00},
	{0xDD, 0x42},
	{0xE0, 0x08},
	{0xE1, 0x11},
	{0xE2, 0x0A},
	{0xE3, 0xA1},
	{0xE4, 0x00},
	{0xE6, 0x00},
	{0xE7, 0x00},
	{0xEA, 0x08},
	{0xF0, 0x8D},
	{0xF1, 0x16},
	{0xF2, 0x26},
	{0xF3, 0x1C},
	{0xF4, 0x11},
	{0xF5, 0x19},
	{0xF6, 0x05},
	{0xF7, 0x00},
	{0xF8, 0x48},
	{0xF9, 0x07},
	{0xFA, 0x21},
	{0xFB, 0x02},
	{0x09, 0x01},
	{0xEF, 0x02},
	{0x10, 0x00},
	{0x20, 0x03},
	{0x21, 0x18},
	{0x22, 0x0C},
	{0x23, 0x08},
	{0x24, 0x05},
	{0x25, 0x03},
	{0x26, 0x02},
	{0x27, 0x18},
	{0x28, 0x0C},
	{0x29, 0x08},
	{0x2A, 0x05},
	{0x2B, 0x03},
	{0x2C, 0x02},
	{0x2E, 0x04},
	{0x30, 0x9F},
	{0x31, 0x06},
	{0x32, 0x07},
	{0x33, 0x84},
	{0x34, 0x00},
	{0x35, 0x00},
	{0x36, 0x01},
	{0x37, 0x00},
	{0x38, 0x00},
	{0x39, 0x00},
	{0x3A, 0xCE},
	{0x3B, 0x15},
	{0x3C, 0xFA},
	{0x3D, 0x04},
	{0x3E, 0x00},
	{0x3F, 0x0A},
	{0x40, 0x08},
	{0x41, 0x09},
	{0x42, 0x0B},
	{0x43, 0x0A},
	{0x45, 0x00},
	{0x46, 0x00},
	{0x47, 0x00},
	{0x48, 0x00},
	{0x49, 0x00},
	{0x4A, 0x00},
	{0x4B, 0x00},
	{0x4C, 0x00},
	{0x4D, 0x00},
	{0x4E, 0x02},
	{0xA0, 0x00},
	{0xA1, 0x00},
	{0xA2, 0x00},
	{0xA3, 0x00},
	{0xA4, 0x00},
	{0xA5, 0x00},
	{0xA6, 0x00},
	{0xA7, 0x00},
	{0xA9, 0x00},
	{0xAA, 0x00},
	{0xAB, 0x00},
	{0xB2, 0x01},
	{0xB3, 0x00},
	{0xB4, 0x03},
	{0xB5, 0x01},
	{0xB6, 0x03},
	{0xB9, 0x00},
	{0xBA, 0x00},
	{0xBB, 0x00},
	{0xBC, 0x00},
	{0xBD, 0x00},
	{0xBE, 0x00},
	{0xED, 0x01},
	{0xEF, 0x05},
	{0x03, 0x10},
	{0x04, 0x38},
	{0x05, 0x04},
	{0x06, 0x64},//RAW10
	{0x07, 0x80},
	{0x08, 0x07},
	{0x09, 0x09},
	{0x0A, 0x05},
	{0x0B, 0x06},
	{0x0C, 0x04},
	{0x0D, 0x5E},
	{0x0E, 0x01},
	{0x0F, 0x00},//MIPI CSI disable
	{0x10, 0x02},//2Lane
	{0x11, 0x01},
	{0x12, 0x00},
	{0x13, 0x00},
	{0x14, 0xB8},
	{0x15, 0x07},
	{0x16, 0x06},
	{0x17, 0x06},
	{0x18, 0x05},
	{0x19, 0x04},
	{0x1A, 0x06},
	{0x1B, 0x03},
	{0x1C, 0x04},
	{0x1D, 0x08},
	{0x1E, 0x1A},
	{0x1F, 0x00},
	{0x20, 0x00},
	{0x21, 0x1E},
	{0x22, 0x1E},
	{0x23, 0x01},
	{0x24, 0x04},
	{0x25, 0x00},
	{0x27, 0x00},
	{0x28, 0x00},
	{0x2A, 0x18},
	{0x2B, 0x06},
	{0x2C, 0xC4},
	{0x2D, 0x09},
	{0x2E, 0x00},
	{0x2F, 0x06},
	{0x30, 0x00},
	{0x31, 0x06},
	{0x32, 0x00},
	{0x33, 0x00},
	{0x34, 0x00},
	{0x35, 0x00},
	{0x36, 0x00},
	{0x37, 0x00},
	{0x38, 0x2F},
	{0x3A, 0x02},
	{0x3B, 0x00},
	{0x3C, 0x00},
	{0x3D, 0x00},
	{0x3E, 0x00},
	{0x40, 0x16},
	{0x41, 0x22},//2b->22
	{0x43, 0x02},
	{0x44, 0x01},
	{0x49, 0x01},
	{0x4A, 0x02},
	{0x4D, 0x02},
	{0x4F, 0x01},
	{0x5B, 0x00},
	{0x5C, 0x03},
	{0x5D, 0x00},
	{0x5E, 0x07},
	{0x5F, 0x08},
	{0x60, 0x00},
	{0x61, 0x00},
	{0x62, 0x00},
	{0x63, 0x28},
	{0x64, 0x30},
	{0x65, 0x9E},
	{0x66, 0xB9},
	{0x67, 0x52},
	{0x68, 0x70},
	{0x69, 0x4E},
	{0x70, 0x00},
	{0x71, 0x00},
	{0x72, 0x00},
	{0x90, 0x04},
	{0x91, 0x01},
	{0x92, 0x00},
	{0x93, 0x00},
	{0x94, 0x04},
	{0x96, 0x00},
	{0x97, 0x01},
	{0x98, 0x01},
	{0x99, 0x01},
	{0x9A, 0x01},
	{0xA0, 0x00},
	{0xA1, 0x01},
	{0xA2, 0x00},
	{0xA3, 0x01},
	{0xA4, 0x00},
	{0xA5, 0x01},
	{0xA6, 0x00},
	{0xA7, 0x00},
	{0xAA, 0x00},
	{0xAB, 0x0F},
	{0xAC, 0x08},
	{0xAD, 0x09},
	{0xAE, 0x0A},
	{0xAF, 0x0B},
	{0xB0, 0x00},
	{0xB1, 0x04},
	{0xB2, 0x01},
	{0xB3, 0x00},
	{0xB4, 0x00},
	{0xB5, 0x0A},
	{0xB6, 0x0A},
	{0xB7, 0x0A},
	{0xB8, 0x0A},
	{0xB9, 0x00},
	{0xBA, 0x00},
	{0xBB, 0x00},
	{0xBC, 0x00},
	{0xBD, 0x00},
	{0xBE, 0x00},
	{0xBF, 0x00},
	{0xC5, 0x00},
	{0xC6, 0x00},
	{0xC7, 0x00},
	{0xC8, 0x00},
	{0xCE, 0x00},
	{0xCF, 0x63},
	{0xD3, 0x80},
	{0xD4, 0x00},
	{0xD5, 0x00},
	{0xD6, 0x03},
	{0xD7, 0x77},
	{0xD8, 0x00},
	{0xED, 0x01},
	{0xEF, 0x06},
	{0x00, 0x0C},
	{0x01, 0x00},
	{0x02, 0x13},
	{0x03, 0x5C},
	{0x04, 0x00},
	{0x05, 0x30},
	{0x06, 0x02},
	{0x07, 0x00},
	{0x08, 0x00},
	{0x09, 0x02},
	{0x0A, 0x15},
	{0x0B, 0x90},
	{0x0C, 0x90},
	{0x0D, 0x90},
	{0x0E, 0x00},
	{0x0F, 0x1B},
	{0x10, 0x20},
	{0x11, 0x1B},
	{0x12, 0x20},
	{0x17, 0x00},
	{0x18, 0x40},
	{0x19, 0x00},
	{0x1A, 0x40},
	{0x28, 0x03},
	{0x29, 0x00},
	{0x2A, 0x00},
	{0x2B, 0x20},
	{0x2C, 0x08},
	{0x2D, 0x00},
	{0x2E, 0x20},
	{0x2F, 0x20},
	{0x45, 0x80},
	{0x46, 0x00},
	{0x49, 0x0C},
	{0x4A, 0x40},
	{0x4B, 0x40},
	{0x5A, 0x00},
	{0x5B, 0x00},
	{0x5C, 0x00},
	{0x5D, 0x00},
	{0x5E, 0x90},
	{0x5F, 0x90},
	{0x60, 0x90},
	{0x61, 0x10},
	{0x70, 0x00},
	{0x71, 0x00},
	{0x72, 0x00},
	{0x7C, 0x00},
	{0x7D, 0x00},
	{0x7E, 0x00},
	{0x83, 0x00},
	{0x84, 0x00},
	{0x85, 0x00},
	{0x8A, 0x00},
	{0x8B, 0x00},
	{0x8C, 0x00},
	{0x91, 0x00},
	{0x92, 0x00},
	{0x93, 0x00},
	{0x98, 0x05},
	{0x99, 0x60},
	{0x9A, 0x67},
	{0x9B, 0x44},
	{0x9C, 0x44},
	{0x9D, 0x04},
	{0x9E, 0x42},
	{0x9F, 0x44}, // Cmd_LTM_LThd[4:0]=8
	{0xA0, 0x00},
	{0xA1, 0x01},
	{0xA2, 0x00},
	{0xA3, 0x00},
	{0xAC, 0x00},
	{0xAD, 0x03},
	{0xAE, 0x00},
	{0xAF, 0xE8},
	{0xB0, 0x00},
	{0xB1, 0x80},
	{0xB2, 0x80},
	{0xB3, 0x80},
	{0xB4, 0x80},
	{0xBD, 0x00},
	{0xBE, 0x03},
	{0xBF, 0x00},
	{0xC0, 0xE8},
	{0xC1, 0x00},
	{0xC2, 0x80},
	{0xC3, 0x80},
	{0xC4, 0x80},
	{0xC5, 0x80},
	{0xD8, 0x00},
	{0xD9, 0x00},
	{0xDA, 0x00},
	{0xDB, 0x00},
	{0xDC, 0x00},
	{0xDD, 0x00},
	{0xDE, 0x00},
	{0xDF, 0x00},
	{0xE0, 0x01},
	{0xE1, 0x01},
	{0xE2, 0x03},
	{0xE3, 0x03},
	{0xF0, 0x00},
	{0xF3, 0x00},
	{0xF4, 0x00},
	{0xF5, 0x00},
	{0xF6, 0x00},
	{0xF7, 0x00},
	{0xF8, 0x00},
	{0xF9, 0x00},
	{0xFA, 0x00},
	{0xFB, 0x00},
	{0xFC, 0x00},
	{0xFD, 0x00},
	{0xFE, 0x00},
	{0xF1, 0x01},
	{0xEF, 0x05},
	{0x3B, 0x00},//R_Cmd_Gated_MIPI_Clk=0
	{0xED, 0x01},
	{0xEF, 0x01},
	{0x02, 0xFB},//ResetTG
	{0x09, 0x01},
	{0xEF, 0x00},
	{0x11, 0x00},
	{0xEF, 0x05},
	{0x0F, 0x01},//MIPI CSI enable
	{0xED, 0x01},
	{REG_NULL_XC530, 0x00},
};

static const struct regval xc7080_2592x1944_regs[] = {
	// XC7080 setting
	//init
	{0xfffd, 0x80},
	{0xfffe, 0x50},
	{0x001c, 0xff},
	{0x001d, 0xff},
	{0x001e, 0xff},
	{0x001f, 0xff},
	{0x0018, 0x00},
	{0x0019, 0x00},
	{0x001a, 0x00},
	{0x001b, 0x00},
	{0xfffe, 0x50},
	{0x0050, 0x03},
	{0x0054, 0x03},
	{0x0058, 0x02},
	{0x0030, 0x44},  //44,
	{0x0031, 0x58},  //58,
	{0x0032, 0x34},  //34,
	{0x0033, 0x70},  //70,
	{0x0020, 0x02},  //0x05   02
	{0x0021, 0x0d},
	{0x0022, 0x02},
	{0x0023, 0x85},
	{0x0024, 0x0A},  //0x02 01
	{0x0025, 0x00},  //0x0d
	{0x0026, 0x01},
	{0x0027, 0x06},
	{0x0028, 0x01},
	{0x0029, 0x00},
	{0x002a, 0x02},
	{0x002b, 0x05},
	{0x002e, 0x05},
	{0x00bc, 0x19},
	{0x0090, 0x38},
	//datapath
	{0xfffe, 0x26},
	{0x8000, 0x1d},
	{0x8001, 0x20},
	{0x8002, 0x0A},
	{0x8003, 0x98},
	{0x8004, 0x07},
	{0x8010, 0x04},
	{0x8012, 0x20},
	{0x8013, 0x0A},
	{0x8014, 0x98},
	{0x8015, 0x07},
	{0x8016, 0x00},
	{0x8017, 0x00},
	{0x8018, 0x00},
	{0x8019, 0x00},
	{0xfffe, 0x30},
	{0x1900, 0x00},
	{0x1901, 0x00},
	{0x1902, 0x00},
	{0x1903, 0x00},
	{0x1904, 0x0A},
	{0x1905, 0x20},
	{0x1906, 0x07},
	{0x1907, 0x98},
	{0x1908, 0x00},
	{0xfffe, 0x2c},
	{0x0000, 0x00},
	{0x0001, 0x0A},
	{0x0002, 0x20},
	{0x0004, 0x07},
	{0x0005, 0x98},
	{0x0008, 0x10},
	{0x0044, 0x08},
	{0x0045, 0x04},
	{0x0048, 0x14},
	{0x0049, 0x30},
	{0xfffe, 0x26},
	{0x2019, 0x0A},
	{0x201a, 0x20},
	{0x201b, 0x07},
	{0x201c, 0x98},
	{0x201d, 0x00},
	{0x201e, 0x00},
	{0x201f, 0x00},
	{0x2020, 0x00},
	{0x2015, 0x81},
	{0x2017, 0x1E},
	{0x2018, 0x00},
	{0x2023, 0x03},
	{0x0000, 0x20},
	{0x0009, 0xc4},
	{0xfffe, 0x30},
	{0x0000, 0x01},
	{0x0001, 0x00},
	{0x0002, 0x10},
	{0x0003, 0x20},
	{0x0004, 0x10},
	{0x0050, 0x20},
	{0x0019, 0x08},
	{0x005e, 0x1F},
	{0x005f, 0x0A},
	{0x0060, 0x97},
	{0x0061, 0x07},
	{0x0064, 0x20},
	{0x0065, 0x0A},
	{0x0066, 0x98},
	{0x0067, 0x07},
	{0x0006, 0x0A},
	{0x0007, 0x20},
	{0x0008, 0x07},
	{0x0009, 0x98},
	{0x000a, 0x0A},
	{0x000b, 0x20},
	{0x000c, 0x07},
	{0x000d, 0x98},
	{0x0715, 0x04},
	{0xfffe, 0x50},
	{0x0226, 0x02},
	//retiming
	{0xfffe, 0x2e},
	{0x0000, 0x42},
	{0x0001, 0xee},
	{0x0003, 0x01},
	{0x0004, 0xe0},
	{0xfffe, 0x25},
	{0x0002, 0x80},
	{0xfffe, 0x30},
	{0x0004, 0x10},
	{0x2300, 0x0b},
	{0xfffe, 0x30},
	{0x1a0f, 0x5a},
	{0xfffd, 0x80},
	{0xfffe, 0x50},
	{0x000e, 0x54},
	{0xfffe, 0x14},
	{0x0006, 0x09},
	{0x0007, 0x44},
	{0x0014, 0x00},
	{0x0015, 0x14},
	{0x0016, 0x26},
	{0x0017, 0x94},
	{0x2114, 0x1b},
	{0x2115, 0x23},
	{0x2116, 0x2a},
	{0x2117, 0x2c},
	{0x2118, 0x37},
	{0x2119, 0x44},
	{0x211a, 0x3c},
	{0x211b, 0x4d},
	{0x211c, 0x5d},
	{0x211d, 0x4b},
	{0x211e, 0x5f},
	{0x211f, 0x74},
	{0x2120, 0x5a},
	{0x2121, 0x70},
	{0x2122, 0x87},
	{0x2123, 0x64},
	{0x2124, 0x7d},
	{0x2125, 0x96},
	{0x2126, 0x6c},
	{0x2127, 0x87},
	{0x2128, 0xa2},
	{0x2129, 0x71},
	{0x212a, 0x8e},
	{0x212b, 0xab},
	{0x212c, 0x75},
	{0x212d, 0x93},
	{0x212e, 0xb1},
	{0x212f, 0x78},
	{0x2130, 0x97},
	{0x2131, 0xb6},
	{0x2132, 0x7a},
	{0x2133, 0x9a},
	{0x2134, 0xb9},
	{0x2135, 0x7d},
	{0x2136, 0x9c},
	{0x2137, 0xbc},
	{0x2138, 0x7d},
	{0x2139, 0x9e},
	{0x213a, 0xbd},
	{0x213b, 0x7e},
	{0x213c, 0x9f},
	{0x213d, 0xbf},
	{0x213e, 0x7f},
	{0x213f, 0xa0},
	{0x2140, 0xc0},
	{0x2141, 0x7f},
	{0x2142, 0xa0},
	{0x2143, 0xc0},
	{0x2144, 0x00},
	{0x2145, 0x00},
	{0x2146, 0x04},
	{0x2147, 0x00},
	{0x2148, 0x00},
	{0x2149, 0x00},
	{0x214a, 0x00},
	{0x214b, 0x68},
	{0x214c, 0x00},
	{0x214d, 0x00},
	{0x214e, 0x20},
	{0x214f, 0x00},
	{0x2150, 0x00},
	{0x2151, 0x00},
	{0x2152, 0x00},
	{0x2153, 0x67},
	{0x2154, 0x00},
	{0x2155, 0x00},
	{0x2156, 0x30},
	{0x2157, 0x00},
	{0x2158, 0x00},
	{0x2159, 0x00},
	{0x215a, 0x00},
	{0x215b, 0x66},
	{0x215c, 0x00},
	{0x215d, 0x00},
	{0x215e, 0x37},
	{0x215f, 0xd0},
	{0x2160, 0x00},
	{0x2161, 0x00},
	{0x2162, 0x00},
	{0x2163, 0x65},
	{0x2164, 0x00},
	{0x2165, 0x00},
	{0x2166, 0x40},
	{0x2167, 0x00},
	{0x2168, 0x00},
	{0x2169, 0x00},
	{0x216a, 0x00},
	{0x216b, 0x55},
	{0x216c, 0x00},
	{0x216d, 0x00},
	{0x216e, 0x58},
	{0x216f, 0x00},
	{0x2170, 0x00},
	{0x2171, 0x00},
	{0x2172, 0x00},
	{0x2173, 0x54},
	{0x2174, 0x00},
	{0x2175, 0x00},
	{0x2176, 0x60},
	{0x2177, 0x00},
	{0x2178, 0x00},
	{0x2179, 0x00},
	{0x217a, 0x00},
	{0x217b, 0x44},
	{0x217c, 0x00},
	{0x217d, 0x00},
	{0x217e, 0x6c},
	{0x217f, 0x00},
	{0x2180, 0x00},
	{0x2181, 0x00},
	{0x2182, 0x00},
	{0x2183, 0x43},
	{0x2184, 0x00},
	{0x2185, 0x00},
	{0x2186, 0x73},
	{0x2187, 0x08},
	{0x2188, 0x00},
	{0x2189, 0x00},
	{0x218a, 0x00},
	{0x218b, 0x33},
	{0x218c, 0x00},
	{0x218d, 0x00},
	{0x218e, 0x7a},
	{0x218f, 0xd8},
	{0x2190, 0x00},
	{0x2191, 0x00},
	{0x2192, 0x00},
	{0x2193, 0x32},
	{0x2194, 0x00},
	{0x2195, 0x00},
	{0x2196, 0x80},
	{0x2197, 0x00},
	{0x2198, 0x00},
	{0x2199, 0x00},
	{0x219a, 0x00},
	{0x219b, 0x22},
	{0x219c, 0x9c},
	{0x219d, 0x21},
	{0x219e, 0xff},
	{0x219f, 0xf0},
	{0x21a0, 0xd4},
	{0x21a1, 0x01},
	{0x21a2, 0x48},
	{0x21a3, 0x00},
	{0x21a4, 0xd4},
	{0x21a5, 0x01},
	{0x21a6, 0x50},
	{0x21a7, 0x04},
	{0x21a8, 0xd4},
	{0x21a9, 0x01},
	{0x21aa, 0x60},
	{0x21ab, 0x08},
	{0x21ac, 0xd4},
	{0x21ad, 0x01},
	{0x21ae, 0x70},
	{0x21af, 0x0c},
	{0x21b0, 0x07},
	{0x21b1, 0xfb},
	{0x21b2, 0x07},
	{0x21b3, 0x82},
	{0x21b4, 0x15},
	{0x21b5, 0x00},
	{0x21b6, 0x00},
	{0x21b7, 0x00},
	{0x21b8, 0x85},
	{0x21b9, 0xcb},
	{0x21ba, 0x00},
	{0x21bb, 0x08},
	{0x21bc, 0x84},
	{0x21bd, 0x8e},
	{0x21be, 0x01},
	{0x21bf, 0x0c},
	{0x21c0, 0xb8},
	{0x21c1, 0x64},
	{0x21c2, 0x00},
	{0x21c3, 0x03},
	{0x21c4, 0xbc},
	{0x21c5, 0x43},
	{0x21c6, 0x02},
	{0x21c7, 0x00},
	{0x21c8, 0x10},
	{0x21c9, 0x00},
	{0x21ca, 0x00},
	{0x21cb, 0x0d},
	{0x21cc, 0xa9},
	{0x21cd, 0x8b},
	{0x21ce, 0x00},
	{0x21cf, 0x00},
	{0x21d0, 0xb8},
	{0x21d1, 0x64},
	{0x21d2, 0x00},
	{0x21d3, 0x07},
	{0x21d4, 0xb8},
	{0x21d5, 0x63},
	{0x21d6, 0x00},
	{0x21d7, 0x48},
	{0x21d8, 0x9c},
	{0x21d9, 0x63},
	{0x21da, 0xff},
	{0x21db, 0xf0},
	{0x21dc, 0xbc},
	{0x21dd, 0xa3},
	{0x21de, 0x00},
	{0x21df, 0x07},
	{0x21e0, 0x10},
	{0x21e1, 0x00},
	{0x21e2, 0x00},
	{0x21e3, 0x1c},
	{0x21e4, 0x9d},
	{0x21e5, 0x40},
	{0x21e6, 0x00},
	{0x21e7, 0x08},
	{0x21e8, 0xbc},
	{0x21e9, 0xa3},
	{0x21ea, 0x00},
	{0x21eb, 0x50},
	{0x21ec, 0x10},
	{0x21ed, 0x00},
	{0x21ee, 0x00},
	{0x21ef, 0x19},
	{0x21f0, 0xa9},
	{0x21f1, 0x43},
	{0x21f2, 0x00},
	{0x21f3, 0x00},
	{0x21f4, 0x00},
	{0x21f5, 0x00},
	{0x21f6, 0x00},
	{0x21f7, 0x17},
	{0x21f8, 0x9d},
	{0x21f9, 0x40},
	{0x21fa, 0x00},
	{0x21fb, 0x50},
	{0x21fc, 0xbc},
	{0x21fd, 0x43},
	{0x21fe, 0x04},
	{0x21ff, 0x00},
	{0x2200, 0x10},
	{0x2201, 0x00},
	{0x2202, 0x00},
	{0x2203, 0x05},
	{0x2204, 0xbc},
	{0x2205, 0x43},
	{0x2206, 0x08},
	{0x2207, 0x00},
	{0x2208, 0xb8},
	{0x2209, 0x64},
	{0x220a, 0x00},
	{0x220b, 0x06},
	{0x220c, 0x03},
	{0x220d, 0xff},
	{0x220e, 0xff},
	{0x220f, 0xf4},
	{0x2210, 0xb8},
	{0x2211, 0x63},
	{0x2212, 0x00},
	{0x2213, 0x48},
	{0x2214, 0x0c},
	{0x2215, 0x00},
	{0x2216, 0x00},
	{0x2217, 0x08},
	{0x2218, 0xbc},
	{0x2219, 0x43},
	{0x221a, 0x10},
	{0x221b, 0x00},
	{0x221c, 0x10},
	{0x221d, 0x00},
	{0x221e, 0x00},
	{0x221f, 0x0a},
	{0x2220, 0x15},
	{0x2221, 0x00},
	{0x2222, 0x00},
	{0x2223, 0x00},
	{0x2224, 0xe0},
	{0x2225, 0x63},
	{0x2226, 0x18},
	{0x2227, 0x00},
	{0x2228, 0xb8},
	{0x2229, 0x63},
	{0x222a, 0x00},
	{0x222b, 0x48},
	{0x222c, 0x03},
	{0x222d, 0xff},
	{0x222e, 0xff},
	{0x222f, 0xec},
	{0x2230, 0x9c},
	{0x2231, 0x63},
	{0x2232, 0x00},
	{0x2233, 0x20},
	{0x2234, 0xb8},
	{0x2235, 0x64},
	{0x2236, 0x00},
	{0x2237, 0x05},
	{0x2238, 0xb8},
	{0x2239, 0x63},
	{0x223a, 0x00},
	{0x223b, 0x48},
	{0x223c, 0x03},
	{0x223d, 0xff},
	{0x223e, 0xff},
	{0x223f, 0xe8},
	{0x2240, 0x9c},
	{0x2241, 0x63},
	{0x2242, 0x00},
	{0x2243, 0x10},
	{0x2244, 0xb8},
	{0x2245, 0x63},
	{0x2246, 0x00},
	{0x2247, 0x48},
	{0x2248, 0x03},
	{0x2249, 0xff},
	{0x224a, 0xff},
	{0x224b, 0xe5},
	{0x224c, 0x9c},
	{0x224d, 0x63},
	{0x224e, 0x00},
	{0x224f, 0x30},
	{0x2250, 0xa8},
	{0x2251, 0x6c},
	{0x2252, 0x00},
	{0x2253, 0x00},
	{0x2254, 0x9c},
	{0x2255, 0x80},
	{0x2256, 0x00},
	{0x2257, 0xef},
	{0x2258, 0x07},
	{0x2259, 0xfb},
	{0x225a, 0x06},
	{0x225b, 0xab},
	{0x225c, 0x9c},
	{0x225d, 0xa0},
	{0x225e, 0x00},
	{0x225f, 0x01},
	{0x2260, 0x94},
	{0x2261, 0x8e},
	{0x2262, 0x00},
	{0x2263, 0xe8},
	{0x2264, 0xa8},
	{0x2265, 0x6c},
	{0x2266, 0x00},
	{0x2267, 0x00},
	{0x2268, 0x07},
	{0x2269, 0xfb},
	{0x226a, 0x06},
	{0x226b, 0xa7},
	{0x226c, 0xa4},
	{0x226d, 0xaa},
	{0x226e, 0x00},
	{0x226f, 0xff},
	{0x2270, 0xa8},
	{0x2271, 0x6c},
	{0x2272, 0x00},
	{0x2273, 0x00},
	{0x2274, 0x9c},
	{0x2275, 0x80},
	{0x2276, 0x00},
	{0x2277, 0x09},
	{0x2278, 0x07},
	{0x2279, 0xfb},
	{0x227a, 0x06},
	{0x227b, 0xa3},
	{0x227c, 0x9c},
	{0x227d, 0xa0},
	{0x227e, 0x00},
	{0x227f, 0x01},
	{0x2280, 0x85},
	{0x2281, 0x21},
	{0x2282, 0x00},
	{0x2283, 0x00},
	{0x2284, 0x85},
	{0x2285, 0x41},
	{0x2286, 0x00},
	{0x2287, 0x04},
	{0x2288, 0x85},
	{0x2289, 0x81},
	{0x228a, 0x00},
	{0x228b, 0x08},
	{0x228c, 0x85},
	{0x228d, 0xc1},
	{0x228e, 0x00},
	{0x228f, 0x0c},
	{0x2290, 0x44},
	{0x2291, 0x00},
	{0x2292, 0x48},
	{0x2293, 0x00},
	{0x2294, 0x9c},
	{0x2295, 0x21},
	{0x2296, 0x00},
	{0x2297, 0x10},
	{0x2298, 0x9c},
	{0x2299, 0x21},
	{0x229a, 0xff},
	{0x229b, 0xf0},
	{0x229c, 0xd4},
	{0x229d, 0x01},
	{0x229e, 0x48},
	{0x229f, 0x00},
	{0x22a0, 0xd4},
	{0x22a1, 0x01},
	{0x22a2, 0x50},
	{0x22a3, 0x04},
	{0x22a4, 0xd4},
	{0x22a5, 0x01},
	{0x22a6, 0x60},
	{0x22a7, 0x08},
	{0x22a8, 0xd4},
	{0x22a9, 0x01},
	{0x22aa, 0x70},
	{0x22ab, 0x0c},
	{0x22ac, 0x07},
	{0x22ad, 0xfb},
	{0x22ae, 0x07},
	{0x22af, 0x43},
	{0x22b0, 0x15},
	{0x22b1, 0x00},
	{0x22b2, 0x00},
	{0x22b3, 0x00},
	{0x22b4, 0x85},
	{0x22b5, 0xcb},
	{0x22b6, 0x00},
	{0x22b7, 0x08},
	{0x22b8, 0x18},
	{0x22b9, 0xc0},
	{0x22ba, 0x00},
	{0x22bb, 0x14},
	{0x22bc, 0xa8},
	{0x22bd, 0xc6},
	{0x22be, 0x5d},
	{0x22bf, 0x00},
	{0x22c0, 0x84},
	{0x22c1, 0xee},
	{0x22c2, 0x01},
	{0x22c3, 0x08},
	{0x22c4, 0x94},
	{0x22c5, 0xc6},
	{0x22c6, 0x00},
	{0x22c7, 0x00},
	{0x22c8, 0xb8},
	{0x22c9, 0xe7},
	{0x22ca, 0x00},
	{0x22cb, 0x44},
	{0x22cc, 0xa9},
	{0x22cd, 0x8b},
	{0x22ce, 0x00},
	{0x22cf, 0x00},
	{0x22d0, 0x9d},
	{0x22d1, 0x06},
	{0x22d2, 0xff},
	{0x22d3, 0xfe},
	{0x22d4, 0xe0},
	{0x22d5, 0xe6},
	{0x22d6, 0x38},
	{0x22d7, 0x02},
	{0x22d8, 0x9c},
	{0x22d9, 0x80},
	{0x22da, 0x00},
	{0x22db, 0xef},
	{0x22dc, 0xa4},
	{0x22dd, 0xc6},
	{0x22de, 0xff},
	{0x22df, 0xff},
	{0x22e0, 0xa5},
	{0x22e1, 0x47},
	{0x22e2, 0xff},
	{0x22e3, 0xff},
	{0x22e4, 0xa8},
	{0x22e5, 0x6b},
	{0x22e6, 0x00},
	{0x22e7, 0x00},
	{0x22e8, 0x9c},
	{0x22e9, 0xa0},
	{0x22ea, 0x00},
	{0x22eb, 0x01},
	{0x22ec, 0xbc},
	{0x22ed, 0x4a},
	{0x22ee, 0x00},
	{0x22ef, 0x03},
	{0x22f0, 0x10},
	{0x22f1, 0x00},
	{0x22f2, 0x00},
	{0x22f3, 0x03},
	{0x22f4, 0x9c},
	{0x22f5, 0xc6},
	{0x22f6, 0xff},
	{0x22f7, 0xfe},
	{0x22f8, 0x9d},
	{0x22f9, 0x40},
	{0x22fa, 0x00},
	{0x22fb, 0x04},
	{0x22fc, 0xe5},
	{0x22fd, 0xaa},
	{0x22fe, 0x30},
	{0x22ff, 0x00},
	{0x2300, 0x10},
	{0x2301, 0x00},
	{0x2302, 0x00},
	{0x2303, 0x03},
	{0x2304, 0x15},
	{0x2305, 0x00},
	{0x2306, 0x00},
	{0x2307, 0x00},
	{0x2308, 0xa5},
	{0x2309, 0x48},
	{0x230a, 0xff},
	{0x230b, 0xff},
	{0x230c, 0x07},
	{0x230d, 0xfb},
	{0x230e, 0x06},
	{0x230f, 0x7e},
	{0x2310, 0x15},
	{0x2311, 0x00},
	{0x2312, 0x00},
	{0x2313, 0x00},
	{0x2314, 0x94},
	{0x2315, 0x8e},
	{0x2316, 0x00},
	{0x2317, 0xc8},
	{0x2318, 0xb8},
	{0x2319, 0xaa},
	{0x231a, 0x00},
	{0x231b, 0x48},
	{0x231c, 0x07},
	{0x231d, 0xfb},
	{0x231e, 0x06},
	{0x231f, 0x7a},
	{0x2320, 0xa8},
	{0x2321, 0x6c},
	{0x2322, 0x00},
	{0x2323, 0x00},
	{0x2324, 0x94},
	{0x2325, 0x8e},
	{0x2326, 0x00},
	{0x2327, 0xca},
	{0x2328, 0xa8},
	{0x2329, 0x6c},
	{0x232a, 0x00},
	{0x232b, 0x00},
	{0x232c, 0x07},
	{0x232d, 0xfb},
	{0x232e, 0x06},
	{0x232f, 0x76},
	{0x2330, 0xa4},
	{0x2331, 0xaa},
	{0x2332, 0x00},
	{0x2333, 0xff},
	{0x2334, 0xa8},
	{0x2335, 0x6c},
	{0x2336, 0x00},
	{0x2337, 0x00},
	{0x2338, 0x9c},
	{0x2339, 0x80},
	{0x233a, 0x00},
	{0x233b, 0x09},
	{0x233c, 0x07},
	{0x233d, 0xfb},
	{0x233e, 0x06},
	{0x233f, 0x72},
	{0x2340, 0x9c},
	{0x2341, 0xa0},
	{0x2342, 0x00},
	{0x2343, 0x01},
	{0x2344, 0x85},
	{0x2345, 0x21},
	{0x2346, 0x00},
	{0x2347, 0x00},
	{0x2348, 0x85},
	{0x2349, 0x41},
	{0x234a, 0x00},
	{0x234b, 0x04},
	{0x234c, 0x85},
	{0x234d, 0x81},
	{0x234e, 0x00},
	{0x234f, 0x08},
	{0x2350, 0x85},
	{0x2351, 0xc1},
	{0x2352, 0x00},
	{0x2353, 0x0c},
	{0x2354, 0x44},
	{0x2355, 0x00},
	{0x2356, 0x48},
	{0x2357, 0x00},
	{0x2358, 0x9c},
	{0x2359, 0x21},
	{0x235a, 0x00},
	{0x235b, 0x10},
	{0x235c, 0x9c},
	{0x235d, 0x21},
	{0x235e, 0xff},
	{0x235f, 0x58},
	{0x2360, 0xd4},
	{0x2361, 0x01},
	{0x2362, 0x48},
	{0x2363, 0x00},
	{0x2364, 0xd4},
	{0x2365, 0x01},
	{0x2366, 0x50},
	{0x2367, 0x04},
	{0x2368, 0xd4},
	{0x2369, 0x01},
	{0x236a, 0x60},
	{0x236b, 0x08},
	{0x236c, 0xd4},
	{0x236d, 0x01},
	{0x236e, 0x70},
	{0x236f, 0x0c},
	{0x2370, 0xd4},
	{0x2371, 0x01},
	{0x2372, 0x80},
	{0x2373, 0x10},
	{0x2374, 0xd4},
	{0x2375, 0x01},
	{0x2376, 0x90},
	{0x2377, 0x14},
	{0x2378, 0xd4},
	{0x2379, 0x01},
	{0x237a, 0xa0},
	{0x237b, 0x18},
	{0x237c, 0xd4},
	{0x237d, 0x01},
	{0x237e, 0xb0},
	{0x237f, 0x1c},
	{0x2380, 0x07},
	{0x2381, 0xfb},
	{0x2382, 0x07},
	{0x2383, 0x0e},
	{0x2384, 0x9e},
	{0x2385, 0x80},
	{0x2386, 0x00},
	{0x2387, 0x00},
	{0x2388, 0xa9},
	{0x2389, 0xcb},
	{0x238a, 0x00},
	{0x238b, 0x00},
	{0x238c, 0x85},
	{0x238d, 0x8b},
	{0x238e, 0x00},
	{0x238f, 0x08},
	{0x2390, 0x9c},
	{0x2391, 0x61},
	{0x2392, 0x00},
	{0x2393, 0x78},
	{0x2394, 0x18},
	{0x2395, 0x80},
	{0x2396, 0x00},
	{0x2397, 0x14},
	{0x2398, 0xa8},
	{0x2399, 0x84},
	{0x239a, 0x21},
	{0x239b, 0x14},
	{0x239c, 0x07},
	{0x239d, 0xfb},
	{0x239e, 0x1e},
	{0x239f, 0x1f},
	{0x23a0, 0x9c},
	{0x23a1, 0xa0},
	{0x23a2, 0x00},
	{0x23a3, 0x30},
	{0x23a4, 0x9c},
	{0x23a5, 0xa0},
	{0x23a6, 0x00},
	{0x23a7, 0x58},
	{0x23a8, 0x9c},
	{0x23a9, 0x61},
	{0x23aa, 0x00},
	{0x23ab, 0x20},
	{0x23ac, 0x18},
	{0x23ad, 0x80},
	{0x23ae, 0x00},
	{0x23af, 0x14},
	{0x23b0, 0xa8},
	{0x23b1, 0x84},
	{0x23b2, 0x21},
	{0x23b3, 0x44},
	{0x23b4, 0x07},
	{0x23b5, 0xfb},
	{0x23b6, 0x1e},
	{0x23b7, 0x19},
	{0x23b8, 0xaa},
	{0x23b9, 0xd4},
	{0x23ba, 0x00},
	{0x23bb, 0x00},
	{0x23bc, 0xa8},
	{0x23bd, 0x6e},
	{0x23be, 0x00},
	{0x23bf, 0x00},
	{0x23c0, 0x9c},
	{0x23c1, 0x80},
	{0x23c2, 0x00},
	{0x23c3, 0x1b},
	{0x23c4, 0x07},
	{0x23c5, 0xfb},
	{0x23c6, 0x06},
	{0x23c7, 0x80},
	{0x23c8, 0xaa},
	{0x23c9, 0x54},
	{0x23ca, 0x00},
	{0x23cb, 0x00},
	{0x23cc, 0xa5},
	{0x23cd, 0x6b},
	{0x23ce, 0x00},
	{0x23cf, 0x0f},
	{0x23d0, 0xa8},
	{0x23d1, 0x6e},
	{0x23d2, 0x00},
	{0x23d3, 0x00},
	{0x23d4, 0xb9},
	{0x23d5, 0x4b},
	{0x23d6, 0x00},
	{0x23d7, 0x08},
	{0x23d8, 0x07},
	{0x23d9, 0xfb},
	{0x23da, 0x06},
	{0x23db, 0x7b},
	{0x23dc, 0x9c},
	{0x23dd, 0x80},
	{0x23de, 0x00},
	{0x23df, 0x1c},
	{0x23e0, 0x84},
	{0x23e1, 0x6c},
	{0x23e2, 0x01},
	{0x23e3, 0x0c},
	{0x23e4, 0xe2},
	{0x23e5, 0x0a},
	{0x23e6, 0x58},
	{0x23e7, 0x04},
	{0x23e8, 0xb9},
	{0x23e9, 0x83},
	{0x23ea, 0x00},
	{0x23eb, 0x05},
	{0x23ec, 0x9c},
	{0x23ed, 0xa0},
	{0x23ee, 0x00},
	{0x23ef, 0x01},
	{0x23f0, 0x9c},
	{0x23f1, 0x80},
	{0x23f2, 0x00},
	{0x23f3, 0xef},
	{0x23f4, 0x07},
	{0x23f5, 0xfb},
	{0x23f6, 0x06},
	{0x23f7, 0x44},
	{0x23f8, 0xa8},
	{0x23f9, 0x6e},
	{0x23fa, 0x00},
	{0x23fb, 0x00},
	{0x23fc, 0xa8},
	{0x23fd, 0x6e},
	{0x23fe, 0x00},
	{0x23ff, 0x00},
	{0x2400, 0x07},
	{0x2401, 0xfb},
	{0x2402, 0x06},
	{0x2403, 0x71},
	{0x2404, 0x9c},
	{0x2405, 0x80},
	{0x2406, 0x00},
	{0x2407, 0x1b},
	{0x2408, 0x18},
	{0x2409, 0xa0},
	{0x240a, 0x00},
	{0x240b, 0x14},
	{0x240c, 0xa8},
	{0x240d, 0xa5},
	{0x240e, 0x5d},
	{0x240f, 0x08},
	{0x2410, 0xa5},
	{0x2411, 0x4b},
	{0x2412, 0xff},
	{0x2413, 0xff},
	{0x2414, 0x94},
	{0x2415, 0x65},
	{0x2416, 0x00},
	{0x2417, 0x00},
	{0x2418, 0xe4},
	{0x2419, 0x03},
	{0x241a, 0x50},
	{0x241b, 0x00},
	{0x241c, 0x10},
	{0x241d, 0x00},
	{0x241e, 0x00},
	{0x241f, 0x15},
	{0x2420, 0xb8},
	{0x2421, 0x6a},
	{0x2422, 0x00},
	{0x2423, 0x47},
	{0x2424, 0xa4},
	{0x2425, 0x8a},
	{0x2426, 0x00},
	{0x2427, 0xf0},
	{0x2428, 0xdc},
	{0x2429, 0x05},
	{0x242a, 0x50},
	{0x242b, 0x00},
	{0x242c, 0xa5},
	{0x242d, 0x63},
	{0x242e, 0x00},
	{0x242f, 0x01},
	{0x2430, 0xb8},
	{0x2431, 0x84},
	{0x2432, 0x00},
	{0x2433, 0x08},
	{0x2434, 0xe1},
	{0x2435, 0x70},
	{0x2436, 0x58},
	{0x2437, 0x02},
	{0x2438, 0xa8},
	{0x2439, 0x6e},
	{0x243a, 0x00},
	{0x243b, 0x00},
	{0x243c, 0xe0},
	{0x243d, 0xab},
	{0x243e, 0x20},
	{0x243f, 0x04},
	{0x2440, 0xa5},
	{0x2441, 0x45},
	{0x2442, 0xff},
	{0x2443, 0xff},
	{0x2444, 0x9c},
	{0x2445, 0x80},
	{0x2446, 0x00},
	{0x2447, 0x1b},
	{0x2448, 0x07},
	{0x2449, 0xfb},
	{0x244a, 0x06},
	{0x244b, 0x2f},
	{0x244c, 0xb8},
	{0x244d, 0xaa},
	{0x244e, 0x00},
	{0x244f, 0x48},
	{0x2450, 0xa8},
	{0x2451, 0x6e},
	{0x2452, 0x00},
	{0x2453, 0x00},
	{0x2454, 0xa4},
	{0x2455, 0xaa},
	{0x2456, 0x00},
	{0x2457, 0xff},
	{0x2458, 0x07},
	{0x2459, 0xfb},
	{0x245a, 0x06},
	{0x245b, 0x2b},
	{0x245c, 0x9c},
	{0x245d, 0x80},
	{0x245e, 0x00},
	{0x245f, 0x1c},
	{0x2460, 0xa8},
	{0x2461, 0x6e},
	{0x2462, 0x00},
	{0x2463, 0x00},
	{0x2464, 0x9c},
	{0x2465, 0x80},
	{0x2466, 0x00},
	{0x2467, 0x09},
	{0x2468, 0x07},
	{0x2469, 0xfb},
	{0x246a, 0x06},
	{0x246b, 0x27},
	{0x246c, 0x9c},
	{0x246d, 0xa0},
	{0x246e, 0x00},
	{0x246f, 0x01},
	{0x2470, 0x18},
	{0x2471, 0xe0},
	{0x2472, 0x00},
	{0x2473, 0x14},
	{0x2474, 0xa8},
	{0x2475, 0xe7},
	{0x2476, 0x5d},
	{0x2477, 0x04},
	{0x2478, 0x84},
	{0x2479, 0x67},
	{0x247a, 0x00},
	{0x247b, 0x00},
	{0x247c, 0xe4},
	{0x247d, 0x03},
	{0x247e, 0x60},
	{0x247f, 0x00},
	{0x2480, 0x10},
	{0x2481, 0x00},
	{0x2482, 0x00},
	{0x2483, 0x3e},
	{0x2484, 0x84},
	{0x2485, 0x61},
	{0x2486, 0x00},
	{0x2487, 0x24},
	{0x2488, 0x18},
	{0x2489, 0xa0},
	{0x248a, 0x00},
	{0x248b, 0x14},
	{0x248c, 0xa8},
	{0x248d, 0xa5},
	{0x248e, 0x5d},
	{0x248f, 0x02},
	{0x2490, 0xd8},
	{0x2491, 0x05},
	{0x2492, 0xa0},
	{0x2493, 0x00},
	{0x2494, 0xa8},
	{0x2495, 0xc5},
	{0x2496, 0x00},
	{0x2497, 0x00},
	{0x2498, 0x8c},
	{0x2499, 0x66},
	{0x249a, 0x00},
	{0x249b, 0x00},
	{0x249c, 0x9d},
	{0x249d, 0x01},
	{0x249e, 0x00},
	{0x249f, 0xa8},
	{0x24a0, 0x9c},
	{0x24a1, 0x83},
	{0x24a2, 0x00},
	{0x24a3, 0x01},
	{0x24a4, 0xb8},
	{0x24a5, 0x63},
	{0x24a6, 0x00},
	{0x24a7, 0x03},
	{0x24a8, 0xe0},
	{0x24a9, 0x63},
	{0x24aa, 0x40},
	{0x24ab, 0x00},
	{0x24ac, 0x84},
	{0x24ad, 0x63},
	{0x24ae, 0xff},
	{0x24af, 0x78},
	{0x24b0, 0xe4},
	{0x24b1, 0x83},
	{0x24b2, 0x60},
	{0x24b3, 0x00},
	{0x24b4, 0x0c},
	{0x24b5, 0x00},
	{0x24b6, 0x00},
	{0x24b7, 0x07},
	{0x24b8, 0x15},
	{0x24b9, 0x00},
	{0x24ba, 0x00},
	{0x24bb, 0x00},
	{0x24bc, 0xd8},
	{0x24bd, 0x06},
	{0x24be, 0x20},
	{0x24bf, 0x00},
	{0x24c0, 0x8c},
	{0x24c1, 0x65},
	{0x24c2, 0x00},
	{0x24c3, 0x00},
	{0x24c4, 0xbc},
	{0x24c5, 0x63},
	{0x24c6, 0x00},
	{0x24c7, 0x0b},
	{0x24c8, 0x0f},
	{0x24c9, 0xff},
	{0x24ca, 0xff},
	{0x24cb, 0xf4},
	{0x24cc, 0x15},
	{0x24cd, 0x00},
	{0x24ce, 0x00},
	{0x24cf, 0x00},
	{0x24d0, 0xd4},
	{0x24d1, 0x07},
	{0x24d2, 0x60},
	{0x24d3, 0x00},
	{0x24d4, 0x18},
	{0x24d5, 0xe0},
	{0x24d6, 0x00},
	{0x24d7, 0x14},
	{0x24d8, 0xa8},
	{0x24d9, 0xe7},
	{0x24da, 0x5d},
	{0x24db, 0x03},
	{0x24dc, 0x8c},
	{0x24dd, 0x67},
	{0x24de, 0x00},
	{0x24df, 0x00},
	{0x24e0, 0xe4},
	{0x24e1, 0x03},
	{0x24e2, 0x90},
	{0x24e3, 0x00},
	{0x24e4, 0x10},
	{0x24e5, 0x00},
	{0x24e6, 0x00},
	{0x24e7, 0x11},
	{0x24e8, 0xa8},
	{0x24e9, 0x6e},
	{0x24ea, 0x00},
	{0x24eb, 0x00},
	{0x24ec, 0x9c},
	{0x24ed, 0x80},
	{0x24ee, 0x00},
	{0x24ef, 0xef},
	{0x24f0, 0x9c},
	{0x24f1, 0xa0},
	{0x24f2, 0x00},
	{0x24f3, 0x06},
	{0x24f4, 0x07},
	{0x24f5, 0xfb},
	{0x24f6, 0x06},
	{0x24f7, 0x04},
	{0x24f8, 0xd8},
	{0x24f9, 0x07},
	{0x24fa, 0x90},
	{0x24fb, 0x00},
	{0x24fc, 0xa8},
	{0x24fd, 0x6e},
	{0x24fe, 0x00},
	{0x24ff, 0x00},
	{0x2500, 0xa8},
	{0x2501, 0xb2},
	{0x2502, 0x00},
	{0x2503, 0x00},
	{0x2504, 0x9c},
	{0x2505, 0x80},
	{0x2506, 0x00},
	{0x2507, 0x99},
	{0x2508, 0x07},
	{0x2509, 0xfb},
	{0x250a, 0x05},
	{0x250b, 0xff},
	{0x250c, 0x15},
	{0x250d, 0x00},
	{0x250e, 0x00},
	{0x250f, 0x00},
	{0x2510, 0xa8},
	{0x2511, 0x6e},
	{0x2512, 0x00},
	{0x2513, 0x00},
	{0x2514, 0x9c},
	{0x2515, 0x80},
	{0x2516, 0x00},
	{0x2517, 0xf1},
	{0x2518, 0x07},
	{0x2519, 0xfb},
	{0x251a, 0x05},
	{0x251b, 0xfb},
	{0x251c, 0x9c},
	{0x251d, 0xa0},
	{0x251e, 0x00},
	{0x251f, 0x01},
	{0x2520, 0x00},
	{0x2521, 0x00},
	{0x2522, 0x00},
	{0x2523, 0x28},
	{0x2524, 0x15},
	{0x2525, 0x00},
	{0x2526, 0x00},
	{0x2527, 0x00},
	{0x2528, 0x18},
	{0x2529, 0x60},
	{0x252a, 0x00},
	{0x252b, 0x14},
	{0x252c, 0xa8},
	{0x252d, 0x63},
	{0x252e, 0x5d},
	{0x252f, 0x02},
	{0x2530, 0x8c},
	{0x2531, 0x63},
	{0x2532, 0x00},
	{0x2533, 0x00},
	{0x2534, 0xbc},
	{0x2535, 0xa3},
	{0x2536, 0x00},
	{0x2537, 0x00},
	{0x2538, 0x10},
	{0x2539, 0x00},
	{0x253a, 0x00},
	{0x253b, 0x04},
	{0x253c, 0xe4},
	{0x253d, 0x16},
	{0x253e, 0xa0},
	{0x253f, 0x00},
	{0x2540, 0x9e},
	{0x2541, 0x80},
	{0x2542, 0x00},
	{0x2543, 0x01},
	{0x2544, 0xe4},
	{0x2545, 0x16},
	{0x2546, 0xa0},
	{0x2547, 0x00},
	{0x2548, 0x10},
	{0x2549, 0x00},
	{0x254a, 0x00},
	{0x254b, 0x1e},
	{0x254c, 0xa8},
	{0x254d, 0x6e},
	{0x254e, 0x00},
	{0x254f, 0x00},
	{0x2550, 0x9c},
	{0x2551, 0x80},
	{0x2552, 0x00},
	{0x2553, 0xef},
	{0x2554, 0x07},
	{0x2555, 0xfb},
	{0x2556, 0x05},
	{0x2557, 0xec},
	{0x2558, 0x9c},
	{0x2559, 0xa0},
	{0x255a, 0x00},
	{0x255b, 0x06},
	{0x255c, 0xb8},
	{0x255d, 0xb4},
	{0x255e, 0x00},
	{0x255f, 0x03},
	{0x2560, 0x9c},
	{0x2561, 0xc1},
	{0x2562, 0x00},
	{0x2563, 0xa8},
	{0x2564, 0xa8},
	{0x2565, 0x6e},
	{0x2566, 0x00},
	{0x2567, 0x00},
	{0x2568, 0xe0},
	{0x2569, 0xa5},
	{0x256a, 0x30},
	{0x256b, 0x00},
	{0x256c, 0x9c},
	{0x256d, 0x80},
	{0x256e, 0x00},
	{0x256f, 0x9a},
	{0x2570, 0x03},
	{0x2571, 0xff},
	{0x2572, 0xff},
	{0x2573, 0xe6},
	{0x2574, 0x84},
	{0x2575, 0xa5},
	{0x2576, 0xff},
	{0x2577, 0x7c},
	{0x2578, 0x18},
	{0x2579, 0xe0},
	{0x257a, 0x00},
	{0x257b, 0x14},
	{0x257c, 0xa8},
	{0x257d, 0xe7},
	{0x257e, 0x5d},
	{0x257f, 0x03},
	{0x2580, 0xa4},
	{0x2581, 0x83},
	{0x2582, 0x00},
	{0x2583, 0x0f},
	{0x2584, 0x8c},
	{0x2585, 0xc7},
	{0x2586, 0x00},
	{0x2587, 0x00},
	{0x2588, 0xb8},
	{0x2589, 0x64},
	{0x258a, 0x00},
	{0x258b, 0x01},
	{0x258c, 0xe0},
	{0x258d, 0x63},
	{0x258e, 0x20},
	{0x258f, 0x00},
	{0x2590, 0x9c},
	{0x2591, 0x81},
	{0x2592, 0x00},
	{0x2593, 0xa8},
	{0x2594, 0xe0},
	{0x2595, 0xa3},
	{0x2596, 0x20},
	{0x2597, 0x00},
	{0x2598, 0x8e},
	{0x2599, 0x45},
	{0x259a, 0xff},
	{0x259b, 0xd1},
	{0x259c, 0xe4},
	{0x259d, 0xa6},
	{0x259e, 0x90},
	{0x259f, 0x00},
	{0x25a0, 0x0c},
	{0x25a1, 0x00},
	{0x25a2, 0x00},
	{0x25a3, 0x06},
	{0x25a4, 0xe4},
	{0x25a5, 0x66},
	{0x25a6, 0x90},
	{0x25a7, 0x00},
	{0x25a8, 0x13},
	{0x25a9, 0xff},
	{0x25aa, 0xff},
	{0x25ab, 0xcd},
	{0x25ac, 0x15},
	{0x25ad, 0x00},
	{0x25ae, 0x00},
	{0x25af, 0x00},
	{0x25b0, 0x03},
	{0x25b1, 0xff},
	{0x25b2, 0xff},
	{0x25b3, 0xcb},
	{0x25b4, 0x9e},
	{0x25b5, 0x52},
	{0x25b6, 0x00},
	{0x25b7, 0x01},
	{0x25b8, 0x03},
	{0x25b9, 0xff},
	{0x25ba, 0xff},
	{0x25bb, 0xc9},
	{0x25bc, 0x9e},
	{0x25bd, 0x52},
	{0x25be, 0xff},
	{0x25bf, 0xff},
	{0x25c0, 0x85},
	{0x25c1, 0x21},
	{0x25c2, 0x00},
	{0x25c3, 0x00},
	{0x25c4, 0x85},
	{0x25c5, 0x41},
	{0x25c6, 0x00},
	{0x25c7, 0x04},
	{0x25c8, 0x85},
	{0x25c9, 0x81},
	{0x25ca, 0x00},
	{0x25cb, 0x08},
	{0x25cc, 0x85},
	{0x25cd, 0xc1},
	{0x25ce, 0x00},
	{0x25cf, 0x0c},
	{0x25d0, 0x86},
	{0x25d1, 0x01},
	{0x25d2, 0x00},
	{0x25d3, 0x10},
	{0x25d4, 0x86},
	{0x25d5, 0x41},
	{0x25d6, 0x00},
	{0x25d7, 0x14},
	{0x25d8, 0x86},
	{0x25d9, 0x81},
	{0x25da, 0x00},
	{0x25db, 0x18},
	{0x25dc, 0x86},
	{0x25dd, 0xc1},
	{0x25de, 0x00},
	{0x25df, 0x1c},
	{0x25e0, 0x44},
	{0x25e1, 0x00},
	{0x25e2, 0x48},
	{0x25e3, 0x00},
	{0x25e4, 0x9c},
	{0x25e5, 0x21},
	{0x25e6, 0x00},
	{0x25e7, 0xa8},
	{0x25e8, 0x9c},
	{0x25e9, 0x21},
	{0x25ea, 0xff},
	{0x25eb, 0xf8},
	{0x25ec, 0xd4},
	{0x25ed, 0x01},
	{0x25ee, 0x48},
	{0x25ef, 0x00},
	{0x25f0, 0xd4},
	{0x25f1, 0x01},
	{0x25f2, 0x50},
	{0x25f3, 0x04},
	{0x25f4, 0x18},
	{0x25f5, 0x80},
	{0x25f6, 0x00},
	{0x25f7, 0x14},
	{0x25f8, 0xa8},
	{0x25f9, 0x84},
	{0x25fa, 0x20},
	{0x25fb, 0x1c},
	{0x25fc, 0x19},
	{0x25fd, 0x40},
	{0x25fe, 0x00},
	{0x25ff, 0x14},
	{0x2600, 0xa9},
	{0x2601, 0x4a},
	{0x2602, 0x5d},
	{0x2603, 0x0c},
	{0x2604, 0x84},
	{0x2605, 0x84},
	{0x2606, 0x00},
	{0x2607, 0x00},
	{0x2608, 0xbc},
	{0x2609, 0x03},
	{0x260a, 0x04},
	{0x260b, 0x0b},
	{0x260c, 0x84},
	{0x260d, 0x84},
	{0x260e, 0x00},
	{0x260f, 0x14},
	{0x2610, 0x10},
	{0x2611, 0x00},
	{0x2612, 0x00},
	{0x2613, 0x07},
	{0x2614, 0xd4},
	{0x2615, 0x0a},
	{0x2616, 0x20},
	{0x2617, 0x00},
	{0x2618, 0xbc},
	{0x2619, 0x03},
	{0x261a, 0x04},
	{0x261b, 0x0c},
	{0x261c, 0x0c},
	{0x261d, 0x00},
	{0x261e, 0x00},
	{0x261f, 0x1a},
	{0x2620, 0x15},
	{0x2621, 0x00},
	{0x2622, 0x00},
	{0x2623, 0x00},
	{0x2624, 0x00},
	{0x2625, 0x00},
	{0x2626, 0x00},
	{0x2627, 0x06},
	{0x2628, 0x15},
	{0x2629, 0x00},
	{0x262a, 0x00},
	{0x262b, 0x00},
	{0x262c, 0x07},
	{0x262d, 0xff},
	{0x262e, 0xff},
	{0x262f, 0x1b},
	{0x2630, 0x15},
	{0x2631, 0x00},
	{0x2632, 0x00},
	{0x2633, 0x00},
	{0x2634, 0x00},
	{0x2635, 0x00},
	{0x2636, 0x00},
	{0x2637, 0x14},
	{0x2638, 0x15},
	{0x2639, 0x00},
	{0x263a, 0x00},
	{0x263b, 0x00},
	{0x263c, 0x07},
	{0x263d, 0xff},
	{0x263e, 0xfe},
	{0x263f, 0xd8},
	{0x2640, 0x15},
	{0x2641, 0x00},
	{0x2642, 0x00},
	{0x2643, 0x00},
	{0x2644, 0x18},
	{0x2645, 0x60},
	{0x2646, 0x00},
	{0x2647, 0x14},
	{0x2648, 0xa8},
	{0x2649, 0x63},
	{0x264a, 0x5d},
	{0x264b, 0x14},
	{0x264c, 0x8c},
	{0x264d, 0x63},
	{0x264e, 0x00},
	{0x264f, 0x00},
	{0x2650, 0xbc},
	{0x2651, 0x23},
	{0x2652, 0x00},
	{0x2653, 0x01},
	{0x2654, 0x10},
	{0x2655, 0x00},
	{0x2656, 0x00},
	{0x2657, 0x0c},
	{0x2658, 0x15},
	{0x2659, 0x00},
	{0x265a, 0x00},
	{0x265b, 0x00},
	{0x265c, 0x18},
	{0x265d, 0x80},
	{0x265e, 0x00},
	{0x265f, 0x14},
	{0x2660, 0xa8},
	{0x2661, 0x84},
	{0x2662, 0x5d},
	{0x2663, 0x10},
	{0x2664, 0x84},
	{0x2665, 0x6a},
	{0x2666, 0x00},
	{0x2667, 0x00},
	{0x2668, 0x07},
	{0x2669, 0xfb},
	{0x266a, 0x77},
	{0x266b, 0x5b},
	{0x266c, 0x84},
	{0x266d, 0x84},
	{0x266e, 0x00},
	{0x266f, 0x00},
	{0x2670, 0xbc},
	{0x2671, 0x2b},
	{0x2672, 0x00},
	{0x2673, 0x00},
	{0x2674, 0x10},
	{0x2675, 0x00},
	{0x2676, 0x00},
	{0x2677, 0x04},
	{0x2678, 0x15},
	{0x2679, 0x00},
	{0x267a, 0x00},
	{0x267b, 0x00},
	{0x267c, 0x07},
	{0x267d, 0xff},
	{0x267e, 0xff},
	{0x267f, 0x38},
	{0x2680, 0x15},
	{0x2681, 0x00},
	{0x2682, 0x00},
	{0x2683, 0x00},
	{0x2684, 0x85},
	{0x2685, 0x21},
	{0x2686, 0x00},
	{0x2687, 0x00},
	{0x2688, 0x85},
	{0x2689, 0x41},
	{0x268a, 0x00},
	{0x268b, 0x04},
	{0x268c, 0x44},
	{0x268d, 0x00},
	{0x268e, 0x48},
	{0x268f, 0x00},
	{0x2690, 0x9c},
	{0x2691, 0x21},
	{0x2692, 0x00},
	{0x2693, 0x08},
	{0x2694, 0x9c},
	{0x2695, 0x21},
	{0x2696, 0xff},
	{0x2697, 0xfc},
	{0x2698, 0xd4},
	{0x2699, 0x01},
	{0x269a, 0x48},
	{0x269b, 0x00},
	{0x269c, 0x07},
	{0x269d, 0xff},
	{0x269e, 0xff},
	{0x269f, 0xd3},
	{0x26a0, 0x15},
	{0x26a1, 0x00},
	{0x26a2, 0x00},
	{0x26a3, 0x00},
	{0x26a4, 0x9d},
	{0x26a5, 0x60},
	{0x26a6, 0x00},
	{0x26a7, 0x00},
	{0x26a8, 0x85},
	{0x26a9, 0x21},
	{0x26aa, 0x00},
	{0x26ab, 0x00},
	{0x26ac, 0x44},
	{0x26ad, 0x00},
	{0x26ae, 0x48},
	{0x26af, 0x00},
	{0x26b0, 0x9c},
	{0x26b1, 0x21},
	{0x26b2, 0x00},
	{0x26b3, 0x04},
	{0x26b4, 0x00},
	{0x26b5, 0x00},
	{0x26b6, 0x00},
	{0x26b7, 0x00},
	{0x26b8, 0x00},
	{0x26b9, 0x00},
	{0x26ba, 0x00},
	{0x26bb, 0x00},
	{0x26bc, 0x00},
	{0x26bd, 0x00},
	{0x26be, 0x00},
	{0x26bf, 0x00},
	{0x26c0, 0x00},
	{0x26c1, 0x00},
	{0x26c2, 0x00},
	{0x26c3, 0x00},
	{0x26c4, 0x00},
	{0x26c5, 0x00},
	{0x26c6, 0x00},
	{0x26c7, 0x00},
	{0x26c8, 0x00},
	{0x26c9, 0x00},
	{0x26ca, 0x00},
	{0x26cb, 0x00},
	{0x5d00, 0x07},
	{0x5d01, 0xbb},
	{0x5d02, 0x00},//u8ltmtarget
	{0x5d03, 0x00},//u32pretb1ltm_work
	{0x5d04, 0x00},
	{0x5d05, 0x00},
	{0x5d06, 0x00},
	{0x5d07, 0x01},//u32again
	{0x5d08, 0x00},//su16hsize
	{0x5d10, 0x00},
	{0x5d11, 0x00},
	{0x5d12, 0x00},
	{0x5d13, 0x02},//FC
	{0x5d14, 0x00},
	{0xfffe, 0x50},
	{0x0137, 0x99},
	{0xfffe, 0x14},
	{0x002b, 0x01},
	//AE
	{0xfffe, 0x30},
	{0x1f00, 0x00},   // WIN start X  100
	{0x1f01, 0x00},
	{0x1f02, 0x00},   // WIN stat Y
	{0x1f03, 0x00},
	{0x1f04, 0x0a},   // WIN width
	{0x1f05, 0x20},
	{0x1f06, 0x05},   // WIN height
	{0x1f07, 0xa0},
	{0x1f08, 0x03},
	{0x0051, 0x01},     //gamma
	{0xfffe, 0x25},
	{0x0002, 0xf0},
	{0xfffe, 0x14},
	{0x0022, 0x1E},    //fps
	/////////////////////////////////////////////////////
	////////////////////    SENSOR     //////////////
	/////////////////////////////////////////////////////
	{0xfffe, 0x14},
	{0x000e, 0x00},
	{0x010e, 0x90},
	{0x010f, 0x00},
	{0x0110, 0x05},
	{0x0111, 0x05},
	//exposure
	{0x0114, 0x00},  //write camera line exposure[7:0]
	{0x0115, 0x0c},
	{0x0116, 0x00},
	{0x0117, 0x0d},
	{0x011c, 0x00},  //camera exposure addr mask 1
	{0x011d, 0xff},
	{0x011e, 0x00},
	{0x011f, 0xff},
	//gain
	{0x0134, 0x00},//camera gain addr
	{0x0135, 0x83},
	{0x0136, 0x00},//camera gain addr
	{0x0137, 0x00},
	{0x013c, 0x00},//camera gain addr mask 1
	{0x013d, 0xff},
	{0x013e, 0x00},
	{0x013f, 0x00},
	///////////////////////////////////////////////////////
	////////////////////////   AE START     ///////////////
	///////////////////////////////////////////////////////
	{0x004c, 0x00},
	{0x002b, 0x01},
	{0x004d, 0x01},
	{0x00fa, 0x02},
	{0x00fb, 0x00},
	{0x00fc, 0x00},
	{0x00fd, 0x30},
	{0x00e2, 0x70},
	{0x00e3, 0x00},
	{0x00de, 0x00},
	{0x00df, 0x10},
	{0x00a0, 0x01},
	{0x00a1, 0x40},
	{0x0104, 0x00},
	{0x0105, 0x00},
	{0x0106, 0x11},
	{0x0107, 0x90},
	{0x0108, 0x15},
	{0x0109, 0x10},
	{0x0145, 0x00},
	{0x0055, 0x10},
	{0x0056, 0x10},
	{0x0057, 0x10},
	{0x0058, 0x10},
	{0x0059, 0x10},
	{0x005a, 0x10},
	{0x005b, 0x40},
	{0x005c, 0x40},
	{0x005d, 0x40},
	{0x005e, 0x10},
	{0x005f, 0x10},
	{0x0060, 0x40},
	{0x0061, 0x80},
	{0x0062, 0x40},
	{0x0063, 0x10},
	{0x0064, 0x10},
	{0x0065, 0x40},
	{0x0066, 0x40},
	{0x0067, 0x40},
	{0x0068, 0x10},
	{0x0069, 0x10},
	{0x006a, 0x10},
	{0x006b, 0x10},
	{0x006c, 0x10},
	{0x006d, 0x10},
	{0x0088, 0x00},
	{0x0089, 0x07},
	{0x008a, 0x39},
	{0x008b, 0xc0},//weight
	{0x0050, 0x01},//refresh
	{0x00c6, 0x01},
	{0x01bc, 0x00},
	{0x01bd, 0x60},
	{0x01be, 0x00},
	{0x01bf, 0x60},
	{0x00ca, 0x00},
	{0x00cb, 0x60},
	{0x00cc, 0x00},
	{0x00cd, 0x80},
	{0x00c7, 0x18},
	{0x00d8, 0x40},
	{0x0212, 0x00},
	{0x0215, 0x50},
	{0x0217, 0x50},
	{0x0219, 0x50},
	{0x021b, 0xff},
	{0x01b6, 0x50},
	{0x00c8, 0x01},
	{0x0208, 0x02},
	{0x020b, 0x40},
	{0x020d, 0x04},
	{0x00da, 0x00},
	{0x00db, 0x00},
	{0x00dc, 0x03},
	{0x00dd, 0x40},
	{0x00ce, 0x00},
	{0x00cf, 0x40},
	{0x00d0, 0x00},
	{0x00d1, 0x80},			//[ 7: 0] nSpecialAvgDiffThr;
	{0x00d4, 0x00},			//[15: 8] JumpMutiple;
	{0x00d5, 0x80},			//[ 7: 0] JumpMutiple;
	{0x00d6, 0x00},			//[15: 8] MaxJumpRatio;
	{0x00d7, 0x80},			//[ 7: 0] MaxJumpRatio;
	{0x00d9, 0x00},			//[ 7: 0]JumpCntMax;
	{0xfffe, 0x50},
	{0x004d, 0x00},
	{0xfffd, 0x80},
	{0xfffe, 0x30},
	{0x0013, 0x00},
	{0x071b, 0x00},
	{0xfffe, 0x30},
	{0x0000, 0x01},
	{0x0002, 0x96},
	{0xfffe, 0x30},
	{0x03ca, 0x09},
	{0x03cb, 0x7B},
	{0x03cc, 0x0C},
	{0x03cd, 0xA4},
	{0x03ce, 0x09},
	{0x03cf, 0x7B},
	{0x03d0, 0x06},
	{0x03d1, 0x52},
	{0xfffe, 0x14},
	{0x002c, 0x01},
	{0x0030, 0x01},
	{0x0620, 0x01},
	{0x0621, 0x01},
	{0xfffe, 0x14},
	{0x0928, 0x00},
	{0x0929, 0x61},
	{0x092a, 0x00},
	{0x092b, 0xb8},
	{0x092c, 0x01},
	{0x092d, 0x60},
	{0x06e5, 0x1c},
	{0x06e6, 0x12},
	{0x06e7, 0x0f},
	{0x06e8, 0x0d},
	{0x06e9, 0x0e},
	{0x06ea, 0x10},
	{0x06eb, 0x1a},
	{0x06ec, 0x0c},
	{0x06ed, 0x06},
	{0x06ee, 0x03},
	{0x06ef, 0x02},
	{0x06f0, 0x02},
	{0x06f1, 0x02},
	{0x06f2, 0x03},
	{0x06f3, 0x04},
	{0x06f4, 0x0d},
	{0x06f5, 0x02},
	{0x06f6, 0x01},
	{0x06f7, 0x01},
	{0x06f8, 0x01},
	{0x06f9, 0x01},
	{0x06fa, 0x01},
	{0x06fb, 0x01},
	{0x06fc, 0x01},
	{0x06fd, 0x01},
	{0x06fe, 0x00},
	{0x06ff, 0x00},
	{0x0700, 0x00},
	{0x0701, 0x00},
	{0x0702, 0x00},
	{0x0703, 0x01},
	{0x0704, 0x02},
	{0x0705, 0x01},
	{0x0706, 0x00},
	{0x0707, 0x01},
	{0x0708, 0x00},
	{0x0709, 0x00},
	{0x070a, 0x00},
	{0x070b, 0x00},
	{0x070c, 0x01},
	{0x070d, 0x02},
	{0x070e, 0x01},
	{0x070f, 0x01},
	{0x0710, 0x00},
	{0x0711, 0x00},
	{0x0712, 0x00},
	{0x0713, 0x01},
	{0x0714, 0x03},
	{0x0715, 0x06},
	{0x0716, 0x03},
	{0x0717, 0x02},
	{0x0718, 0x02},
	{0x0719, 0x02},
	{0x071a, 0x02},
	{0x071b, 0x04},
	{0x071c, 0x07},
	{0x071d, 0x19},
	{0x071e, 0x10},
	{0x071f, 0x0c},
	{0x0720, 0x0a},
	{0x0721, 0x0b},
	{0x0722, 0x0d},
	{0x0723, 0x14},
	{0x0724, 0x23},
	{0x0725, 0x16},
	{0x0726, 0x18},
	{0x0727, 0x19},
	{0x0728, 0x19},
	{0x0729, 0x19},
	{0x072a, 0x19},
	{0x072b, 0x17},
	{0x072c, 0x1a},
	{0x072d, 0x1c},
	{0x072e, 0x1d},
	{0x072f, 0x1e},
	{0x0730, 0x1e},
	{0x0731, 0x1e},
	{0x0732, 0x1e},
	{0x0733, 0x1d},
	{0x0734, 0x1b},
	{0x0735, 0x1d},
	{0x0736, 0x1e},
	{0x0737, 0x1f},
	{0x0738, 0x1f},
	{0x0739, 0x20},
	{0x073a, 0x1f},
	{0x073b, 0x1f},
	{0x073c, 0x1e},
	{0x073d, 0x1e},
	{0x073e, 0x1f},
	{0x073f, 0x20},
	{0x0740, 0x20},
	{0x0741, 0x20},
	{0x0742, 0x20},
	{0x0743, 0x20},
	{0x0744, 0x1f},
	{0x0745, 0x1e},
	{0x0746, 0x1f},
	{0x0747, 0x20},
	{0x0748, 0x21},
	{0x0749, 0x21},
	{0x074a, 0x20},
	{0x074b, 0x20},
	{0x074c, 0x1f},
	{0x074d, 0x1e},
	{0x074e, 0x20},
	{0x074f, 0x20},
	{0x0750, 0x20},
	{0x0751, 0x21},
	{0x0752, 0x20},
	{0x0753, 0x1f},
	{0x0754, 0x1f},
	{0x0755, 0x1e},
	{0x0756, 0x1e},
	{0x0757, 0x20},
	{0x0758, 0x20},
	{0x0759, 0x1f},
	{0x075a, 0x1e},
	{0x075b, 0x1d},
	{0x075c, 0x1d},
	{0x075d, 0x17},
	{0x075e, 0x1c},
	{0x075f, 0x1b},
	{0x0760, 0x1b},
	{0x0761, 0x1c},
	{0x0762, 0x1c},
	{0x0763, 0x1a},
	{0x0764, 0x15},
	{0x0765, 0x21},
	{0x0766, 0x21},
	{0x0767, 0x21},
	{0x0768, 0x22},
	{0x0769, 0x22},
	{0x076a, 0x21},
	{0x076b, 0x21},
	{0x076c, 0x24},
	{0x076d, 0x21},
	{0x076e, 0x21},
	{0x076f, 0x21},
	{0x0770, 0x21},
	{0x0771, 0x21},
	{0x0772, 0x21},
	{0x0773, 0x22},
	{0x0774, 0x21},
	{0x0775, 0x20},
	{0x0776, 0x21},
	{0x0777, 0x21},
	{0x0778, 0x21},
	{0x0779, 0x21},
	{0x077a, 0x21},
	{0x077b, 0x21},
	{0x077c, 0x21},
	{0x077d, 0x20},
	{0x077e, 0x20},
	{0x077f, 0x20},
	{0x0780, 0x20},
	{0x0781, 0x20},
	{0x0782, 0x20},
	{0x0783, 0x20},
	{0x0784, 0x21},
	{0x0785, 0x20},
	{0x0786, 0x20},
	{0x0787, 0x20},
	{0x0788, 0x20},
	{0x0789, 0x20},
	{0x078a, 0x20},
	{0x078b, 0x20},
	{0x078c, 0x21},
	{0x078d, 0x20},
	{0x078e, 0x21},
	{0x078f, 0x21},
	{0x0790, 0x21},
	{0x0791, 0x21},
	{0x0792, 0x21},
	{0x0793, 0x21},
	{0x0794, 0x22},
	{0x0795, 0x22},
	{0x0796, 0x22},
	{0x0797, 0x22},
	{0x0798, 0x22},
	{0x0799, 0x22},
	{0x079a, 0x22},
	{0x079b, 0x22},
	{0x079c, 0x22},
	{0x079d, 0x21},
	{0x079e, 0x23},
	{0x079f, 0x23},
	{0x07a0, 0x24},
	{0x07a1, 0x23},
	{0x07a2, 0x23},
	{0x07a3, 0x23},
	{0x07a4, 0x23},
	{0x07a5, 0x1c},
	{0x07a6, 0x13},
	{0x07a7, 0x0f},
	{0x07a8, 0x0d},
	{0x07a9, 0x0e},
	{0x07aa, 0x10},
	{0x07ab, 0x19},
	{0x07ac, 0x18},
	{0x07ad, 0x06},
	{0x07ae, 0x03},
	{0x07af, 0x02},
	{0x07b0, 0x02},
	{0x07b1, 0x02},
	{0x07b2, 0x03},
	{0x07b3, 0x04},
	{0x07b4, 0x0b},
	{0x07b5, 0x01},
	{0x07b6, 0x01},
	{0x07b7, 0x01},
	{0x07b8, 0x01},
	{0x07b9, 0x01},
	{0x07ba, 0x01},
	{0x07bb, 0x01},
	{0x07bc, 0x01},
	{0x07bd, 0x01},
	{0x07be, 0x00},
	{0x07bf, 0x00},
	{0x07c0, 0x00},
	{0x07c1, 0x00},
	{0x07c2, 0x00},
	{0x07c3, 0x01},
	{0x07c4, 0x01},
	{0x07c5, 0x01},
	{0x07c6, 0x00},
	{0x07c7, 0x01},
	{0x07c8, 0x00},
	{0x07c9, 0x00},
	{0x07ca, 0x00},
	{0x07cb, 0x00},
	{0x07cc, 0x01},
	{0x07cd, 0x01},
	{0x07ce, 0x01},
	{0x07cf, 0x00},
	{0x07d0, 0x00},
	{0x07d1, 0x00},
	{0x07d2, 0x00},
	{0x07d3, 0x01},
	{0x07d4, 0x02},
	{0x07d5, 0x05},
	{0x07d6, 0x03},
	{0x07d7, 0x02},
	{0x07d8, 0x02},
	{0x07d9, 0x01},
	{0x07da, 0x02},
	{0x07db, 0x04},
	{0x07dc, 0x07},
	{0x07dd, 0x18},
	{0x07de, 0x0f},
	{0x07df, 0x0b},
	{0x07e0, 0x09},
	{0x07e1, 0x0a},
	{0x07e2, 0x0d},
	{0x07e3, 0x13},
	{0x07e4, 0x21},
	{0x07e5, 0x18},
	{0x07e6, 0x18},
	{0x07e7, 0x18},
	{0x07e8, 0x1a},
	{0x07e9, 0x18},
	{0x07ea, 0x19},
	{0x07eb, 0x18},
	{0x07ec, 0x18},
	{0x07ed, 0x1b},
	{0x07ee, 0x1d},
	{0x07ef, 0x1d},
	{0x07f0, 0x1e},
	{0x07f1, 0x1e},
	{0x07f2, 0x1d},
	{0x07f3, 0x1c},
	{0x07f4, 0x1b},
	{0x07f5, 0x1d},
	{0x07f6, 0x1e},
	{0x07f7, 0x1f},
	{0x07f8, 0x1f},
	{0x07f9, 0x20},
	{0x07fa, 0x1f},
	{0x07fb, 0x1e},
	{0x07fc, 0x1d},
	{0x07fd, 0x1e},
	{0x07fe, 0x1f},
	{0x07ff, 0x20},
	{0x0800, 0x21},
	{0x0801, 0x20},
	{0x0802, 0x20},
	{0x0803, 0x20},
	{0x0804, 0x1f},
	{0x0805, 0x1e},
	{0x0806, 0x20},
	{0x0807, 0x21},
	{0x0808, 0x21},
	{0x0809, 0x21},
	{0x080a, 0x20},
	{0x080b, 0x20},
	{0x080c, 0x1e},
	{0x080d, 0x1e},
	{0x080e, 0x1f},
	{0x080f, 0x20},
	{0x0810, 0x20},
	{0x0811, 0x20},
	{0x0812, 0x20},
	{0x0813, 0x1e},
	{0x0814, 0x1f},
	{0x0815, 0x1c},
	{0x0816, 0x1e},
	{0x0817, 0x1f},
	{0x0818, 0x1f},
	{0x0819, 0x1f},
	{0x081a, 0x1e},
	{0x081b, 0x1d},
	{0x081c, 0x1b},
	{0x081d, 0x19},
	{0x081e, 0x1a},
	{0x081f, 0x1c},
	{0x0820, 0x1a},
	{0x0821, 0x1b},
	{0x0822, 0x1b},
	{0x0823, 0x18},
	{0x0824, 0x18},
	{0x0825, 0x1d},
	{0x0826, 0x1d},
	{0x0827, 0x1d},
	{0x0828, 0x1e},
	{0x0829, 0x1e},
	{0x082a, 0x1e},
	{0x082b, 0x1d},
	{0x082c, 0x1e},
	{0x082d, 0x1e},
	{0x082e, 0x1f},
	{0x082f, 0x1f},
	{0x0830, 0x20},
	{0x0831, 0x1f},
	{0x0832, 0x1f},
	{0x0833, 0x1f},
	{0x0834, 0x1f},
	{0x0835, 0x1f},
	{0x0836, 0x20},
	{0x0837, 0x20},
	{0x0838, 0x20},
	{0x0839, 0x20},
	{0x083a, 0x20},
	{0x083b, 0x20},
	{0x083c, 0x20},
	{0x083d, 0x1f},
	{0x083e, 0x1f},
	{0x083f, 0x20},
	{0x0840, 0x20},
	{0x0841, 0x20},
	{0x0842, 0x20},
	{0x0843, 0x20},
	{0x0844, 0x20},
	{0x0845, 0x1f},
	{0x0846, 0x20},
	{0x0847, 0x20},
	{0x0848, 0x20},
	{0x0849, 0x20},
	{0x084a, 0x20},
	{0x084b, 0x20},
	{0x084c, 0x20},
	{0x084d, 0x20},
	{0x084e, 0x20},
	{0x084f, 0x20},
	{0x0850, 0x20},
	{0x0851, 0x20},
	{0x0852, 0x20},
	{0x0853, 0x20},
	{0x0854, 0x21},
	{0x0855, 0x1f},
	{0x0856, 0x20},
	{0x0857, 0x20},
	{0x0858, 0x21},
	{0x0859, 0x20},
	{0x085a, 0x21},
	{0x085b, 0x20},
	{0x085c, 0x20},
	{0x085d, 0x1f},
	{0x085e, 0x1f},
	{0x085f, 0x21},
	{0x0860, 0x20},
	{0x0861, 0x21},
	{0x0862, 0x20},
	{0x0863, 0x1f},
	{0x0864, 0x1f},
	{0x0865, 0x1c},
	{0x0866, 0x13},
	{0x0867, 0x0f},
	{0x0868, 0x0d},
	{0x0869, 0x0e},
	{0x086a, 0x10},
	{0x086b, 0x1a},
	{0x086c, 0x16},
	{0x086d, 0x06},
	{0x086e, 0x03},
	{0x086f, 0x02},
	{0x0870, 0x02},
	{0x0871, 0x02},
	{0x0872, 0x03},
	{0x0873, 0x04},
	{0x0874, 0x0c},
	{0x0875, 0x01},
	{0x0876, 0x01},
	{0x0877, 0x01},
	{0x0878, 0x01},
	{0x0879, 0x01},
	{0x087a, 0x01},
	{0x087b, 0x01},
	{0x087c, 0x01},
	{0x087d, 0x01},
	{0x087e, 0x00},
	{0x087f, 0x00},
	{0x0880, 0x00},
	{0x0881, 0x00},
	{0x0882, 0x00},
	{0x0883, 0x01},
	{0x0884, 0x01},
	{0x0885, 0x01},
	{0x0886, 0x00},
	{0x0887, 0x01},
	{0x0888, 0x00},
	{0x0889, 0x00},
	{0x088a, 0x00},
	{0x088b, 0x00},
	{0x088c, 0x00},
	{0x088d, 0x02},
	{0x088e, 0x01},
	{0x088f, 0x00},
	{0x0890, 0x00},
	{0x0891, 0x00},
	{0x0892, 0x00},
	{0x0893, 0x01},
	{0x0894, 0x02},
	{0x0895, 0x05},
	{0x0896, 0x03},
	{0x0897, 0x02},
	{0x0898, 0x01},
	{0x0899, 0x01},
	{0x089a, 0x02},
	{0x089b, 0x04},
	{0x089c, 0x06},
	{0x089d, 0x17},
	{0x089e, 0x0f},
	{0x089f, 0x0b},
	{0x08a0, 0x09},
	{0x08a1, 0x0a},
	{0x08a2, 0x0d},
	{0x08a3, 0x13},
	{0x08a4, 0x21},
	{0x08a5, 0x19},
	{0x08a6, 0x1a},
	{0x08a7, 0x1b},
	{0x08a8, 0x1b},
	{0x08a9, 0x1b},
	{0x08aa, 0x1b},
	{0x08ab, 0x1a},
	{0x08ac, 0x1b},
	{0x08ad, 0x1c},
	{0x08ae, 0x1d},
	{0x08af, 0x1e},
	{0x08b0, 0x1e},
	{0x08b1, 0x1e},
	{0x08b2, 0x1d},
	{0x08b3, 0x1d},
	{0x08b4, 0x1c},
	{0x08b5, 0x1d},
	{0x08b6, 0x1f},
	{0x08b7, 0x1f},
	{0x08b8, 0x20},
	{0x08b9, 0x20},
	{0x08ba, 0x20},
	{0x08bb, 0x1f},
	{0x08bc, 0x1d},
	{0x08bd, 0x1f},
	{0x08be, 0x1f},
	{0x08bf, 0x20},
	{0x08c0, 0x20},
	{0x08c1, 0x20},
	{0x08c2, 0x20},
	{0x08c3, 0x20},
	{0x08c4, 0x1f},
	{0x08c5, 0x1e},
	{0x08c6, 0x1f},
	{0x08c7, 0x20},
	{0x08c8, 0x21},
	{0x08c9, 0x21},
	{0x08ca, 0x20},
	{0x08cb, 0x20},
	{0x08cc, 0x1e},
	{0x08cd, 0x1e},
	{0x08ce, 0x1f},
	{0x08cf, 0x20},
	{0x08d0, 0x20},
	{0x08d1, 0x20},
	{0x08d2, 0x20},
	{0x08d3, 0x1f},
	{0x08d4, 0x1e},
	{0x08d5, 0x1e},
	{0x08d6, 0x1e},
	{0x08d7, 0x1f},
	{0x08d8, 0x1f},
	{0x08d9, 0x1f},
	{0x08da, 0x1e},
	{0x08db, 0x1d},
	{0x08dc, 0x1d},
	{0x08dd, 0x19},
	{0x08de, 0x1c},
	{0x08df, 0x1c},
	{0x08e0, 0x1b},
	{0x08e1, 0x1c},
	{0x08e2, 0x1c},
	{0x08e3, 0x1b},
	{0x08e4, 0x18},
	{0x08e5, 0x1c},
	{0x08e6, 0x1e},
	{0x08e7, 0x1e},
	{0x08e8, 0x1f},
	{0x08e9, 0x1f},
	{0x08ea, 0x1f},
	{0x08eb, 0x1d},
	{0x08ec, 0x20},
	{0x08ed, 0x1f},
	{0x08ee, 0x1f},
	{0x08ef, 0x21},
	{0x08f0, 0x21},
	{0x08f1, 0x21},
	{0x08f2, 0x21},
	{0x08f3, 0x20},
	{0x08f4, 0x20},
	{0x08f5, 0x20},
	{0x08f6, 0x20},
	{0x08f7, 0x21},
	{0x08f8, 0x21},
	{0x08f9, 0x21},
	{0x08fa, 0x20},
	{0x08fb, 0x21},
	{0x08fc, 0x21},
	{0x08fd, 0x1f},
	{0x08fe, 0x20},
	{0x08ff, 0x20},
	{0x0900, 0x20},
	{0x0901, 0x20},
	{0x0902, 0x20},
	{0x0903, 0x20},
	{0x0904, 0x21},
	{0x0905, 0x20},
	{0x0906, 0x20},
	{0x0907, 0x20},
	{0x0908, 0x21},
	{0x0909, 0x20},
	{0x090a, 0x20},
	{0x090b, 0x21},
	{0x090c, 0x21},
	{0x090d, 0x20},
	{0x090e, 0x21},
	{0x090f, 0x21},
	{0x0910, 0x21},
	{0x0911, 0x21},
	{0x0912, 0x21},
	{0x0913, 0x21},
	{0x0914, 0x22},
	{0x0915, 0x20},
	{0x0916, 0x21},
	{0x0917, 0x21},
	{0x0918, 0x22},
	{0x0919, 0x21},
	{0x091a, 0x22},
	{0x091b, 0x22},
	{0x091c, 0x22},
	{0x091d, 0x21},
	{0x091e, 0x21},
	{0x091f, 0x22},
	{0x0920, 0x22},
	{0x0921, 0x22},
	{0x0922, 0x21},
	{0x0923, 0x21},
	{0x0924, 0x1d},
	//blc
	{0xfffe, 0x30},
	{0x0013, 0x00},
	{0x0014, 0x00},
	{0x071b, 0x00},
	{0x0000, 0x01},
	{0x0002, 0x90},
	{0x0003, 0x31},
	{0x006C, 0x80},
	{0x0715, 0x04},
	//AWB
	{0xfffe, 0x14},
	{0x0248, 0x02},
	{0x0282, 0x06},
	{0x0283, 0x00},
	{0x0286, 0x04},
	{0x0287, 0x00},
	{0x028a, 0x04},
	{0x028b, 0x00},
	{0x028e, 0x04},
	{0x028f, 0x04},
	{0x02b6, 0x06},
	{0x02b7, 0x48},
	{0x02ba, 0x04},
	{0x02bb, 0x00},
	{0x02be, 0x07},
	{0x02bf, 0x17},
	{0xfffe, 0x14},
	{0x0248, 0x01},
	{0x0249, 0x01},
	{0x024a, 0x00},
	{0x027a, 0x08},
	{0x027b, 0x00},
	{0x027c, 0x0f},
	{0x027d, 0xff},
	{0x002c, 0x01},
	{0xfffe, 0x30},
	{0x0708, 0x03},
	{0x0709, 0xf0},
	{0x070a, 0x00},
	{0x070b, 0xc0},
	{0x071c, 0x0a},
	{0xfffd, 0x80},
	{0xfffe, 0x30},
	{0x0730, 0x90},    // win1 startx
	{0x0731, 0xc7},    // win1 endx
	{0x0732, 0x3a},    // win1 starty
	{0x0733, 0x55},    // win1 endy
	{0x0734, 0x80},    // win2 startx
	{0x0735, 0xa8},    // win2 endx
	{0x0736, 0x50},    // win2 starty
	{0x0737, 0x70},    // win2 endy
	{0x0738, 0x50},    // win3 startx
	{0x0739, 0x77},    // win3 endx
	{0x073a, 0x60},    // win3 starty
	{0x073b, 0x8b},    // win3 endy
	{0x073c, 0x77},    // win4 startx
	{0x073d, 0x9e},    // win4 endx
	{0x073e, 0x49},    // win4 starty
	{0x073f, 0x60},    // win4 endy
	{0x0740, 0x75},    // win5 startx
	{0x0741, 0x83},    // win5 endx
	{0x0742, 0x62},    // win5 starty
	{0x0743, 0x70},    // win5 endy
	{0x0744, 0x00},    // win6 startx
	{0x0745, 0x00},    // win6 endx
	{0x0746, 0x00},    // win6 starty
	{0x0747, 0x00},    // win6 endy
	{0x0748, 0x00},    // win7 startx
	{0x0749, 0x00},    // win7 endx
	{0x074a, 0x00},    // win7 starty
	{0x074b, 0x00},    // win7 endy
	{0x074c, 0x00},    // win8 startx
	{0x074d, 0x00},    // win8 endx
	{0x074e, 0x00},    // win8 starty
	{0x074f, 0x00},    // win8 endy
	{0x0750, 0x00},    // win9 startx
	{0x0751, 0x00},    // win9 endx
	{0x0752, 0x00},    // win9 starty
	{0x0753, 0x00},    // win9 endy
	{0x0754, 0x00},    // win10 startx
	{0x0755, 0x00},    // win10 endx
	{0x0756, 0x00},    // win10 starty
	{0x0757, 0x00},    // win10 endy
	{0x0758, 0x00},    // win11 startx
	{0x0759, 0x00},    // win11 endx
	{0x075a, 0x00},    // win11 starty
	{0x075b, 0x00},    // win11 endy
	{0x075c, 0x00},    // win12 startx
	{0x075d, 0x00},    // win12 endx
	{0x075e, 0x00},    // win12 starty
	{0x075f, 0x00},    // win12 endy
	{0x0760, 0x00},    // win13 startx
	{0x0761, 0x00},    // win13 endx
	{0x0762, 0x00},    // win13 starty
	{0x0763, 0x00},    // win13 endy
	{0x0764, 0x00},    // win14 startx
	{0x0765, 0x00},    // win14 endx
	{0x0766, 0x00},    // win14 starty
	{0x0767, 0x00},    // win14 endy
	{0x0768, 0x00},    // win15 startx
	{0x0769, 0x00},    // win15 endx
	{0x076a, 0x00},    // win15 starty
	{0x076b, 0x00},    // win15 endy
	{0x076c, 0x00},    // win16 startx
	{0x076d, 0x00},    // win16 endx
	{0x076e, 0x00},    // win16 starty
	{0x076f, 0x00},    // win16 endy
	{0x0770, 0x22},    // wt1 wt2
	{0x0771, 0x21},    // wt3 wt4
	{0x0772, 0x10},    // wt5 wt6
	{0x0773, 0x00},    // wt7 wt8
	{0x0774, 0x00},    // wt9 wt10
	{0x0775, 0x00},    // wt11 wt12
	{0x0776, 0x00},    // wt13 wt14
	{0x0777, 0x00},    // wt15 wt16
	{0xfffd, 0x80},
	{0xfffe, 0x30},
	{0x1400, 0x00},
	{0x1401, 0x09},
	{0x1402, 0x11},
	{0x1403, 0x19},
	{0x1404, 0x21},
	{0x1405, 0x29},
	{0x1406, 0x31},
	{0x1407, 0x39},
	{0x1408, 0x40},
	{0x1409, 0x46},
	{0x140a, 0x4c},
	{0x140b, 0x52},
	{0x140c, 0x58},
	{0x140d, 0x5d},
	{0x140e, 0x61},
	{0x140f, 0x66},
	{0x1410, 0x6a},
	{0x1411, 0x6e},
	{0x1412, 0x71},
	{0x1413, 0x75},
	{0x1414, 0x78},
	{0x1415, 0x7b},
	{0x1416, 0x7e},
	{0x1417, 0x81},
	{0x1418, 0x83},
	{0x1419, 0x86},
	{0x141a, 0x88},
	{0x141b, 0x8b},
	{0x141c, 0x8d},
	{0x141d, 0x8f},
	{0x141e, 0x92},
	{0x141f, 0x94},
	{0x1420, 0x96},
	{0x1421, 0x99},
	{0x1422, 0x9d},
	{0x1423, 0xa0},
	{0x1424, 0xa4},
	{0x1425, 0xa6},
	{0x1426, 0xa9},
	{0x1427, 0xac},
	{0x1428, 0xae},
	{0x1429, 0xb0},
	{0x142a, 0xb2},
	{0x142b, 0xb4},
	{0x142c, 0xb6},
	{0x142d, 0xb8},
	{0x142e, 0xba},
	{0x142f, 0xbc},
	{0x1430, 0xbd},
	{0x1431, 0xc1},
	{0x1432, 0xc4},
	{0x1433, 0xc7},
	{0x1434, 0xca},
	{0x1435, 0xcd},
	{0x1436, 0xd0},
	{0x1437, 0xd4},
	{0x1438, 0xd7},
	{0x1439, 0xdc},
	{0x143a, 0xe0},
	{0x143b, 0xe5},
	{0x143c, 0xea},
	{0x143d, 0xef},
	{0x143e, 0xf5},
	{0x143f, 0xfa},
	{0x1440, 0xff},
	{0xfffe, 0x30},
	{0x1200, 0x05},
	{0x1201, 0xC5},
	{0x1202, 0x00},
	{0x1203, 0xe0},
	{0x1204, 0x01},
	{0x1205, 0x1A},
	{0x1206, 0x01},
	{0x1207, 0x35},
	{0x1208, 0x00},
	{0x1209, 0x44},
	{0x120A, 0x04},
	{0x120B, 0xE6},
	{0x120C, 0x0C},
	{0x120D, 0x6A},
	{0x120E, 0x03},
	{0x120F, 0x0D},
	{0x1210, 0x00},
	{0x1211, 0xF2},
	{0x1212, 0x03},
	{0x1213, 0x50},
	{0x1214, 0x00},
	{0x1215, 0xF8},
	{0x1216, 0x02},
	{0x1217, 0x3F},
	{0x1218, 0x07},
	{0x1219, 0xE6},
	{0x121A, 0x00},
	{0x121B, 0x87},
	{0x121C, 0x00},
	{0x121D, 0x3C},
	{0x121E, 0x01},
	{0x121F, 0xea},
	{0x1220, 0x00},
	{0x1221, 0x4E},
	{0x1222, 0x04},
	{0x1223, 0xCF},
	{0x122e, 0x00},
	{0x122F, 0x00},
	{0x1230, 0x02},
	{0x1228, 0x00},
	{0x1229, 0x6D},
	{0x122A, 0x00},
	{0x122B, 0xAE},
	{0x122C, 0x01},
	{0x122D, 0x37},
	{0x1231, 0x02},
	{0xfffe, 0x30},
	{0x2000, 0x0f},
	{0x2001, 0x16},
	{0x2002, 0x1a},
	{0x2003, 0x1f},
	{0x2004, 0x20},
	{0x2005, 0x24},
	{0x2006, 0x18},
	{0x2007, 0x12},
	{0x2008, 0x10},
	{0x2009, 0x06},
	{0x200a, 0x01},
	{0x200b, 0x01},
	{0x200c, 0x10},
	{0x200d, 0x0e},
	{0x200e, 0x0a},
	{0x200f, 0x06},
	{0x2010, 0x01},
	{0x2011, 0x01},
	{0x2012, 0x28},
	{0x2013, 0x2a},
	{0x2014, 0x2e},
	{0x2015, 0x20},
	{0x2016, 0x24},
	{0x2017, 0x28},
	{0x2018, 0x20},
	{0x2019, 0x24},
	{0x201a, 0x28},
	{0x201b, 0x2c},
	{0x201c, 0x30},
	{0x201d, 0x34},
	{0x201e, 0x2c},
	{0x201f, 0x28},
	{0x2020, 0x30},
	{0x2021, 0x20},
	{0x2022, 0x30},
	{0x2023, 0x34},
	{0x2024, 0x10},
	{0x2025, 0x2c},
	{0x2026, 0x30},
	{0x2027, 0x24},
	{0x2028, 0x30},
	{0x2029, 0x34},
	{0x202a, 0x20},
	{0x202b, 0x20},
	{0x202c, 0x20},
	{0x202d, 0x20},
	{0x202e, 0x20},
	{0x202f, 0x20},
	{0x2030, 0x20},
	{0x2031, 0x20},
	{0x2032, 0x20},
	{0x2033, 0x20},
	{0x2034, 0x20},
	{0x2035, 0x20},
	{0x2036, 0x20},
	{0x2037, 0x20},
	{0x2038, 0x20},
	{0x2039, 0x20},
	{0x203a, 0x20},
	{0x203b, 0x20},
	{0x203c, 0x20},
	{0x203d, 0x20},
	{0x203e, 0x20},
	{0x203f, 0x20},
	{0x2040, 0x20},
	{0x2041, 0x20},
	{0x2044, 0x10},
	{0xfffe, 0x30},
	{0x2100, 0x01},
	{0x2101, 0x01},
	{0x2102, 0x01},
	{0x2103, 0x01},
	{0x2104, 0x3f},
	{0x2105, 0x4f},
	{0x2106, 0x5f},
	{0x2107, 0x08},
	{0xfffe, 0x30},
	{0x0f00, 0x65},
	{0x0f02, 0x00},
	{0x0f03, 0x00},
	{0x0f04, 0x00},
	{0x0f05, 0x00},
	{0x0f06, 0x00},
	{0x0f07, 0x00},
	{0x0f08, 0x00},
	{0x0f09, 0x00},
	{0x0f0a, 0x02},
	{0x0f0b, 0x2f},
	{0x0f0c, 0x01},
	{0x0f0d, 0x03},
	{0x0f0e, 0x06},
	{0x0f0f, 0x0c},
	{0x0f10, 0x30},
	{0xfffe, 0x30},
	{0x2b04, 0x08},
	{0x2b16, 0x05},
	{0xfffe, 0x14},
	{0x002f, 0x01},
	{0x09a8, 0x01},
	{0x09ab, 0x02},
	{0x09ac, 0x08},
	{0x09ad, 0xb4},
	{0x09ae, 0x41},
	{0x09af, 0xFF},
	{0x09b2, 0x02},
	{0x09b3, 0x00},
	{0x09b4, 0x10},
	{0x09b5, 0x14},
	{0x09b6, 0x00},
	{0x09bb, 0x30},
	{0x09bc, 0x31},
	{0x09bd, 0x67},
	{0x09be, 0x68},
	{0x09bf, 0xf7},
	{0x09c0, 0xf8},
	{0x09c1, 0xff},
	{0x09c2, 0x0c},
	{0x09c3, 0x65},
	{0x09c4, 0x66},
	{0x09c5, 0x89},
	{0x09c6, 0x8a},
	{0x09c7, 0xf8},
	{0x09c8, 0xfa},
	{0x09c9, 0xff},
	{0xfffe, 0x30},
	{0x2b9a, 0x00},
	{0x2b9b, 0x00},
	{0x2b9c, 0x10},
	{0x2b9d, 0x00},
	{0x2b9e, 0x20},
	{0x2b9f, 0x00},
	{0x2ba0, 0x40},
	{0x2ba1, 0x00},
	{0x2ba2, 0x70},
	{0x2ba3, 0x00},
	{0x2ba4, 0xa0},
	{0x2ba5, 0x00},
	{0x2ba6, 0xd0},
	{0x2ba7, 0x00},
	{0x2ba8, 0x00},
	{0x2ba9, 0x01},
	{0x2bac, 0x18},
	{0x2bad, 0x18},
	{0x2bae, 0x1a},
	{0x2baf, 0x1a},
	{0x2bb0, 0x1b},
	{0x2bb1, 0x1b},
	{0x2bb2, 0x1d},
	{0x2bb3, 0x20},
	{0x2bb4, 0x7f},
	{0x2bb5, 0x7f},
	{0x2bb6, 0x7f},
	{0x2bb7, 0x7f},
	{0x2bb8, 0x7f},
	{0x2bb9, 0x7f},
	{0x2bba, 0x7f},
	{0x2bbb, 0x7f},
	{0x2bbc, 0x24},
	{0x2bbd, 0x24},
	{0x2bbe, 0x20},
	{0x2bbf, 0x12},
	{0x2bc0, 0x0e},
	{0x2bc1, 0x0e},
	{0x2bc2, 0x0c},
	{0x2bc3, 0x0c},
	{0xfffe, 0x14},
	{0x026c, 0x00},
	{0x026e, 0x50},
	{0x026f, 0x40},
	{0x0270, 0x40},
	{0x0271, 0x30},
	{0x0272, 0x30},
	{0x0273, 0x30},
	{0x0274, 0x50},
	{0x0275, 0x3A},
	{0x0276, 0x3A},
	{0x0277, 0x2A},
	{0x0278, 0x2A},
	{0x0279, 0x2A},
	{0xfffe, 0x30},
	{0x0000, 0x47},
	{0x0001, 0x92},
	{0x0002, 0x96},
	{0x0003, 0x35},
	{0x2b16, 0x04},
	{0x2b04, 0x08},
	{0x1700, 0x09},
	{0x1701, 0x32},
	{0x1702, 0x43},
	{0x1704, 0x22},
	{0x071b, 0x00},
	{0x071c, 0x0a},
	{0xfffe, 0x14},
	{0x024a, 0x02},
	{0x026a, 0x84},
	{0x026b, 0x80},
	{REG_NULL_XC7080, 0x00},
};

static const struct regval1 xc7080_xc530_2592x1944_regs[] = {
	{0xEF, 0x05},
	{0x0F, 0x00},
	{0x43, 0x02},
	{0x44, 0x00},
	{0xED, 0x01},
	{0xEF, 0x01},
	{0xF5, 0x01},
	{0x09, 0x01},
	{0xEF, 0x00},
	{0x06, 0x02},
	{0x0B, 0x00},
	{0x0C, 0xA0},
	{0x10, 0x80},
	{0x11, 0x80},
	{0x12, 0x80},
	{0x13, 0x00},
	{0x14, 0xBD},
	{0x15, 0x1F},
	{0x16, 0xBD},
	{0x17, 0xBD},
	{0x18, 0xBD},
	{0x19, 0x64},
	{0x1B, 0x64},
	{0x1C, 0x64},
	{0x1E, 0x64},
	{0x1F, 0x64},
	{0x20, 0x64},
	{0x21, 0x00},
	{0x23, 0x00},
	{0x24, 0x00},
	{0x26, 0x00},
	{0x27, 0x00},
	{0x28, 0x00},
	{0x29, 0x64},
	{0x2B, 0x64},
	{0x2C, 0x64},
	{0x2E, 0x64},
	{0x2F, 0x64},
	{0x30, 0x64},
	{0x31, 0x0F},
	{0x32, 0x00},
	{0x33, 0x64},
	{0x34, 0x64},
	{0x35, 0x01},
	{0x36, 0x0F},
	{0x37, 0x0F},
	{0x38, 0xE0},
	{0x39, 0x00},
	{0x3A, 0x00},
	{0x3B, 0x00},
	{0x3C, 0x00},
	{0x45, 0x00},
	{0x46, 0x00},
	{0x48, 0x00},
	{0x49, 0xD2},
	{0x4A, 0x0A},
	{0x4B, 0x6E},
	{0x4C, 0x01},
	{0x4D, 0xD2},
	{0x4E, 0xC4},
	{0x4F, 0x28},
	{0x53, 0x00},
	{0x57, 0x00},
	{0x58, 0x06},
	{0x59, 0x05},
	{0x5A, 0x04},
	{0x5B, 0x03},
	{0x5C, 0x02},
	{0x5D, 0x01},
	{0x5F, 0xC2},
	{0x60, 0x44},
	{0x61, 0x54},
	{0x62, 0x29},
	{0x69, 0x10},
	{0x6A, 0x40},
	{0x6B, 0x00},
	{0x6C, 0x00},
	{0x6D, 0x00},
	{0x6E, 0x00},
	{0x6F, 0x32},
	{0x74, 0x00},
	{0x75, 0x78},
	{0x76, 0x00},
	{0x77, 0x20},
	{0x7E, 0x32},
	{0x7F, 0x90},
	{0x80, 0x02},
	{0x81, 0x80},
	{0x84, 0xFE},
	{0x85, 0x22},
	{0x86, 0x09},
	{0x87, 0x60},
	{0x88, 0x0B},
	{0x89, 0x01},
	{0x8B, 0x76},
	{0x8C, 0x00},
	{0x8D, 0x00},
	{0x8E, 0x00},
	{0x8F, 0x00},
	{0x90, 0x00},
	{0x91, 0x00},
	{0x92, 0x11},
	{0x93, 0x10},
	{0x94, 0x00},
	{0x95, 0x00},
	{0x96, 0x00},
	{0x97, 0x00},
	{0x98, 0x02},
	{0x99, 0x00},
	{0x9A, 0x00},
	{0x9B, 0x09},
	{0x9C, 0x00},
	{0x9D, 0x00},
	{0x9E, 0x00},
	{0x9F, 0x00},
	{0xA0, 0x02},
	{0xA1, 0x00},
	{0xA2, 0x0A},
	{0xA3, 0x07},
	{0xA4, 0xFF},
	{0xA5, 0x03},
	{0xA6, 0xFF},
	{0xA7, 0x00},
	{0xA8, 0x00},
	{0xA9, 0x11},
	{0xAA, 0x54},
	{0xAB, 0x10},
	{0xAC, 0x11},
	{0xAD, 0x00},
	{0xAE, 0x00},
	{0xAF, 0x00},
	{0xB0, 0x02},
	{0xB1, 0x00},
	{0xB2, 0x13},
	{0xBE, 0x05},
	{0xBF, 0x00},
	{0xC0, 0x10},
	{0xC7, 0x10},
	{0xC8, 0x01},
	{0xC9, 0x00},
	{0xCA, 0x55},
	{0xCB, 0x06},
	{0xCC, 0x09},
	{0xCD, 0x00},
	{0xCE, 0xA2},
	{0xCF, 0x00},
	{0xD0, 0x02},
	{0xD1, 0x10},
	{0xD2, 0x1E},
	{0xD3, 0x19},
	{0xD4, 0x04},
	{0xD5, 0x18},
	{0xD6, 0xC8},
	{0xD8, 0x10},
	{0xD9, 0x30},
	{0xDA, 0x40},
	{0xDE, 0x00},
	{0xDF, 0x24},
	{0xE0, 0x0D},
	{0xE1, 0x05},
	{0xE2, 0x05},
	{0xE3, 0x24},
	{0xE4, 0x0D},
	{0xE5, 0x05},
	{0xE6, 0x05},
	{0xF0, 0x80},
	{0xF1, 0x00},
	{0xF2, 0x00},
	{0xF3, 0xB0},
	{0xF4, 0x00},
	{0xF5, 0x48},
	{0xF6, 0x00},
	{0xF7, 0x00},
	{0xF8, 0x2A},
	{0xF9, 0x00},
	{0xFA, 0x00},
	{0xFB, 0x00},
	{0xFC, 0x00},
	{0xFD, 0x00},
	{0xFE, 0x00},
	{0xED, 0x01},
	{0xEF, 0x01},
	{0x02, 0xFF},
	{0x03, 0x03},
	{0x04, 0x00},
	{0x05, 0x0B},
	{0x06, 0xFF},
	{0x07, 0x04},
	{0x08, 0x00},
	{0x09, 0x00},
	{0x0A, 0x07},
	{0x0B, 0xBB},
	{0x0C, 0x00},
	{0x0D, 0x03},
	{0x0E, 0x0A},
	{0x0F, 0x12},
	{0x10, 0x00},
	{0x11, 0x78},
	{0x12, 0x78},
	{0x13, 0x78},
	{0x14, 0x01},
	{0x15, 0x00},
	{0x16, 0x00},
	{0x17, 0x00},
	{0x1A, 0x00},
	{0x1B, 0x0A},
	{0x1C, 0x30},
	{0x1D, 0x07},
	{0x1E, 0xAE},
	{0x1F, 0x00},
	{0x20, 0x02},
	{0x21, 0x00},
	{0x22, 0x00},
	{0x23, 0x00},
	{0x24, 0x00},
	{0x25, 0x00},
	{0x26, 0x08},
	{0x27, 0x15},
	{0x28, 0x18},
	{0x29, 0x1E},
	{0x2A, 0x56},
	{0x2C, 0x00},
	{0x2D, 0x06},
	{0x2E, 0x78},
	{0x2F, 0x00},
	{0x34, 0x01},
	{0x35, 0x01},
	{0x37, 0x2C},
	{0x39, 0x36},
	{0x3D, 0x00},
	{0x3E, 0x10},
	{0x3F, 0xA6},
	{0x40, 0x96},
	{0x41, 0x11},
	{0x42, 0xF4},
	{0x43, 0xD6},
	{0x47, 0x00},
	{0x48, 0x08},
	{0x49, 0x00},
	{0x4A, 0x0E},
	{0x4B, 0x07},
	{0x4C, 0x98},
	{0x4D, 0x00},
	{0x4E, 0x02},
	{0x4F, 0x08},
	{0x50, 0x08},
	{0x51, 0x0A},
	{0x52, 0x08},
	{0x53, 0x00},
	{0x54, 0x02},
	{0x55, 0x00},
	{0x56, 0x08},
	{0x57, 0x00},
	{0x58, 0x00},
	{0x59, 0x0A},
	{0x5A, 0x20},
	{0x5B, 0x20},
	{0x5C, 0x1E},
	{0x5D, 0x0A},
	{0x5E, 0xB2},
	{0x5F, 0x00},
	{0x60, 0xC8},
	{0x61, 0x05},
	{0x62, 0xDC},
	{0x63, 0x07},
	{0x64, 0x98},
	{0x65, 0x03},
	{0x66, 0xCC},
	{0x67, 0x00},
	{0x68, 0xFA},
	{0x69, 0xC8},
	{0x6B, 0x08},
	{0x6C, 0x00},
	{0x6D, 0xC8},
	{0x6E, 0x00},
	{0x6F, 0x08},
	{0x70, 0x00},
	{0x71, 0xC8},
	{0x72, 0x00},
	{0x73, 0x98},
	{0x74, 0x00},
	{0x75, 0x56},
	{0x76, 0x00},
	{0x77, 0x00},
	{0x78, 0x00},
	{0x79, 0x0F},
	{0x7A, 0x02},
	{0x7B, 0xC8},
	{0x7C, 0x02},
	{0x7D, 0x06},
	{0x7E, 0x04},
	{0x7F, 0x00},
	{0x80, 0x00},
	{0x81, 0x00},
	{0x82, 0x00},
	{0x83, 0x00},
	{0x84, 0x12},
	{0x85, 0xC0},
	{0x86, 0x00},
	{0x87, 0x00},
	{0x88, 0x0C},
	{0x89, 0x00},
	{0x8A, 0x07},
	{0x8B, 0xAE},
	{0x8C, 0x00},
	{0x8D, 0x02},
	{0x8E, 0x10},
	{0x8F, 0x0F},
	{0x90, 0x01},
	{0x91, 0x01},
	{0x92, 0x80},
	{0x93, 0x00},
	{0x94, 0xFF},
	{0x96, 0x80},
	{0x97, 0x01},
	{0x99, 0x12},
	{0x9A, 0x14},
	{0x9B, 0x0E},
	{0x9C, 0x14},
	{0x9E, 0x14},
	{0x9F, 0x00},
	{0xA0, 0x00},
	{0xA1, 0x00},
	{0xA2, 0x00},
	{0xA3, 0x00},
	{0xA4, 0x16},	// 2592x1944
	{0xA5, 0x07},
	{0xA6, 0x98},	// 2592x1944
	{0xA7, 0x00},
	{0xA8, 0x08},	// 2592x1944
	{0xA9, 0x0A},
	{0xAA, 0x20},	// 2592x1944
	{0xAB, 0x01},
	{0xAD, 0x00},
	{0xAE, 0x50},
	{0xAF, 0x00},
	{0xB0, 0x50},
	{0xB1, 0x02},
	{0xB2, 0x01},
	{0xB3, 0x03},
	{0xB5, 0x04},
	{0xB6, 0x0A},
	{0xB7, 0x14},
	{0xB8, 0x01},
	{0xB9, 0x2C},
	{0xBA, 0xA0},
	{0xBB, 0xA0},
	{0xBC, 0x00},
	{0xBD, 0x01},
	{0xBE, 0x07},
	{0xBF, 0xFF},
	{0xC0, 0x07},
	{0xC1, 0xFF},
	{0xC2, 0x00},
	{0xC3, 0x30},
	{0xC4, 0x54},
	{0xC6, 0x10},
	{0xC7, 0x10},
	{0xC8, 0x60},
	{0xC9, 0x55},
	{0xCB, 0x32},
	{0xCC, 0x96},
	{0xCD, 0x64},
	{0xCE, 0x30},
	{0xCF, 0x82},
	{0xD0, 0x02},
	{0xD1, 0x60},
	{0xD3, 0x01},
	{0xD4, 0x04},
	{0xD5, 0x61},
	{0xD6, 0x00},
	{0xD7, 0x0A},
	{0xD8, 0xA0},
	{0xDA, 0x70},
	{0xDC, 0x00},
	{0xDD, 0x42},
	{0xE0, 0x08},
	{0xE1, 0x11},
	{0xE2, 0x0A},
	{0xE3, 0xA1},
	{0xE4, 0x00},
	{0xE6, 0x00},
	{0xE7, 0x00},
	{0xEA, 0x08},
	{0xF0, 0x8D},
	{0xF1, 0x16},
	{0xF2, 0x26},
	{0xF3, 0x1C},
	{0xF4, 0x11},
	{0xF5, 0x19},
	{0xF6, 0x05},
	{0xF7, 0x00},
	{0xF8, 0x48},
	{0xF9, 0x07},
	{0xFA, 0x21},
	{0xFB, 0x02},
	{0x09, 0x01},
	{0xEF, 0x02},
	{0x10, 0x00},
	{0x20, 0x03},
	{0x21, 0x18},
	{0x22, 0x0C},
	{0x23, 0x08},
	{0x24, 0x05},
	{0x25, 0x03},
	{0x26, 0x02},
	{0x27, 0x18},
	{0x28, 0x0C},
	{0x29, 0x08},
	{0x2A, 0x05},
	{0x2B, 0x03},
	{0x2C, 0x02},
	{0x2E, 0x04},
	{0x30, 0x9F},
	{0x31, 0x06},
	{0x32, 0x07},
	{0x33, 0x84},
	{0x34, 0x00},
	{0x35, 0x00},
	{0x36, 0x01},
	{0x37, 0x00},
	{0x38, 0x00},
	{0x39, 0x00},
	{0x3A, 0xCE},
	{0x3B, 0x15},
	{0x3C, 0xFA},
	{0x3D, 0x04},
	{0x3E, 0x00},
	{0x3F, 0x0A},
	{0x40, 0x08},
	{0x41, 0x09},
	{0x42, 0x0B},
	{0x43, 0x0A},
	{0x45, 0x00},
	{0x46, 0x00},
	{0x47, 0x00},
	{0x48, 0x00},
	{0x49, 0x00},
	{0x4A, 0x00},
	{0x4B, 0x00},
	{0x4C, 0x00},
	{0x4D, 0x00},
	{0x4E, 0x02},
	{0xA0, 0x00},
	{0xA1, 0x00},
	{0xA2, 0x00},
	{0xA3, 0x00},
	{0xA4, 0x00},
	{0xA5, 0x00},
	{0xA6, 0x00},
	{0xA7, 0x00},
	{0xA9, 0x00},
	{0xAA, 0x00},
	{0xAB, 0x00},
	{0xB2, 0x01},
	{0xB3, 0x00},
	{0xB4, 0x03},
	{0xB5, 0x01},
	{0xB6, 0x03},
	{0xB9, 0x00},
	{0xBA, 0x00},
	{0xBB, 0x00},
	{0xBC, 0x00},
	{0xBD, 0x00},
	{0xBE, 0x00},
	{0xED, 0x01},
	{0xEF, 0x05},
	{0x03, 0x10},
	{0x04, 0x38},
	{0x05, 0x04},
	{0x06, 0x64},//RAW10
	{0x07, 0x80},
	{0x08, 0x07},
	{0x09, 0x09},
	{0x0A, 0x05},
	{0x0B, 0x06},
	{0x0C, 0x04},
	{0x0D, 0xa1},
	{0x0E, 0x00},
	{0x0F, 0x00},//MIPI CSI disable
	{0x10, 0x02},//2Lane
	{0x11, 0x01},
	{0x12, 0x00},
	{0x13, 0x00},
	{0x14, 0xB8},
	{0x15, 0x07},
	{0x16, 0x06},
	{0x17, 0x06},
	{0x18, 0x05},
	{0x19, 0x04},
	{0x1A, 0x06},
	{0x1B, 0x03},
	{0x1C, 0x04},
	{0x1D, 0x08},
	{0x1E, 0x1A},
	{0x1F, 0x00},
	{0x20, 0x00},
	{0x21, 0x1E},
	{0x22, 0x1E},
	{0x23, 0x01},
	{0x24, 0x04},
	{0x25, 0x00},
	{0x27, 0x00},
	{0x28, 0x00},
	{0x2A, 0x18},
	{0x2B, 0x06},
	{0x2C, 0xC4},
	{0x2D, 0x09},
	{0x2E, 0x00},
	{0x2F, 0x06},
	{0x30, 0x00},
	{0x31, 0x06},
	{0x32, 0x00},
	{0x33, 0x00},
	{0x34, 0x00},
	{0x35, 0x00},
	{0x36, 0x00},
	{0x37, 0x00},
	{0x38, 0x2F},
	{0x3A, 0x02},
	{0x3B, 0x00},
	{0x3C, 0x00},
	{0x3D, 0x00},
	{0x3E, 0x00},
	{0x40, 0x16},
	{0x41, 0x15},
	{0x43, 0x02},
	{0x44, 0x01},
	{0x49, 0x01},
	{0x4A, 0x02},
	{0x4D, 0x02},
	{0x4F, 0x01},
	{0x5B, 0x00},
	{0x5C, 0x03},
	{0x5D, 0x00},
	{0x5E, 0x07},
	{0x5F, 0x08},
	{0x60, 0x00},
	{0x61, 0x00},
	{0x62, 0x00},
	{0x63, 0x28},
	{0x64, 0x30},
	{0x65, 0x9E},
	{0x66, 0xB9},
	{0x67, 0x52},
	{0x68, 0x70},
	{0x69, 0x4E},
	{0x70, 0x00},
	{0x71, 0x00},
	{0x72, 0x00},
	{0x90, 0x04},
	{0x91, 0x01},
	{0x92, 0x00},
	{0x93, 0x00},
	{0x94, 0x04},
	{0x96, 0x00},
	{0x97, 0x01},
	{0x98, 0x01},
	{0x99, 0x01},
	{0x9A, 0x01},
	{0xA0, 0x00},
	{0xA1, 0x01},
	{0xA2, 0x00},
	{0xA3, 0x01},
	{0xA4, 0x00},
	{0xA5, 0x01},
	{0xA6, 0x00},
	{0xA7, 0x00},
	{0xAA, 0x00},
	{0xAB, 0x0F},
	{0xAC, 0x08},
	{0xAD, 0x09},
	{0xAE, 0x0A},
	{0xAF, 0x0B},
	{0xB0, 0x00},
	{0xB1, 0x04},
	{0xB2, 0x01},
	{0xB3, 0x00},
	{0xB4, 0x00},
	{0xB5, 0x0A},
	{0xB6, 0x0A},
	{0xB7, 0x0A},
	{0xB8, 0x0A},
	{0xB9, 0x00},
	{0xBA, 0x00},
	{0xBB, 0x00},
	{0xBC, 0x00},
	{0xBD, 0x00},
	{0xBE, 0x00},
	{0xBF, 0x00},
	{0xC5, 0x00},
	{0xC6, 0x00},
	{0xC7, 0x00},
	{0xC8, 0x00},
	{0xCE, 0x00},
	{0xCF, 0x63},
	{0xD3, 0x80},
	{0xD4, 0x00},
	{0xD5, 0x00},
	{0xD6, 0x03},
	{0xD7, 0x77},
	{0xD8, 0x00},
	{0xED, 0x01},
	{0xEF, 0x06},
	{0x00, 0x0C},
	{0x01, 0x00},
	{0x02, 0x13},
	{0x03, 0x5C},
	{0x04, 0x00},
	{0x05, 0x30},
	{0x06, 0x02},
	{0x07, 0x00},
	{0x08, 0x00},
	{0x09, 0x02},
	{0x0A, 0x15},
	{0x0B, 0x90},
	{0x0C, 0x90},
	{0x0D, 0x90},
	{0x0E, 0x00},
	{0x0F, 0x1B},
	{0x10, 0x20},
	{0x11, 0x1B},
	{0x12, 0x20},
	{0x17, 0x00},
	{0x18, 0x40},
	{0x19, 0x00},
	{0x1A, 0x40},
	{0x28, 0x03},
	{0x29, 0x00},
	{0x2A, 0x00},
	{0x2B, 0x20},
	{0x2C, 0x08},
	{0x2D, 0x00},
	{0x2E, 0x20},
	{0x2F, 0x20},
	{0x45, 0x80},
	{0x46, 0x00},
	{0x49, 0x0C},
	{0x4A, 0x40},
	{0x4B, 0x40},
	{0x5A, 0x00},
	{0x5B, 0x00},
	{0x5C, 0x00},
	{0x5D, 0x00},
	{0x5E, 0x90},
	{0x5F, 0x90},
	{0x60, 0x90},
	{0x61, 0x10},
	{0x70, 0x00},
	{0x71, 0x00},
	{0x72, 0x00},
	{0x7C, 0x00},
	{0x7D, 0x00},
	{0x7E, 0x00},
	{0x83, 0x00},
	{0x84, 0x00},
	{0x85, 0x00},
	{0x8A, 0x00},
	{0x8B, 0x00},
	{0x8C, 0x00},
	{0x91, 0x00},
	{0x92, 0x00},
	{0x93, 0x00},
	{0x98, 0x05},
	{0x99, 0x23},
	{0x9A, 0x00},
	{0x9B, 0x44},
	{0x9C, 0x44},
	{0x9D, 0x04},
	{0x9E, 0x42},
	{0x9F, 0x44}, // Cmd_LTM_LThd[4:0]=8
	{0xA0, 0x00},
	{0xA1, 0x01},
	{0xA2, 0x00},
	{0xA3, 0x00},
	{0xAC, 0x00},
	{0xAD, 0x03},
	{0xAE, 0x00},
	{0xAF, 0xE8},
	{0xB0, 0x00},
	{0xB1, 0x80},
	{0xB2, 0x80},
	{0xB3, 0x80},
	{0xB4, 0x80},
	{0xBD, 0x00},
	{0xBE, 0x03},
	{0xBF, 0x00},
	{0xC0, 0xE8},
	{0xC1, 0x00},
	{0xC2, 0x80},
	{0xC3, 0x80},
	{0xC4, 0x80},
	{0xC5, 0x80},
	{0xD8, 0x00},
	{0xD9, 0x00},
	{0xDA, 0x00},
	{0xDB, 0x00},
	{0xDC, 0x00},
	{0xDD, 0x00},
	{0xDE, 0x00},
	{0xDF, 0x00},
	{0xE0, 0x01},
	{0xE1, 0x01},
	{0xE2, 0x03},
	{0xE3, 0x03},
	{0xF0, 0x00},
	{0xF3, 0x00},
	{0xF4, 0x00},
	{0xF5, 0x00},
	{0xF6, 0x00},
	{0xF7, 0x00},
	{0xF8, 0x00},
	{0xF9, 0x00},
	{0xFA, 0x00},
	{0xFB, 0x00},
	{0xFC, 0x00},
	{0xFD, 0x00},
	{0xFE, 0x00},
	{0xF1, 0x01},
	{0xEF, 0x05},
	{0x3B, 0x00},//R_Cmd_Gated_MIPI_Clk=0
	{0xED, 0x01},
	{0xEF, 0x01},
	{0x02, 0xFB},//ResetTG
	{0x09, 0x01},
	{0xEF, 0x00},
	{0x11, 0x00},//GatedAllClk disable,
	{0xEF, 0x05},
	{0x0F, 0x01},//MIPI CSI enable
	{0xED, 0x01},
	{REG_NULL_XC530, 0x00},
};

static struct regval stream_on[] = {
	{0xfffd, 0x80},
	{0xfffe, 0x26},
	{0x8010, 0x0d},
	{REG_NULL_XC7080, 0x00},
};

static struct regval stream_off[] = {
	{0xfffd, 0x80},
	{0xfffe, 0x26},
	{0x8010, 0x09},
	{REG_NULL_XC7080, 0x00},
};

static struct regval XC7080_bypass_on[] = {
	{0xfffd, 0x80},
	{0xfffe, 0x50},
	{0x004d, 0x01},
	{REG_NULL_XC7080, 0x00},
};

static struct regval XC7080_bypass_off[] = {
	{0xfffd, 0x80},
	{0xfffe, 0x50},
	{0x004d, 0x00},
	{REG_NULL_XC7080, 0x00},
};

static const struct xc7080_mode supported_modes[] = {
	{
		.width = 2592,
		.height = 1944,
		.max_fps = {
			.numerator = 10000,
			.denominator = 150000,
		},
		.exp_def = 0x0320,
		.hts_def = 0x0b60,//0x2d8*4
		.vts_def = 0x038e,
		.xc7080_reg_list = xc7080_2592x1944_regs,
		.xc530_reg_list = xc7080_xc530_2592x1944_regs,
	},
	{
		.width = 1920,
		.height = 1080,
		.max_fps = {
			.numerator = 10000,
			.denominator = 250000,
		},
		.exp_def = 0x0320,
		.hts_def = 0x0b60,//0x2d8*4
		.vts_def = 0x038e,
		.xc7080_reg_list = xc7080_1920x1080_regs,
		.xc530_reg_list = xc7080_xc530_2560x1440_regs,
	},
};

static const s64 link_freq_menu_items[] = {
	XC7080_LINK_FREQ_352MHZ
};

static const char * const xc7080_test_pattern_menu[] = {
	"Disabled",
	"Vertical Color Bar Type 1",
	"Vertical Color Bar Type 2",
	"Vertical Color Bar Type 3",
	"Vertical Color Bar Type 4"
};

/* Write registers up to 4 at a time */
static int xc7080_write_reg(struct i2c_client *client, u16 reg,
			    u32 len, u32 val)
{
	u32 buf_i, val_i;
	u8 buf[6];
	u8 *val_p;
	__be32 val_be;

	if (len > 4)
		return -EINVAL;

	buf[0] = reg >> 8;
	buf[1] = reg & 0xff;

	val_be = cpu_to_be32(val);
	val_p = (u8 *)&val_be;
	buf_i = 2;
	val_i = 4 - len;

	while (val_i < 4)
		buf[buf_i++] = val_p[val_i++];

	if (i2c_master_send(client, buf, len + 2) != len + 2)
		return -EIO;

	return 0;
}

static int xc7080_write_array(struct i2c_client *client,
			const struct regval *regs)
{
	u32 i;
	int ret = 0;

	for (i = 0; ret == 0 && regs[i].addr != REG_NULL_XC7080; i++) {
		ret = xc7080_write_reg(client, regs[i].addr,
				       XC7080_REG_VALUE_08BIT, regs[i].val);
		}
	return ret;
}

/* Read registers up to 4 at a time */
static int xc7080_read_reg(struct i2c_client *client, u16 reg,
			unsigned int len, u32 *val)
{
	struct i2c_msg msgs[2];
	u8 *data_be_p;
	__be32 data_be = 0;
	__be16 reg_addr_be = cpu_to_be16(reg);
	int ret;

	if (len > 4 || !len)
		return -EINVAL;

	data_be_p = (u8 *)&data_be;
	/* Write register address */
	msgs[0].addr = client->addr;
	msgs[0].flags = 0;
	msgs[0].len = 2;
	msgs[0].buf = (u8 *)&reg_addr_be;

	/* Read data from register */
	msgs[1].addr = client->addr;
	msgs[1].flags = I2C_M_RD;
	msgs[1].len = len;
	msgs[1].buf = &data_be_p[4 - len];

	ret = i2c_transfer(client->adapter, msgs, ARRAY_SIZE(msgs));
	if (ret != ARRAY_SIZE(msgs))
		return -EIO;

	*val = be32_to_cpu(data_be);

	return 0;
}

/* Write registers up to 4 at a time */
static int xc530_write_reg(struct i2c_client *client, u8 reg, u8 val)
{
	struct i2c_msg msg;
	u8 buf[2];
	int ret;

	buf[0] = reg & 0xFF;
	buf[1] = val;

	msg.addr = client->addr;
	msg.flags = client->flags;
	msg.buf = buf;
	msg.len = sizeof(buf);

	ret = i2c_transfer(client->adapter, &msg, 1);
	if (ret >= 0)
		return 0;

	dev_err(&client->dev,
		"xc530 write reg(0x%x val:0x%x) failed ! ret =  %d\n",
		reg, val, ret);

	return ret;
}

static int xc530_write_array(struct i2c_client *client,
			const struct regval1 *regs)
{
	int i, ret = 0;

	i = 0;
	while (regs[i].addr != REG_NULL_XC530) {
		ret = xc530_write_reg(client, regs[i].addr, regs[i].val);
		if (ret) {
			dev_dbg(&client->dev, "%s failed ! ret = %d\n",
				__func__, ret);
			break;
		}
		i++;
	}

	return ret;
}

static int xc7080_get_reso_dist(const struct xc7080_mode *mode,
			struct v4l2_mbus_framefmt *framefmt)
{
	return abs(mode->width - framefmt->width) +
		abs(mode->height - framefmt->height);
}

static const struct xc7080_mode *
xc7080_find_best_fit(struct v4l2_subdev_format *fmt)
{
	struct v4l2_mbus_framefmt *framefmt = &fmt->format;
	int dist;
	int cur_best_fit = 0;
	int cur_best_fit_dist = -1;
	unsigned int i;

	for (i = 0; i < ARRAY_SIZE(supported_modes); i++) {
		dist = xc7080_get_reso_dist(&supported_modes[i], framefmt);
		if (cur_best_fit_dist == -1 || dist < cur_best_fit_dist) {
			cur_best_fit_dist = dist;
			cur_best_fit = i;
		}
	}

	return &supported_modes[cur_best_fit];
}

static int xc7080_set_fmt(struct v4l2_subdev *sd,
			  struct v4l2_subdev_pad_config *cfg,
			  struct v4l2_subdev_format *fmt)
{
	struct xc7080 *xc7080 = to_xc7080(sd);
	const struct xc7080_mode *mode;
	s64 h_blank, vblank_def;

	mutex_lock(&xc7080->mutex);

	mode = xc7080_find_best_fit(fmt);
	fmt->format.code = XC7080_PIX_FORMAT;
	fmt->format.width = mode->width;
	fmt->format.height = mode->height;
	fmt->format.field = V4L2_FIELD_NONE;
	if (fmt->which == V4L2_SUBDEV_FORMAT_TRY) {
#ifdef CONFIG_VIDEO_V4L2_SUBDEV_API
		*v4l2_subdev_get_try_format(sd, cfg, fmt->pad) = fmt->format;
#else
		mutex_unlock(&xc7080->mutex);
		return -ENOTTY;
#endif
	} else {
		xc7080->cur_mode = mode;
		h_blank = mode->hts_def - mode->width;
		__v4l2_ctrl_modify_range(xc7080->hblank, h_blank,
					 h_blank, 1, h_blank);
		vblank_def = mode->vts_def - mode->height;
		__v4l2_ctrl_modify_range(xc7080->vblank, vblank_def,
					 XC7080_VTS_MAX - mode->height,
					 1, vblank_def);
	}

	mutex_unlock(&xc7080->mutex);

	return 0;
}

static int xc7080_get_fmt(struct v4l2_subdev *sd,
			  struct v4l2_subdev_pad_config *cfg,
			  struct v4l2_subdev_format *fmt)
{
	struct xc7080 *xc7080 = to_xc7080(sd);
	const struct xc7080_mode *mode = xc7080->cur_mode;

	mutex_lock(&xc7080->mutex);
	if (fmt->which == V4L2_SUBDEV_FORMAT_TRY) {
#ifdef CONFIG_VIDEO_V4L2_SUBDEV_API
		fmt->format = *v4l2_subdev_get_try_format(sd, cfg, fmt->pad);
#else
		mutex_unlock(&xc7080->mutex);
		return -ENOTTY;
#endif
	} else {
		fmt->format.width = mode->width;
		fmt->format.height = mode->height;
		fmt->format.code = XC7080_PIX_FORMAT;
		fmt->format.field = V4L2_FIELD_NONE;
	}
	mutex_unlock(&xc7080->mutex);

	return 0;
}

static int xc7080_enum_mbus_code(struct v4l2_subdev *sd,
				 struct v4l2_subdev_pad_config *cfg,
				 struct v4l2_subdev_mbus_code_enum *code)
{
	if (code->index != 0)
		return -EINVAL;
	code->code = XC7080_PIX_FORMAT;

	return 0;
}

static int xc7080_enum_frame_sizes(struct v4l2_subdev *sd,
				   struct v4l2_subdev_pad_config *cfg,
				   struct v4l2_subdev_frame_size_enum *fse)
{
	if (fse->index >= ARRAY_SIZE(supported_modes))
		return -EINVAL;

	if (fse->code != XC7080_PIX_FORMAT)
		return -EINVAL;

	fse->min_width  = supported_modes[fse->index].width;
	fse->max_width  = supported_modes[fse->index].width;
	fse->max_height = supported_modes[fse->index].height;
	fse->min_height = supported_modes[fse->index].height;

	return 0;
}

static int xc7080_enable_test_pattern(struct xc7080 *xc7080, u32 pattern)
{
	u32 val;

	if (pattern)
		val = (pattern - 1) | XC7080_TEST_PATTERN_ENABLE;
	else
		val = XC7080_TEST_PATTERN_DISABLE;

	return xc7080_write_reg(xc7080->client, XC7080_REG_TEST_PATTERN,
				XC7080_REG_VALUE_08BIT, val);
}

static int XC7080_g_frame_interval(struct v4l2_subdev *sd,
				   struct v4l2_subdev_frame_interval *fi)
{
	struct xc7080 *xc7080 = to_xc7080(sd);
	const struct xc7080_mode *mode = xc7080->cur_mode;

	mutex_lock(&xc7080->mutex);
	fi->interval = mode->max_fps;
	mutex_unlock(&xc7080->mutex);

	return 0;
}

static void xc7080_get_module_inf(struct xc7080 *xc7080,
				  struct rkmodule_inf *inf)
{
	memset(inf, 0, sizeof(*inf));
	strlcpy(inf->base.sensor, XC7080_NAME, sizeof(inf->base.sensor));
	strlcpy(inf->base.module, xc7080->module_name,
		sizeof(inf->base.module));
	strlcpy(inf->base.lens, xc7080->len_name, sizeof(inf->base.lens));
}

static long xc7080_ioctl(struct v4l2_subdev *sd, unsigned int cmd, void *arg)
{
	struct xc7080 *xc7080 = to_xc7080(sd);
	long ret = 0;

	switch (cmd) {
	case RKMODULE_GET_MODULE_INFO:
		xc7080_get_module_inf(xc7080, (struct rkmodule_inf *)arg);
		break;
	default:
		ret = -ENOIOCTLCMD;
		break;
	}

	return ret;
}

#ifdef CONFIG_COMPAT
static long xc7080_compat_ioctl32(struct v4l2_subdev *sd,
				  unsigned int cmd, unsigned long arg)
{
	void __user *up = compat_ptr(arg);
	struct rkmodule_inf *inf;
	struct rkmodule_awb_cfg *cfg;
	long ret;

	switch (cmd) {
	case RKMODULE_GET_MODULE_INFO:
		inf = kzalloc(sizeof(*inf), GFP_KERNEL);
		if (!inf) {
			ret = -ENOMEM;
			return ret;
		}

		ret = xc7080_ioctl(sd, cmd, inf);
		if (!ret)
			ret = copy_to_user(up, inf, sizeof(*inf));
		kfree(inf);
		break;
	case RKMODULE_AWB_CFG:
		cfg = kzalloc(sizeof(*cfg), GFP_KERNEL);
		if (!cfg) {
			ret = -ENOMEM;
			return ret;
		}

		ret = copy_from_user(cfg, up, sizeof(*cfg));
		if (!ret)
			ret = xc7080_ioctl(sd, cmd, cfg);
		kfree(cfg);
		break;
	default:
		ret = -ENOIOCTLCMD;
		break;
	}

	return ret;
}
#endif

static int __xc7080_start_stream(struct xc7080 *xc7080)
{
	int ret;

	ret = xc7080_write_array(xc7080->client, XC7080_bypass_on);
	if (ret)
		return ret;
	ret = xc530_write_array(xc7080->sub_client,
		xc7080->cur_mode->xc530_reg_list);
	if (ret)
		return ret;
	ret = xc7080_write_array(xc7080->client, XC7080_bypass_off);
	if (ret)
		return ret;

	/* In case these controls are set before streaming */
	mutex_unlock(&xc7080->mutex);
	ret = v4l2_ctrl_handler_setup(&xc7080->ctrl_handler);
	mutex_lock(&xc7080->mutex);
	if (ret)
		return ret;
	ret = xc7080_write_array(xc7080->client, stream_on);
	return ret;
}

static int __xc7080_stop_stream(struct xc7080 *xc7080)
{
	int ret;

	ret = xc7080_write_array(xc7080->client, stream_off);
	return ret;
}

static int xc7080_s_stream(struct v4l2_subdev *sd, int on)
{
	struct xc7080 *xc7080 = to_xc7080(sd);
	struct i2c_client *client = xc7080->client;
	int ret = 0;

	mutex_lock(&xc7080->mutex);
	on = !!on;
	if (on == xc7080->streaming)
		goto unlock_and_return;

	if (on) {
		ret = pm_runtime_get_sync(&client->dev);
		if (ret < 0) {
			pm_runtime_put_noidle(&client->dev);
			goto unlock_and_return;
		}
		ret = xc7080_write_array(xc7080->client,
			xc7080->cur_mode->xc7080_reg_list);
		if (ret) {
			v4l2_err(sd, "could not set init registers\n");
			pm_runtime_put_noidle(&client->dev);
			goto unlock_and_return;
			}
		ret = __xc7080_start_stream(xc7080);
		if (ret) {
			v4l2_err(sd, "start stream failed while write regs\n");
			pm_runtime_put(&client->dev);
			goto unlock_and_return;
		}
	} else {
		__xc7080_stop_stream(xc7080);
		pm_runtime_put(&client->dev);
	}

	xc7080->streaming = on;

unlock_and_return:
	mutex_unlock(&xc7080->mutex);

	return ret;
}

static int xc7080_s_power(struct v4l2_subdev *sd, int on)
{
	struct xc7080 *xc7080 = to_xc7080(sd);
	struct i2c_client *client = xc7080->client;
	int ret = 0;

	mutex_lock(&xc7080->mutex);

	/* If the power state is not modified - no work to do. */
	if (xc7080->power_on == !!on)
		goto unlock_and_return;

	if (on) {
		ret = pm_runtime_get_sync(&client->dev);
		if (ret < 0) {
			pm_runtime_put_noidle(&client->dev);
			goto unlock_and_return;
		}
		xc7080->power_on = true;
	} else {
		pm_runtime_put(&client->dev);
		xc7080->power_on = false;
	}

unlock_and_return:
	mutex_unlock(&xc7080->mutex);

	return ret;
}

/* Calculate the delay in us by clock rate and clock cycles */
static inline u32 xc7080_cal_delay(u32 cycles)
{
	return DIV_ROUND_UP(cycles, XC7080_XVCLK_FREQ / 1000 / 1000);
}

static int __xc7080_power_on(struct xc7080 *xc7080)
{
	int ret;
	u32 delay_us;
	struct device *dev = &xc7080->client->dev;

	if (!IS_ERR_OR_NULL(xc7080->pins_default)) {
		ret = pinctrl_select_state(xc7080->pinctrl,
					   xc7080->pins_default);
		if (ret < 0)
			dev_err(dev, "could not set pins\n");
	}

	ret = clk_set_rate(xc7080->xvclk, XC7080_XVCLK_FREQ);
	if (ret < 0) {
		dev_err(dev, "Failed to set xvclk rate (24MHz)\n");
		return ret;
	}
	if (clk_get_rate(xc7080->xvclk) != XC7080_XVCLK_FREQ)
		dev_warn(dev, "xvclk mismatched, modes are based on 24MHz\n");

	ret = clk_prepare_enable(xc7080->xvclk);
	if (ret < 0) {
		dev_err(dev, "Failed to enable xvclk\n");
		return ret;
	}

	if (!IS_ERR(xc7080->reset_gpio))
		gpiod_set_value_cansleep(xc7080->reset_gpio, 0);

	ret = regulator_bulk_enable(XC7080_NUM_SUPPLIES, xc7080->supplies);
	if (ret < 0) {
		dev_err(dev, "Failed to enable regulators\n");
		goto disable_clk;
	}

	if (!IS_ERR(xc7080->reset_gpio))
		gpiod_set_value_cansleep(xc7080->reset_gpio, 1);

	usleep_range(500, 1000);
	if (!IS_ERR(xc7080->pwdn_gpio))
		gpiod_set_value_cansleep(xc7080->pwdn_gpio, 1);

	/* 8192 cycles prior to first SCCB transaction */
	delay_us = xc7080_cal_delay(8192);
	usleep_range(delay_us, delay_us * 2);

	return 0;

disable_clk:
	clk_disable_unprepare(xc7080->xvclk);

	return ret;
}

static void __xc7080_power_off(struct xc7080 *xc7080)
{
	int ret;
	struct device *dev = &xc7080->client->dev;

	if (!IS_ERR(xc7080->pwdn_gpio))
		gpiod_set_value_cansleep(xc7080->pwdn_gpio, 0);
	clk_disable_unprepare(xc7080->xvclk);
	if (!IS_ERR(xc7080->reset_gpio))
		gpiod_set_value_cansleep(xc7080->reset_gpio, 0);
	if (!IS_ERR_OR_NULL(xc7080->pins_sleep)) {
		ret = pinctrl_select_state(xc7080->pinctrl,
					xc7080->pins_sleep);
		if (ret < 0)
			dev_err(dev, "could not set pins\n");
	}
	regulator_bulk_disable(XC7080_NUM_SUPPLIES, xc7080->supplies);
}

static int xc7080_runtime_resume(struct device *dev)
{
	struct i2c_client *client = to_i2c_client(dev);
	struct v4l2_subdev *sd = i2c_get_clientdata(client);
	struct xc7080 *xc7080 = to_xc7080(sd);

	return __xc7080_power_on(xc7080);
}

static int xc7080_runtime_suspend(struct device *dev)
{
	struct i2c_client *client = to_i2c_client(dev);
	struct v4l2_subdev *sd = i2c_get_clientdata(client);
	struct xc7080 *xc7080 = to_xc7080(sd);

	__xc7080_power_off(xc7080);

	return 0;
}

#ifdef CONFIG_VIDEO_V4L2_SUBDEV_API
static int xc7080_open(struct v4l2_subdev *sd, struct v4l2_subdev_fh *fh)
{
	struct xc7080 *xc7080 = to_xc7080(sd);
	struct v4l2_mbus_framefmt *try_fmt =
				v4l2_subdev_get_try_format(sd, fh->pad, 0);
	const struct xc7080_mode *def_mode = &supported_modes[0];

	mutex_lock(&xc7080->mutex);
	/* Initialize try_fmt */
	try_fmt->width = def_mode->width;
	try_fmt->height = def_mode->height;
	try_fmt->code = XC7080_PIX_FORMAT;
	try_fmt->field = V4L2_FIELD_NONE;

	mutex_unlock(&xc7080->mutex);
	/* No crop or compose */

	return 0;
}
#endif

static int xc7080_enum_frame_interval(struct v4l2_subdev *sd,
				       struct v4l2_subdev_pad_config *cfg,
				       struct v4l2_subdev_frame_interval_enum *fie)
{
	if (fie->index >= ARRAY_SIZE(supported_modes))
		return -EINVAL;

	if (fie->code != XC7080_PIX_FORMAT)
		return -EINVAL;

	fie->width = supported_modes[fie->index].width;
	fie->height = supported_modes[fie->index].height;
	fie->interval = supported_modes[fie->index].max_fps;
	return 0;
}

static const struct dev_pm_ops xc7080_pm_ops = {
	SET_RUNTIME_PM_OPS(xc7080_runtime_suspend,
				xc7080_runtime_resume, NULL)
};

#ifdef CONFIG_VIDEO_V4L2_SUBDEV_API
static const struct v4l2_subdev_internal_ops xc7080_internal_ops = {
	.open = xc7080_open,
};
#endif

static const struct v4l2_subdev_core_ops xc7080_core_ops = {
	.s_power = xc7080_s_power,
	.ioctl = xc7080_ioctl,
#ifdef CONFIG_COMPAT
	.compat_ioctl32 = xc7080_compat_ioctl32,
#endif
};

static const struct v4l2_subdev_video_ops xc7080_video_ops = {
	.s_stream = xc7080_s_stream,
	.g_frame_interval = XC7080_g_frame_interval,
};

static const struct v4l2_subdev_pad_ops xc7080_pad_ops = {
	.enum_mbus_code = xc7080_enum_mbus_code,
	.enum_frame_size = xc7080_enum_frame_sizes,
	.enum_frame_interval = xc7080_enum_frame_interval,
	.get_fmt = xc7080_get_fmt,
	.set_fmt = xc7080_set_fmt,
};

static const struct v4l2_subdev_ops xc7080_subdev_ops = {
	.core	= &xc7080_core_ops,
	.video	= &xc7080_video_ops,
	.pad	= &xc7080_pad_ops,
};

static int xc7080_set_ctrl(struct v4l2_ctrl *ctrl)
{
	struct xc7080 *xc7080 = container_of(ctrl->handler,
					struct xc7080, ctrl_handler);
	struct i2c_client *client = xc7080->client;
	int ret = 0;

	if (pm_runtime_get(&client->dev) <= 0)
		return 0;

	switch (ctrl->id) {
	case V4L2_CID_TEST_PATTERN:
		ret = xc7080_enable_test_pattern(xc7080, ctrl->val);
		break;
	default:
		dev_warn(&client->dev, "%s Unhandled id:0x%x, val:0x%x\n",
			__func__, ctrl->id, ctrl->val);
		break;
	}

	pm_runtime_put(&client->dev);

	return ret;
}

static const struct v4l2_ctrl_ops xc7080_ctrl_ops = {
	.s_ctrl = xc7080_set_ctrl,
};

static int xc7080_initialize_controls(struct xc7080 *xc7080)
{
	struct v4l2_ctrl_handler *handler;
	struct v4l2_ctrl *ctrl;
	int ret;

	handler = &xc7080->ctrl_handler;
	ret = v4l2_ctrl_handler_init(handler, 8);
	if (ret)
		return ret;
	handler->lock = &xc7080->mutex;

	ctrl = v4l2_ctrl_new_int_menu(handler, NULL, V4L2_CID_LINK_FREQ,
					0, 0, link_freq_menu_items);
	if (ctrl)
		ctrl->flags |= V4L2_CTRL_FLAG_READ_ONLY;

	v4l2_ctrl_new_std(handler, NULL, V4L2_CID_PIXEL_RATE,
			0, XC7080_PIXEL_RATE, 1, XC7080_PIXEL_RATE);

	xc7080->test_pattern =
		v4l2_ctrl_new_std_menu_items(handler,
			&xc7080_ctrl_ops,
			V4L2_CID_TEST_PATTERN,
			ARRAY_SIZE(xc7080_test_pattern_menu) - 1,
			0, 0, xc7080_test_pattern_menu);

	if (handler->error) {
		ret = handler->error;
		dev_err(&xc7080->client->dev,
			"Failed to init controls(%d)\n", ret);
		goto err_free_handler;
	}

	xc7080->subdev.ctrl_handler = handler;

	return 0;

err_free_handler:
	v4l2_ctrl_handler_free(handler);

	return ret;
}

static int xc7080_check_sensor_id(struct xc7080 *xc7080,
				  struct i2c_client *client)
{
	struct device *dev = &xc7080->client->dev;
	u32 id = 0;
	int ret;

	ret = xc7080_read_reg(client, XC7080_REG_CHIP_ID,
		XC7080_REG_VALUE_08BIT, &id);
	if (id != XC7080_CHIP_ID) {
		dev_err(dev, "Unexpected sensor id(%06x), ret(%d)\n", id, ret);
		return -ENODEV;
	}

	dev_info(dev, "Detected XC%02x sensor\n", XC7080_CHIP_ID);

	return 0;
}

static int xc7080_configure_regulators(struct xc7080 *xc7080)
{
	unsigned int i;

	for (i = 0; i < XC7080_NUM_SUPPLIES; i++)
		xc7080->supplies[i].supply = xc7080_supply_names[i];

	return devm_regulator_bulk_get(&xc7080->client->dev,
					XC7080_NUM_SUPPLIES,
					xc7080->supplies);
}

static struct xc7080 *xc7080_xc530;

static int xc7080_probe(struct i2c_client *client,
			const struct i2c_device_id *id)
{
	struct device *dev = &client->dev;
	struct device_node *node = dev->of_node;
	struct xc7080 *xc7080;
	struct v4l2_subdev *sd;
	char facing[2];
	int ret;

	dev_info(dev, "driver version: %02x.%02x.%02x",
		DRIVER_VERSION >> 16,
		(DRIVER_VERSION & 0xff00) >> 8,
		DRIVER_VERSION & 0x00ff);

	if (strcmp(client->name, EXTREANL_ISP) == 0) {
		xc7080 = devm_kzalloc(dev, sizeof(*xc7080), GFP_KERNEL);
		if (!xc7080)
			return -ENOMEM;

		ret = of_property_read_u32(node,
				RKMODULE_CAMERA_MODULE_INDEX,
				&xc7080->module_index);
		ret |= of_property_read_string(node,
				RKMODULE_CAMERA_MODULE_FACING,
				&xc7080->module_facing);
		ret |= of_property_read_string(node,
				RKMODULE_CAMERA_MODULE_NAME,
				&xc7080->module_name);
		ret |= of_property_read_string(node,
				RKMODULE_CAMERA_LENS_NAME,
				&xc7080->len_name);
		if (ret) {
			dev_err(dev, "could not get module information!\n");
			return -EINVAL;
		}

		xc7080->client = client;
		xc7080->cur_mode = &supported_modes[0];

		xc7080->xvclk = devm_clk_get(dev, "xvclk");
		if (IS_ERR(xc7080->xvclk)) {
			dev_err(dev, "Failed to get xvclk\n");
			return -EINVAL;
		}

		xc7080->reset_gpio = devm_gpiod_get(dev, "rst", GPIOD_OUT_LOW);
		if (IS_ERR(xc7080->reset_gpio))
			dev_warn(dev, "Failed to get rst-gpios\n");

		xc7080->pwdn_gpio = devm_gpiod_get(dev, "pwd", GPIOD_OUT_LOW);
		if (IS_ERR(xc7080->pwdn_gpio))
			dev_warn(dev, "Failed to get pwd-gpios\n");

		xc7080->pinctrl = devm_pinctrl_get(dev);
		if (!IS_ERR(xc7080->pinctrl)) {
			xc7080->pins_default =
				pinctrl_lookup_state(xc7080->pinctrl,
					OF_CAMERA_PINCTRL_STATE_DEFAULT);
			if (IS_ERR(xc7080->pins_default))
				dev_err(dev,
					"could not get default pinstate\n");

			xc7080->pins_sleep =
				pinctrl_lookup_state(xc7080->pinctrl,
					OF_CAMERA_PINCTRL_STATE_SLEEP);
			if (IS_ERR(xc7080->pins_sleep))
				dev_err(dev, "could not get sleep pinstate\n");
		} else {
			dev_err(dev, "no pinctrl\n");
		}

		ret = xc7080_configure_regulators(xc7080);
		if (ret) {
			dev_err(dev, "Failed to get power regulators\n");
			return ret;
		}

		mutex_init(&xc7080->mutex);

		sd = &xc7080->subdev;
		v4l2_i2c_subdev_init(sd, client, &xc7080_subdev_ops);
		ret = xc7080_initialize_controls(xc7080);
		if (ret)
			goto err_destroy_mutex;

		ret = __xc7080_power_on(xc7080);
		if (ret)
			goto err_free_handler;

		ret = xc7080_check_sensor_id(xc7080, client);
		if (ret)
			goto err_power_off;

#ifdef CONFIG_VIDEO_V4L2_SUBDEV_API
		sd->internal_ops = &xc7080_internal_ops;
		sd->flags |= V4L2_SUBDEV_FL_HAS_DEVNODE;
#endif
#if defined(CONFIG_MEDIA_CONTROLLER)
		xc7080->pad.flags = MEDIA_PAD_FL_SOURCE;
		sd->entity.type = MEDIA_ENT_T_V4L2_SUBDEV_SENSOR;
		ret = media_entity_init(&sd->entity, 1, &xc7080->pad, 0);
		if (ret < 0)
			goto err_power_off;
#endif

		memset(facing, 0, sizeof(facing));
		if (strcmp(xc7080->module_facing, "back") == 0)
			facing[0] = 'b';
		else
			facing[0] = 'f';

		snprintf(sd->name, sizeof(sd->name), "m%02d_%s_%s %s",
			 xc7080->module_index, facing,
			 XC7080_NAME, dev_name(sd->dev));
		ret = v4l2_async_register_subdev_sensor_common(sd);
		if (ret) {
			dev_err(dev, "v4l2 async register subdev failed\n");
			goto err_clean_entity;
		}

		pm_runtime_set_active(dev);
		pm_runtime_enable(dev);
		pm_runtime_idle(dev);

		xc7080_xc530 = xc7080;
		dev_info(&client->dev, "probing successful\n");
	} else {
		if (!xc7080_xc530)
			return -EINVAL;
		xc7080_xc530->sub_client = client;
		dev_info(&client->dev, "probing successful\n");
	}

	return 0;

err_clean_entity:
#if defined(CONFIG_MEDIA_CONTROLLER)
	media_entity_cleanup(&sd->entity);
#endif
err_power_off:
	__xc7080_power_off(xc7080);
err_free_handler:
	v4l2_ctrl_handler_free(&xc7080->ctrl_handler);
err_destroy_mutex:
	mutex_destroy(&xc7080->mutex);

	return ret;
}

static int xc7080_remove(struct i2c_client *client)
{
	struct v4l2_subdev *sd = i2c_get_clientdata(client);
	struct xc7080 *xc7080 = to_xc7080(sd);

	v4l2_async_unregister_subdev(sd);
#if defined(CONFIG_MEDIA_CONTROLLER)
	media_entity_cleanup(&sd->entity);
#endif
	v4l2_ctrl_handler_free(&xc7080->ctrl_handler);
	mutex_destroy(&xc7080->mutex);

	pm_runtime_disable(&client->dev);
	if (!pm_runtime_status_suspended(&client->dev))
		__xc7080_power_off(xc7080);
	pm_runtime_set_suspended(&client->dev);

	return 0;
}

#if IS_ENABLED(CONFIG_OF)
static const struct of_device_id xc7080_of_match[] = {
	{ .compatible = "xchip,xc7080-xc530" },
	{ .compatible = "xchip,xc7080" },

	{},
};
MODULE_DEVICE_TABLE(of, xc7080_of_match);
#endif

static const struct i2c_device_id xc7080_match_id[] = {
	{ "xchip,xc7080", 0 },
	{ },
};

static struct i2c_driver xc7080_i2c_driver = {
	.driver = {
		.name = XC7080_NAME,
		.pm = &xc7080_pm_ops,
		.of_match_table = of_match_ptr(xc7080_of_match),
	},
	.probe		= &xc7080_probe,
	.remove		= &xc7080_remove,
	.id_table	= xc7080_match_id,
};

static int __init sensor_mod_init(void)
{
	return i2c_add_driver(&xc7080_i2c_driver);
}

static void __exit sensor_mod_exit(void)
{
	i2c_del_driver(&xc7080_i2c_driver);
}

device_initcall_sync(sensor_mod_init);
module_exit(sensor_mod_exit);

MODULE_DESCRIPTION("Xchip xc7080 and xc530 sensor driver");
MODULE_AUTHOR("zack.zeng");
MODULE_LICENSE("GPL v2");
