/*
 * TI SN65DSI83 initializing routine
 *
 * Copyright (C) 2022 AlphaTheta Corp.
 *
 * Based on ti-sn65dsi83.c by Marek Vasut
 *
 */
//#define DEBUG_GENERATE_TESTPATTERN
//#define DEBUG_PRINT_INFO

#include <linux/bits.h>
#include <linux/delay.h>
#include <linux/i2c.h>
#include <linux/module.h>
#include <linux/regmap.h>

/* ID registers */
#define REG_ID(n)				(0x00 + (n))
/* Reset and clock registers */
#define REG_RC_RESET				0x09
#define  REG_RC_RESET_SOFT_RESET		BIT(0)
#define REG_RC_LVDS_PLL				0x0a
#define  REG_RC_LVDS_PLL_PLL_EN_STAT		BIT(7)
#define  REG_RC_LVDS_PLL_LVDS_CLK_RANGE(n)	(((n) & 0x7) << 1)
#define  REG_RC_LVDS_PLL_HS_CLK_SRC_DPHY	BIT(0)
#define REG_RC_DSI_CLK				0x0b
#define  REG_RC_DSI_CLK_DSI_CLK_DIVIDER(n)	(((n) & 0x1f) << 3)
#define  REG_RC_DSI_CLK_REFCLK_MULTIPLIER(n)	((n) & 0x3)
#define REG_RC_PLL_EN				0x0d
#define  REG_RC_PLL_EN_PLL_EN			BIT(0)
/* DSI registers */
#define REG_DSI_LANE				0x10
#define  REG_DSI_LANE_LEFT_RIGHT_PIXELS		BIT(7)	/* DSI85-only */
#define  REG_DSI_LANE_DSI_CHANNEL_MODE_DUAL	0	/* DSI85-only */
#define  REG_DSI_LANE_DSI_CHANNEL_MODE_2SINGLE	BIT(6)	/* DSI85-only */
#define  REG_DSI_LANE_DSI_CHANNEL_MODE_SINGLE	BIT(5)
#define  REG_DSI_LANE_CHA_DSI_LANES(n)		(((n) & 0x3) << 3)
#define  REG_DSI_LANE_CHB_DSI_LANES(n)		(((n) & 0x3) << 1)
#define  REG_DSI_LANE_SOT_ERR_TOL_DIS		BIT(0)
#define REG_DSI_EQ				0x11
#define  REG_DSI_EQ_CHA_DSI_DATA_EQ(n)		(((n) & 0x3) << 6)
#define  REG_DSI_EQ_CHA_DSI_CLK_EQ(n)		(((n) & 0x3) << 2)
#define REG_DSI_CLK				0x12
#define  REG_DSI_CLK_CHA_DSI_CLK_RANGE(n)	((n) & 0xff)
/* LVDS registers */
#define REG_LVDS_FMT				0x18
#define  REG_LVDS_FMT_DE_NEG_POLARITY		BIT(7)
#define  REG_LVDS_FMT_HS_NEG_POLARITY		BIT(6)
#define  REG_LVDS_FMT_VS_NEG_POLARITY		BIT(5)
#define  REG_LVDS_FMT_LVDS_LINK_CFG		BIT(4)	/* 0:AB 1:A-only */
#define  REG_LVDS_FMT_CHA_24BPP_MODE		BIT(3)
#define  REG_LVDS_FMT_CHB_24BPP_MODE		BIT(2)
#define  REG_LVDS_FMT_CHA_24BPP_FORMAT1		BIT(1)
#define  REG_LVDS_FMT_CHB_24BPP_FORMAT1		BIT(0)
#define REG_LVDS_VCOM				0x19
#define  REG_LVDS_VCOM_CHA_LVDS_VOCM		BIT(6)
#define  REG_LVDS_VCOM_CHB_LVDS_VOCM		BIT(4)
#define  REG_LVDS_VCOM_CHA_LVDS_VOD_SWING(n)	(((n) & 0x3) << 2)
#define  REG_LVDS_VCOM_CHB_LVDS_VOD_SWING(n)	((n) & 0x3)
#define REG_LVDS_LANE				0x1a
#define  REG_LVDS_LANE_EVEN_ODD_SWAP		BIT(6)
#define  REG_LVDS_LANE_CHA_REVERSE_LVDS		BIT(5)
#define  REG_LVDS_LANE_CHB_REVERSE_LVDS		BIT(4)
#define  REG_LVDS_LANE_CHA_LVDS_TERM		BIT(1)
#define  REG_LVDS_LANE_CHB_LVDS_TERM		BIT(0)
#define REG_LVDS_CM				0x1b
#define  REG_LVDS_CM_CHA_LVDS_CM_ADJUST(n)	(((n) & 0x3) << 4)
#define  REG_LVDS_CM_CHB_LVDS_CM_ADJUST(n)	((n) & 0x3)
/* Video registers */
#define REG_VID_CHA_ACTIVE_LINE_LENGTH_LOW	0x20
#define REG_VID_CHA_ACTIVE_LINE_LENGTH_HIGH	0x21
#define REG_VID_CHA_VERTICAL_DISPLAY_SIZE_LOW	0x24
#define REG_VID_CHA_VERTICAL_DISPLAY_SIZE_HIGH	0x25
#define REG_VID_CHA_SYNC_DELAY_LOW		0x28
#define REG_VID_CHA_SYNC_DELAY_HIGH		0x29
#define REG_VID_CHA_HSYNC_PULSE_WIDTH_LOW	0x2c
#define REG_VID_CHA_HSYNC_PULSE_WIDTH_HIGH	0x2d
#define REG_VID_CHA_VSYNC_PULSE_WIDTH_LOW	0x30
#define REG_VID_CHA_VSYNC_PULSE_WIDTH_HIGH	0x31
#define REG_VID_CHA_HORIZONTAL_BACK_PORCH	0x34
#define REG_VID_CHA_VERTICAL_BACK_PORCH		0x36
#define REG_VID_CHA_HORIZONTAL_FRONT_PORCH	0x38
#define REG_VID_CHA_VERTICAL_FRONT_PORCH	0x3a
#define REG_VID_CHA_TEST_PATTERN		0x3c
#define  REG_VID_CHA_TEST_PATTERN_BIT	BIT(4)
/* IRQ registers */
#define REG_IRQ_GLOBAL				0xe0
#define  REG_IRQ_GLOBAL_IRQ_EN			BIT(0)
#define REG_IRQ_EN				0xe1
#define  REG_IRQ_EN_CHA_SYNCH_ERR_EN		BIT(7)
#define  REG_IRQ_EN_CHA_CRC_ERR_EN		BIT(6)
#define  REG_IRQ_EN_CHA_UNC_ECC_ERR_EN		BIT(5)
#define  REG_IRQ_EN_CHA_COR_ECC_ERR_EN		BIT(4)
#define  REG_IRQ_EN_CHA_LLP_ERR_EN		BIT(3)
#define  REG_IRQ_EN_CHA_SOT_BIT_ERR_EN		BIT(2)
#define  REG_IRQ_EN_CHA_PLL_UNLOCK_EN		BIT(0)
#define REG_IRQ_STAT				0xe5
#define  REG_IRQ_STAT_CHA_SYNCH_ERR		BIT(7)
#define  REG_IRQ_STAT_CHA_CRC_ERR		BIT(6)
#define  REG_IRQ_STAT_CHA_UNC_ECC_ERR		BIT(5)
#define  REG_IRQ_STAT_CHA_COR_ECC_ERR		BIT(4)
#define  REG_IRQ_STAT_CHA_LLP_ERR		BIT(3)
#define  REG_IRQ_STAT_CHA_SOT_BIT_ERR		BIT(2)
#define  REG_IRQ_STAT_CHA_PLL_UNLOCK		BIT(0)

#define INIT_RETRY_COUNT_MAX	(3)

struct sn65dsi83_dsi_lvds_param {
	unsigned char	DSI_DATA_LANES;			/* dsi lanes */
	unsigned char	DSI_BPP;				/* bpp */

	unsigned char	LVDS_DE_NEG_POLARITY;	/* 0: positive polarity, 1: negative polarity */
	unsigned char	LVDS_HS_NEG_POLARITY;	/* 0: positive polarity, 1: negative polarity */
	unsigned char	LVDS_VS_NEG_POLARITY;	/* 0: positive polarity, 1: negative polarity */
	unsigned char	LVDS_CHA_24BPP_MODE;	/* 0: Force 18bpp; LVDS channel A lane 4 (A_Y3P/N) is disabled */
											/* 1: Force 24bpp; LVDS channel A lane 4 (B_Y3P/N) is enabled */
	unsigned char	LVDS_CHA_24BPP_FORMAT1;	/* 0: LVDS channel A lane A_Y3P/N transmits the 2 most significant bits (MSB) per color; Format 2 */
											/* 1: LVDS channel B lane A_Y3P/N transmits the 2 least significant bits (LSB) per color; Format 1 */
											/*  jeida-18: LVDS_CHA_24BPP_MODE = 0, LVDS_CHA_24BPP_FORMAT1 = 0 */
											/*  jeida-24: LVDS_CHA_24BPP_MODE = 1, LVDS_CHA_24BPP_FORMAT1 = 1 */
											/*  vesa-24:  LVDS_CHA_24BPP_MODE = 1, LVDS_CHA_24BPP_FORMAT1 = 0 */
	unsigned char	LVDS_CHA_REVERSE_LVDS;	/* 1: Reversed LVDS Channel A pin order. */
	unsigned char	LVDS_CHA_LVDS_TERM;		/* 0: 100Ω differential termination, 1: 200Ω differential termination */
	unsigned int	LVDS_CHA_SYNC_DELAY;	/* sync delay */

	unsigned int	LVDS_CLOCK;							/* dot clock */
	unsigned int	_DSI_CHA_DSI_CLOCK_RANGE_FORCE;		/* force CHA_DSI_CLK_RANGE to be set (UINT_MAX = not use) */
	unsigned int	_LVDS_CHA_DSI_DIV_FORCE;			/* force DSI_CLK_DIVIDER to be set (UINT_MAX = not use) */
	unsigned int	LVDS_CHA_ACTIVE_LINE_LENGTH;		/* hactive */
	unsigned int	LVDS_CHA_VERTICAL_DISPLAY_SIZE;		/* vactive */
	unsigned char	LVDS_CHA_HORIZONTAL_BACK_PORCH;		/* hback-porch */
	unsigned char	LVDS_CHA_HORIZONTAL_FRONT_PORCH;	/* hfront-porch */
	unsigned char	LVDS_CHA_VERTICAL_BACK_PORCH;		/* vback-porch */
	unsigned char	LVDS_CHA_VERTICAL_FRONT_PORCH;		/* vfront-porch */
	unsigned int	LVDS_CHA_HSYNC_PULSE_WIDTH;			/* hsync-len */
	unsigned int	LVDS_CHA_VSYNC_PULSE_WIDTH;			/* vsync-len */
};
#define _NOT_USE		UINT_MAX

static const struct sn65dsi83_dsi_lvds_param sn65dsi83_dsi_lvds_params[2] = {
	{
		/* Main LCD */
		.DSI_DATA_LANES = 4,
		.DSI_BPP        = 24,						/* 24bpp DSI */

		/* LVDS: 4lane 24bpp */
		.LVDS_DE_NEG_POLARITY   = 0,
		.LVDS_HS_NEG_POLARITY   = 1,
		.LVDS_VS_NEG_POLARITY   = 1,
		.LVDS_CHA_24BPP_MODE    = 1,				/* 1: 24bpp */
		.LVDS_CHA_24BPP_FORMAT1 = 0,				/* 0: Format 2 */
		.LVDS_CHA_REVERSE_LVDS  = 0,
		.LVDS_CHA_LVDS_TERM     = 0,
		.LVDS_CHA_SYNC_DELAY    = 32,

		/* 1280x720 */
		.LVDS_CLOCK                      = 54000,	/* 54.00 [MHz] */
		._DSI_CHA_DSI_CLOCK_RANGE_FORCE  = 36,		/* rk3399でDSI-1クロックが180MHzになるため */
		._LVDS_CHA_DSI_DIV_FORCE         = 3,		/* rk3399でDSI-1クロックが180MHzになるため、3分周させる */
		.LVDS_CHA_ACTIVE_LINE_LENGTH     = 1280,	/* 1280 pixels */
		.LVDS_CHA_VERTICAL_DISPLAY_SIZE  = 720,		/* 720 pixels */
		.LVDS_CHA_HSYNC_PULSE_WIDTH      = 24,		/* hsync-len */
		.LVDS_CHA_HORIZONTAL_FRONT_PORCH = 20,		/* hfront-porch */
		.LVDS_CHA_HORIZONTAL_BACK_PORCH  = 20,		/* hback-porch */
		.LVDS_CHA_VSYNC_PULSE_WIDTH      = 6,		/* vsync-len */
		.LVDS_CHA_VERTICAL_FRONT_PORCH   = 6,		/* vfront-porch */
		.LVDS_CHA_VERTICAL_BACK_PORCH    = 6,		/* vback-porch */
	},
	{
		/* Jog LCD */
		.DSI_DATA_LANES = 4,
//		.DSI_DATA_LANES = 3,						/* 4lanes -> 3lanes */
		.DSI_BPP        = 24,						/* 24bpp DSI */

		/* LVDS: 3lane 18bpp */
		.LVDS_CHA_24BPP_MODE    = 0,				/* 0: 18bpp (lane A_Y3P/N is disabled) */
		.LVDS_CHA_24BPP_FORMAT1 = 1,				/* 1: Format 1 */
		.LVDS_CHA_REVERSE_LVDS  = 0,
		.LVDS_CHA_LVDS_TERM     = 0,
		.LVDS_CHA_SYNC_DELAY    = 32,

		.LVDS_DE_NEG_POLARITY   = 1,
		.LVDS_HS_NEG_POLARITY   = 1,
		.LVDS_VS_NEG_POLARITY   = 0,

		/* 1280x240 ((320x4)x240) */
		.LVDS_CLOCK                      = 36000,	/* 36.000 [MHz] */
		._DSI_CHA_DSI_CLOCK_RANGE_FORCE  = 36,		/* rk3399でDSI-2のDSIクロックがDSI-1に引きずられて180MHzになるため */
		._LVDS_CHA_DSI_DIV_FORCE         = 5,		/* rk3399でDSI-2のDSIクロックがDSI-1に引きずられて180MHzになるため、5分周させる */
		.LVDS_CHA_ACTIVE_LINE_LENGTH     = 1280,	/* 1280 pixels */
		.LVDS_CHA_VERTICAL_DISPLAY_SIZE  = 240,		/* 240 pixels */
		.LVDS_CHA_HSYNC_PULSE_WIDTH      = 1,		/* hsync-len */
		.LVDS_CHA_HORIZONTAL_FRONT_PORCH = 252,		/* hfront-porch */
//		.LVDS_CHA_HORIZONTAL_FRONT_PORCH = 188,		/* hfront-porch */	/* 252 -> 188 */
		.LVDS_CHA_HORIZONTAL_BACK_PORCH  = 251,		/* hback-porch */
//		.LVDS_CHA_HORIZONTAL_BACK_PORCH  = 187,		/* hback-porch */	/* 251 -> 187 */
		.LVDS_CHA_VSYNC_PULSE_WIDTH      = 4,		/* vsync-len */
		.LVDS_CHA_VERTICAL_FRONT_PORCH   = 25,
		.LVDS_CHA_VERTICAL_BACK_PORCH    = 14,
	}
};

struct sn65dsi83 {
	unsigned char			type;
	struct device			*dev;
	struct regmap			*regmap;
};

static const struct regmap_range sn65dsi83_readable_ranges[] = {
	regmap_reg_range(REG_ID(0), REG_ID(8)),
	regmap_reg_range(REG_RC_LVDS_PLL, REG_RC_DSI_CLK),
	regmap_reg_range(REG_RC_PLL_EN, REG_RC_PLL_EN),
	regmap_reg_range(REG_DSI_LANE, REG_DSI_CLK),
	regmap_reg_range(REG_LVDS_FMT, REG_LVDS_CM),
	regmap_reg_range(REG_VID_CHA_ACTIVE_LINE_LENGTH_LOW,
			 REG_VID_CHA_ACTIVE_LINE_LENGTH_HIGH),
	regmap_reg_range(REG_VID_CHA_VERTICAL_DISPLAY_SIZE_LOW,
			 REG_VID_CHA_VERTICAL_DISPLAY_SIZE_HIGH),
	regmap_reg_range(REG_VID_CHA_SYNC_DELAY_LOW,
			 REG_VID_CHA_SYNC_DELAY_HIGH),
	regmap_reg_range(REG_VID_CHA_HSYNC_PULSE_WIDTH_LOW,
			 REG_VID_CHA_HSYNC_PULSE_WIDTH_HIGH),
	regmap_reg_range(REG_VID_CHA_VSYNC_PULSE_WIDTH_LOW,
			 REG_VID_CHA_VSYNC_PULSE_WIDTH_HIGH),
	regmap_reg_range(REG_VID_CHA_HORIZONTAL_BACK_PORCH,
			 REG_VID_CHA_HORIZONTAL_BACK_PORCH),
	regmap_reg_range(REG_VID_CHA_VERTICAL_BACK_PORCH,
			 REG_VID_CHA_VERTICAL_BACK_PORCH),
	regmap_reg_range(REG_VID_CHA_HORIZONTAL_FRONT_PORCH,
			 REG_VID_CHA_HORIZONTAL_FRONT_PORCH),
	regmap_reg_range(REG_VID_CHA_VERTICAL_FRONT_PORCH,
			 REG_VID_CHA_VERTICAL_FRONT_PORCH),
	regmap_reg_range(REG_VID_CHA_TEST_PATTERN, REG_VID_CHA_TEST_PATTERN),
	regmap_reg_range(REG_IRQ_GLOBAL, REG_IRQ_EN),
	regmap_reg_range(REG_IRQ_STAT, REG_IRQ_STAT),
};

static const struct regmap_access_table sn65dsi83_readable_table = {
	.yes_ranges = sn65dsi83_readable_ranges,
	.n_yes_ranges = ARRAY_SIZE(sn65dsi83_readable_ranges),
};

static const struct regmap_range sn65dsi83_writeable_ranges[] = {
	regmap_reg_range(REG_RC_RESET, REG_RC_DSI_CLK),
	regmap_reg_range(REG_RC_PLL_EN, REG_RC_PLL_EN),
	regmap_reg_range(REG_DSI_LANE, REG_DSI_CLK),
	regmap_reg_range(REG_LVDS_FMT, REG_LVDS_CM),
	regmap_reg_range(REG_VID_CHA_ACTIVE_LINE_LENGTH_LOW,
			 REG_VID_CHA_ACTIVE_LINE_LENGTH_HIGH),
	regmap_reg_range(REG_VID_CHA_VERTICAL_DISPLAY_SIZE_LOW,
			 REG_VID_CHA_VERTICAL_DISPLAY_SIZE_HIGH),
	regmap_reg_range(REG_VID_CHA_SYNC_DELAY_LOW,
			 REG_VID_CHA_SYNC_DELAY_HIGH),
	regmap_reg_range(REG_VID_CHA_HSYNC_PULSE_WIDTH_LOW,
			 REG_VID_CHA_HSYNC_PULSE_WIDTH_HIGH),
	regmap_reg_range(REG_VID_CHA_VSYNC_PULSE_WIDTH_LOW,
			 REG_VID_CHA_VSYNC_PULSE_WIDTH_HIGH),
	regmap_reg_range(REG_VID_CHA_HORIZONTAL_BACK_PORCH,
			 REG_VID_CHA_HORIZONTAL_BACK_PORCH),
	regmap_reg_range(REG_VID_CHA_VERTICAL_BACK_PORCH,
			 REG_VID_CHA_VERTICAL_BACK_PORCH),
	regmap_reg_range(REG_VID_CHA_HORIZONTAL_FRONT_PORCH,
			 REG_VID_CHA_HORIZONTAL_FRONT_PORCH),
	regmap_reg_range(REG_VID_CHA_VERTICAL_FRONT_PORCH,
			 REG_VID_CHA_VERTICAL_FRONT_PORCH),
	regmap_reg_range(REG_VID_CHA_TEST_PATTERN, REG_VID_CHA_TEST_PATTERN),
	regmap_reg_range(REG_IRQ_GLOBAL, REG_IRQ_EN),
	regmap_reg_range(REG_IRQ_STAT, REG_IRQ_STAT),
};

static const struct regmap_access_table sn65dsi83_writeable_table = {
	.yes_ranges = sn65dsi83_writeable_ranges,
	.n_yes_ranges = ARRAY_SIZE(sn65dsi83_writeable_ranges),
};

static const struct regmap_range sn65dsi83_volatile_ranges[] = {
	regmap_reg_range(REG_RC_RESET, REG_RC_RESET),
	regmap_reg_range(REG_RC_LVDS_PLL, REG_RC_LVDS_PLL),
	regmap_reg_range(REG_IRQ_STAT, REG_IRQ_STAT),
};

static const struct regmap_access_table sn65dsi83_volatile_table = {
	.yes_ranges = sn65dsi83_volatile_ranges,
	.n_yes_ranges = ARRAY_SIZE(sn65dsi83_volatile_ranges),
};

static const struct regmap_config sn65dsi83_regmap_config = {
	.reg_bits = 8,
	.val_bits = 8,
	.rd_table = &sn65dsi83_readable_table,
	.wr_table = &sn65dsi83_writeable_table,
	.volatile_table = &sn65dsi83_volatile_table,
	.cache_type = REGCACHE_RBTREE,
	.max_register = REG_IRQ_STAT,
};

static u8 sn65dsi83_get_lvds_range(unsigned char type)
{
	unsigned int mode_clock = sn65dsi83_dsi_lvds_params[type].LVDS_CLOCK;
	u8 val;

	/*** from linux device driver ***/
	/*
	 * The encoding of the LVDS_CLK_RANGE is as follows:
	 * 000 - 25 MHz <= LVDS_CLK < 37.5 MHz
	 * 001 - 37.5 MHz <= LVDS_CLK < 62.5 MHz
	 * 010 - 62.5 MHz <= LVDS_CLK < 87.5 MHz
	 * 011 - 87.5 MHz <= LVDS_CLK < 112.5 MHz
	 * 100 - 112.5 MHz <= LVDS_CLK < 137.5 MHz
	 * 101 - 137.5 MHz <= LVDS_CLK <= 154 MHz
	 * which is a range of 12.5MHz..162.5MHz in 50MHz steps, except that
	 * the ends of the ranges are clamped to the supported range. Since
	 * sn65dsi83_mode_valid() already filters the valid modes and limits
	 * the clock to 25..154 MHz, the range calculation can be simplified
	 * as follows:
	 */
//	if (ctx->lvds_dual_link)
//		mode_clock /= 2;

	val = (mode_clock - 12500) / 25000;
#ifdef DEBUG_PRINT_INFO
	pr_info("%s: type = %d, mode_clock = %d, ret = %d\n", __func__, type, mode_clock, val);
#endif
	return val;
}

static u8 sn65dsi83_get_dsi_range(unsigned char type)
{
	unsigned int mode_clock = sn65dsi83_dsi_lvds_params[type].LVDS_CLOCK;
	unsigned int bpp = sn65dsi83_dsi_lvds_params[type].DSI_BPP;
	unsigned int dsi_data_lanes = sn65dsi83_dsi_lvds_params[type].DSI_DATA_LANES;
	u8 val;

	if (sn65dsi83_dsi_lvds_params[type]._DSI_CHA_DSI_CLOCK_RANGE_FORCE == _NOT_USE) {
		/* CHA_DSI_CLOCK_RANGE_FORCEを計算する場合 */
		/*** from linux device driver ***/
		/*
		 * The encoding of the CHA_DSI_CLK_RANGE is as follows:
		 * 0x00 through 0x07 - Reserved
		 * 0x08 - 40 <= DSI_CLK < 45 MHz
		 * 0x09 - 45 <= DSI_CLK < 50 MHz
		 * ...
		 * 0x63 - 495 <= DSI_CLK < 500 MHz
		 * 0x64 - 500 MHz
		 * 0x65 through 0xFF - Reserved
		 * which is DSI clock in 5 MHz steps, clamped to 40..500 MHz.
		 * The DSI clock are calculated as:
		 *  DSI_CLK = mode clock * bpp / dsi_data_lanes / 2
		 * the 2 is there because the bus is DDR.
		 */
		val = DIV_ROUND_UP(clamp(mode_clock * bpp / dsi_data_lanes / 2, 40000U, 500000U), 5000U);
	}
	else {
		/* CHA_DSI_CLOCK_RANGE_FORCEを即値設定する場合 */
		val = sn65dsi83_dsi_lvds_params[type]._DSI_CHA_DSI_CLOCK_RANGE_FORCE;
	}
#ifdef DEBUG_PRINT_INFO
	pr_info("%s: type = %d, mode_clock = %d, bpp = %d, dsi_data_lanes = %d, val = %d\n",
		__func__, type, mode_clock, bpp, dsi_data_lanes, val);
#endif
	return val;
}

static u8 sn65dsi83_get_dsi_div(unsigned char type)
{
	unsigned int bpp = sn65dsi83_dsi_lvds_params[type].DSI_BPP;
	unsigned int dsi_data_lanes = sn65dsi83_dsi_lvds_params[type].DSI_DATA_LANES;
	unsigned int dsi_div = bpp;
	u8 val;

	if (sn65dsi83_dsi_lvds_params[type]._LVDS_CHA_DSI_DIV_FORCE == _NOT_USE) {
		/* CHA_DSI_DIV_FORCEを計算する場合 */
		/*** from linux device driver ***/
		/* The divider is (DSI_CLK / LVDS_CLK) - 1, which really is: */
		dsi_div /= dsi_data_lanes;

		//	if (!ctx->lvds_dual_link)
			dsi_div /= 2;
	}
	else {
		/* CHA_DSI_DIV_FORCEを即値設定する場合 */
		dsi_div = sn65dsi83_dsi_lvds_params[type]._LVDS_CHA_DSI_DIV_FORCE;
	}

	val = dsi_div - 1;
#ifdef DEBUG_PRINT_INFO
	pr_info("%s: type = %d, bpp = %d, dsi_div = %d, val = %d\n",
		__func__, type, bpp, dsi_div, val);
#endif
	return val;
}

static u8 sn65dsi83_get_lvds_fmt(unsigned char type)
{
	unsigned int de_neg_polarity = sn65dsi83_dsi_lvds_params[type].LVDS_DE_NEG_POLARITY;
	unsigned int hs_neg_polarity = sn65dsi83_dsi_lvds_params[type].LVDS_HS_NEG_POLARITY;
	unsigned int vs_neg_polarity = sn65dsi83_dsi_lvds_params[type].LVDS_VS_NEG_POLARITY;
	unsigned int cha_24bpp_mode = sn65dsi83_dsi_lvds_params[type].LVDS_CHA_24BPP_MODE;
	unsigned int cha_24bpp_format1 = sn65dsi83_dsi_lvds_params[type].LVDS_CHA_24BPP_FORMAT1;
	u8 val;

	val = ((de_neg_polarity) ? REG_LVDS_FMT_DE_NEG_POLARITY : 0) |				/* Set up sync signal polarity. */
	      ((hs_neg_polarity) ? REG_LVDS_FMT_HS_NEG_POLARITY : 0) |
	      ((vs_neg_polarity) ? REG_LVDS_FMT_VS_NEG_POLARITY : 0) |
	      ((cha_24bpp_mode) ? REG_LVDS_FMT_CHA_24BPP_MODE : 0) |				/* Set up bits-per-pixel, 18bpp or 24bpp. */
	      ((cha_24bpp_format1) ? REG_LVDS_FMT_CHA_24BPP_FORMAT1 : 0);			/* Set up LVDS format, JEIDA/Format 1 or SPWG/Format 2 */

//	if (ctx->lvds_dual_link) {
//		val |= ((cha_24bpp_mode) ? REG_LVDS_FMT_CHB_24BPP_MODE : 0) |			/* Set up bits-per-pixel, 18bpp or 24bpp. */
//		       ((cha_24bpp_format1) ? REG_LVDS_FMT_CHB_24BPP_FORMAT1 : 0) |		/* Set up LVDS format, JEIDA/Format 1 or SPWG/Format 2 */
//	}
//	else {
		val |= REG_LVDS_FMT_LVDS_LINK_CFG;										/* Set up LVDS output config (DSI84,DSI85) */
//	}
	return val;
}

static int sn65dsi83_init(struct sn65dsi83 *ctx)
{
	unsigned int pval;
	__le16 le16val;
	int ret;
	unsigned char type = ctx->type;

	/* Clear reset, disable PLL */
	regmap_write(ctx->regmap, REG_RC_RESET, 0x00);
	regmap_write(ctx->regmap, REG_RC_PLL_EN, 0x00);

	/* Reference clock derived from DSI link clock. */
	regmap_write(ctx->regmap, REG_RC_LVDS_PLL,
	             REG_RC_LVDS_PLL_LVDS_CLK_RANGE(sn65dsi83_get_lvds_range(type)) |
	             REG_RC_LVDS_PLL_HS_CLK_SRC_DPHY);
	regmap_write(ctx->regmap, REG_DSI_CLK,
	             REG_DSI_CLK_CHA_DSI_CLK_RANGE(sn65dsi83_get_dsi_range(type)));
	regmap_write(ctx->regmap, REG_RC_DSI_CLK,
	             REG_RC_DSI_CLK_DSI_CLK_DIVIDER(sn65dsi83_get_dsi_div(type)));
	/* Set number of DSI lanes and LVDS link config. */
	regmap_write(ctx->regmap, REG_DSI_LANE,
	             REG_DSI_LANE_DSI_CHANNEL_MODE_SINGLE |
	             REG_DSI_LANE_CHA_DSI_LANES(~(sn65dsi83_dsi_lvds_params[type].DSI_DATA_LANES - 1)) |
	             /* CHB is DSI85-only, set to default on DSI83/DSI84 */
	             REG_DSI_LANE_CHB_DSI_LANES(3));
	/* No equalization. */
	regmap_write(ctx->regmap, REG_DSI_EQ, 0x00);

	regmap_write(ctx->regmap, REG_LVDS_FMT, sn65dsi83_get_lvds_fmt(type));
	regmap_write(ctx->regmap, REG_LVDS_VCOM, 0x00); /* 0x05->0x00 */
	regmap_write(ctx->regmap, REG_LVDS_LANE,
	             ((sn65dsi83_dsi_lvds_params[type].LVDS_CHA_REVERSE_LVDS) ? REG_LVDS_LANE_EVEN_ODD_SWAP : 0) |
	             ((sn65dsi83_dsi_lvds_params[type].LVDS_CHA_LVDS_TERM) ? REG_LVDS_LANE_CHA_LVDS_TERM : 0) |
	             REG_LVDS_LANE_CHB_LVDS_TERM);
	regmap_write(ctx->regmap, REG_LVDS_CM, 0x00);

	le16val = cpu_to_le16(sn65dsi83_dsi_lvds_params[type].LVDS_CHA_ACTIVE_LINE_LENGTH);
	regmap_bulk_write(ctx->regmap, REG_VID_CHA_ACTIVE_LINE_LENGTH_LOW,
			  &le16val, 2);
	le16val = cpu_to_le16(sn65dsi83_dsi_lvds_params[type].LVDS_CHA_VERTICAL_DISPLAY_SIZE);
	regmap_bulk_write(ctx->regmap, REG_VID_CHA_VERTICAL_DISPLAY_SIZE_LOW,
			  &le16val, 2);
	/* 32 + 1 pixel clock to ensure proper operation */
	le16val = cpu_to_le16(sn65dsi83_dsi_lvds_params[type].LVDS_CHA_SYNC_DELAY + 1);
	regmap_bulk_write(ctx->regmap, REG_VID_CHA_SYNC_DELAY_LOW, &le16val, 2);
	le16val = cpu_to_le16(sn65dsi83_dsi_lvds_params[type].LVDS_CHA_HSYNC_PULSE_WIDTH);
	regmap_bulk_write(ctx->regmap, REG_VID_CHA_HSYNC_PULSE_WIDTH_LOW,
			  &le16val, 2);
	le16val = cpu_to_le16(sn65dsi83_dsi_lvds_params[type].LVDS_CHA_VSYNC_PULSE_WIDTH);
	regmap_bulk_write(ctx->regmap, REG_VID_CHA_VSYNC_PULSE_WIDTH_LOW,
			  &le16val, 2);
	regmap_write(ctx->regmap, REG_VID_CHA_HORIZONTAL_BACK_PORCH,
		     sn65dsi83_dsi_lvds_params[type].LVDS_CHA_HORIZONTAL_BACK_PORCH);
	regmap_write(ctx->regmap, REG_VID_CHA_VERTICAL_BACK_PORCH,
		     sn65dsi83_dsi_lvds_params[type].LVDS_CHA_VERTICAL_BACK_PORCH);
	regmap_write(ctx->regmap, REG_VID_CHA_HORIZONTAL_FRONT_PORCH,
		     sn65dsi83_dsi_lvds_params[type].LVDS_CHA_HORIZONTAL_FRONT_PORCH);
	regmap_write(ctx->regmap, REG_VID_CHA_VERTICAL_FRONT_PORCH,
		     sn65dsi83_dsi_lvds_params[type].LVDS_CHA_VERTICAL_FRONT_PORCH);
#ifdef DEBUG_GENERATE_TESTPATTERN
	regmap_write(ctx->regmap, REG_VID_CHA_TEST_PATTERN, REG_VID_CHA_TEST_PATTERN_BIT);
#else
	regmap_write(ctx->regmap, REG_VID_CHA_TEST_PATTERN, 0x00);
#endif

	/* Enable PLL */
	regmap_write(ctx->regmap, REG_RC_PLL_EN, REG_RC_PLL_EN_PLL_EN);
	/* Wait for minimum of 3ms */
	usleep_range(3000, 4000);
	ret = regmap_read_poll_timeout(ctx->regmap, REG_RC_LVDS_PLL, pval,
	                               pval & REG_RC_LVDS_PLL_PLL_EN_STAT,
	                               1000, 100000);
	if (ret) {
		dev_err(ctx->dev, "failed to lock PLL, ret=%i\n", ret);
		/* On failure, disable PLL again and exit. */
		regmap_write(ctx->regmap, REG_RC_PLL_EN, 0x00);
		return -EIO;
	}

	/* Trigger reset after CSR register update. */
	regmap_write(ctx->regmap, REG_RC_RESET, REG_RC_RESET_SOFT_RESET);

	/* Clear all errors. */
	regmap_write(ctx->regmap, REG_IRQ_STAT, 0xFF);

	usleep_range(10000, 12000);
	regmap_read(ctx->regmap, REG_IRQ_STAT, &pval);
	if (pval) {
		dev_err(ctx->dev, "sn65dsi83_init: Unexpected link status 0x%02x\n", pval);
		ret = -EAGAIN;
	}
	else {
		dev_info(ctx->dev, "sn65dsi83_init: Done.\n");
		ret = 0;
	}
	return ret;
}

static int sn65dsi83_parse_dt(struct sn65dsi83 *ctx)
{
	int ret;
	unsigned int value;

	ret = of_property_read_u32(ctx->dev->of_node, "alphatheta,type", &value);
	if (ret < 0) {
		value = 0;
	}
	else {
		if (value >= ARRAY_SIZE(sn65dsi83_dsi_lvds_params)) {
			value = ARRAY_SIZE(sn65dsi83_dsi_lvds_params) - 1;
		}
	}
	ctx->type = value;
	dev_info(ctx->dev, "ret = %d, parameter type = %d\n", ret, ctx->type);
	return 0;
}

static int sn65dsi83_probe(struct i2c_client *client,
			   const struct i2c_device_id *id)
{
	struct device *dev = &client->dev;
	struct sn65dsi83 *ctx;
	int ret;
	int retry_count;

	ctx = devm_kzalloc(dev, sizeof(*ctx), GFP_KERNEL);
	if (!ctx) {
		ret = -ENOMEM;
		goto err_driver;
	}

	ctx->dev = dev;

	ret = sn65dsi83_parse_dt(ctx);
	if (ret) {
		goto err_driver;
	}

	ctx->regmap = devm_regmap_init_i2c(client, &sn65dsi83_regmap_config);
	if (IS_ERR(ctx->regmap)) {
		ret = PTR_ERR(ctx->regmap);
		goto err_driver;
	}

	dev_set_drvdata(dev, ctx);
	i2c_set_clientdata(client, ctx);

	for (retry_count = 0; retry_count < INIT_RETRY_COUNT_MAX; retry_count++) {
		ret = sn65dsi83_init(ctx);
		if (!ret) {
			break;
		}
	}
	if (ret) {
		goto err_driver;
	}

	dev_info(dev, "sn65dsi83 driver loaded.\n");
	return 0;

err_driver:
	dev_err(dev, "sn65dsi83 driver can't probe. ret = %d\n", ret);
	return ret;
}

static int sn65dsi83_remove(struct i2c_client *client)
{
	struct device *dev = &client->dev;
//	struct sn65dsi83 *ctx = i2c_get_clientdata(client);

	dev_info(dev, "sn65dsi83 driver removed.\n");
	return 0;
}

static struct i2c_device_id sn65dsi83_id[] = {
	{ "ti,sn65dsi83", 0 },
	{},
};
MODULE_DEVICE_TABLE(i2c, sn65dsi83_id);

static const struct of_device_id sn65dsi83_match_table[] = {
	{ .compatible = "ti,sn65dsi83" },
	{},
};
MODULE_DEVICE_TABLE(of, sn65dsi83_match_table);

static struct i2c_driver sn65dsi83_driver = {
	.probe = sn65dsi83_probe,
	.remove = sn65dsi83_remove,
	.id_table = sn65dsi83_id,
	.driver = {
		.name = "ti,sn65dsi83",
		.of_match_table = sn65dsi83_match_table,
	},
};
module_i2c_driver(sn65dsi83_driver);
