//#define DBG
/*
 * extcon-atc-usb-gpio.c --  USB GPIO extcon driver
 *
 * Copyright (C) 2023 AlphaTheta Corporation

 * based on extcon-usb-gpio.c by Roger Quadros
 */

#include <linux/extcon.h>
#include <linux/gpio.h>
#include <linux/gpio/consumer.h>
#include <linux/init.h>
#include <linux/interrupt.h>
#include <linux/irq.h>
#include <linux/kernel.h>
#include <linux/kobject.h>
#include <linux/module.h>
#include <linux/of_gpio.h>
#include <linux/platform_device.h>
#include <linux/slab.h>
#include <linux/workqueue.h>
#include <linux/timer.h>

#define CC_DEBOUNCE_MS			150		/* tCCDebounce(ms) */
#define CC_DEBOUNCE_STEP		10		/* tCCDebounce step(ms) */
#define SRC_DISCONNECT_MS		10		/* tSRCDisconnect(ms) */
#define SRC_DISCONNECT_STEP		10		/* tSRCDisconnect step(ms) */
#define USBC_DSC_TIME_MS		500		/* USBC_DSC time(ms) */

struct atc_usb_extcon_info {
	struct device *dev;
	struct extcon_dev *edev;
	spinlock_t lock;

	struct gpio_desc *usbc_pe_gpiod;
	struct gpio_desc *usbc_oc_gpiod;
	struct gpio_desc *snkdet_b_gpiod;
	struct gpio_desc *usbc_pe2_gpiod;
	struct gpio_desc *usbc_dsc_gpiod;
	int usbc_oc_irq;
	int snkdet_b_irq;
	unsigned long cc_debounce_step_jiffies;
	unsigned long src_disconnect_step_jiffies;
	unsigned long usbc_dsc_time_jiffies;
	struct work_struct usbc_oc_work;
	struct delayed_work usbc_pe2_work;
	struct delayed_work usbc_dsc_start_work;
	struct timer_list usbc_dsc_end_timer;

	unsigned int cc_debounce_count;
	unsigned int src_disconnect_count;
	bool overcurrent;
};

static const unsigned int usb_extcon_cable[] = {
	EXTCON_USB_HOST,
	EXTCON_NONE,
};

/* USBC_OC IRQ handler */
static irqreturn_t atc_usb_extcon_usbc_oc_irq_handler(int irq, void *dev_id)
{
	struct atc_usb_extcon_info *info = dev_id;

	queue_work(system_power_efficient_wq, &info->usbc_oc_work);
	return IRQ_HANDLED;
}

/* USBC_OC worker */
static void atc_usb_extcon_usbc_oc_work(struct work_struct *work)
{
	struct atc_usb_extcon_info *info = container_of(work,
									   typeof(*info),
									   usbc_oc_work);
	int value;
	const char logstring[] = "%s: overcurrent = %d\n";
	char *over_current_ev[2] = {"OVERCURRENT=1", NULL};

	value = gpiod_get_value(info->usbc_oc_gpiod);
	if (value) {									/* is active? */
		info->overcurrent = true;

		gpiod_set_value(info->usbc_pe_gpiod, 0);	/* inactivate USBC_PE */
		kobject_uevent_env(&info->dev->kobj, KOBJ_CHANGE, over_current_ev);
		dev_warn(info->dev, logstring, __func__, info->overcurrent);
	}
	else {											/* is inactive? */
		info->overcurrent = false;

		gpiod_set_value(info->usbc_pe_gpiod, 1);	/* activate USBC_PE */
		dev_info(info->dev, logstring, __func__, info->overcurrent);
	}
}

/* SNKDET_B IRQ handler */
static irqreturn_t atc_usb_extcon_snkdet_b_irq_handler(int irq, void *dev_id)
{
	struct atc_usb_extcon_info *info = dev_id;
	int value;
	unsigned long flags;

	value = gpiod_get_value(info->snkdet_b_gpiod);

	spin_lock_irqsave(&info->lock, flags);
	/* clear debounce count */
	info->cc_debounce_count = 0;
	info->src_disconnect_count = 0;

	if (value) {
		/* state is active (falling) */
		cancel_delayed_work_sync(&info->usbc_dsc_start_work);
		queue_delayed_work(system_power_efficient_wq,
						&info->usbc_pe2_work,
						info->cc_debounce_step_jiffies);
	}
	else {
		/* state is inactive (rising) */
		cancel_delayed_work_sync(&info->usbc_pe2_work);
		queue_delayed_work(system_power_efficient_wq,
						&info->usbc_dsc_start_work,
						info->src_disconnect_step_jiffies);
	}
#ifdef DBG
	dev_info(info->dev, "%s\n", __func__);
#endif
	spin_unlock_irqrestore(&info->lock, flags);
	return IRQ_HANDLED;
}

/* SNKDET_B active worker */
static void atc_usb_extcon_usbc_pe2_work(struct work_struct *work)
{
	struct atc_usb_extcon_info *info = container_of(to_delayed_work(work),
									   typeof(*info),
									   usbc_pe2_work);
	int value;
	int debounced;
	unsigned long flags;

	value = gpiod_get_value(info->snkdet_b_gpiod);

	if (value) {									/* is active? */
		spin_lock_irqsave(&info->lock, flags);
		info->cc_debounce_count += CC_DEBOUNCE_STEP;
		debounced = (info->cc_debounce_count >= CC_DEBOUNCE_MS);
#ifdef DBG
		dev_info(info->dev, "%s: count = %d\n", __func__, info->cc_debounce_count);
#endif
		spin_unlock_irqrestore(&info->lock, flags);

		if (debounced) {
			gpiod_set_value(info->usbc_pe2_gpiod, 1);	/* USBC_PE2 active */
			gpiod_set_value(info->usbc_dsc_gpiod, 0);	/* USBC_DSC inactive */

			extcon_set_state_sync(info->edev, EXTCON_USB_HOST, true);
#if 0
			{
				/* debug */
				char *over_current_ev[2] = {"OVERCURRENT=1", NULL};
				kobject_uevent_env(&info->dev->kobj, KOBJ_CHANGE, over_current_ev);
			}
#endif
		}
		else {
			queue_delayed_work(system_power_efficient_wq,
							&info->usbc_pe2_work,
							info->cc_debounce_step_jiffies);
		}
	}
}

/* SNKDET_B inactive start worker */
static void atc_usb_extcon_usbc_dsc_start_work(struct work_struct *work)
{
	struct atc_usb_extcon_info *info = container_of(to_delayed_work(work),
									   typeof(*info),
									   usbc_dsc_start_work);
	int value;
	int debounced;
	unsigned long flags;

	value = gpiod_get_value(info->snkdet_b_gpiod);

	if (!value) {									/* is inactive? */
		spin_lock_irqsave(&info->lock, flags);
		info->src_disconnect_count += SRC_DISCONNECT_STEP;
		debounced = (info->src_disconnect_count >= SRC_DISCONNECT_MS);
#ifdef DBG
		dev_info(info->dev, "%s: count = %d\n", __func__, info->src_disconnect_count);
#endif
		spin_unlock_irqrestore(&info->lock, flags);

		if (debounced) {
			gpiod_set_value(info->usbc_pe2_gpiod, 0);	/* USBC_PE2 inactive */
			gpiod_set_value(info->usbc_dsc_gpiod, 1);	/* USBC_DSC active */

			extcon_set_state_sync(info->edev, EXTCON_USB_HOST, false);

			/* start USBC_DSC pulse timer */
			mod_timer(&info->usbc_dsc_end_timer, jiffies + info->usbc_dsc_time_jiffies);
		}
		else {
			queue_delayed_work(system_power_efficient_wq,
							&info->usbc_dsc_start_work,
							info->src_disconnect_step_jiffies);
		}
	}
}

/* SNKDET_B inactive end timer */
static void atc_usb_extcon_usbc_dsc_end_func(struct timer_list *t)
{
	struct atc_usb_extcon_info *info = from_timer(info, t, usbc_dsc_end_timer);

	gpiod_set_value(info->usbc_dsc_gpiod, 0);		/* USBC_DSC inactive */
#ifdef DBG
	dev_info(info->dev, "%s\n", __func__);
#endif
}

/* probe */
static int atc_usb_extcon_probe(struct platform_device *pdev)
{
	struct device *dev = &pdev->dev;
	struct device_node *np = dev->of_node;
	struct atc_usb_extcon_info *info;
	int ret;

	if (!np)
		return -EINVAL;

	info = devm_kzalloc(dev, sizeof(*info), GFP_KERNEL);
	if (!info)
		return -ENOMEM;

	info->dev = dev;
	spin_lock_init(&info->lock);
	info->cc_debounce_count = 0;
	info->src_disconnect_count = 0;
	info->overcurrent = false;

	/* get gpio descriptor */
	info->usbc_pe_gpiod = devm_gpiod_get(dev, "usbc-pe", GPIOD_OUT_LOW);
	if (IS_ERR(info->usbc_pe_gpiod)) {
		dev_err(dev, "%s: failed to get USBC_PE GPIO\n", __func__);
		return PTR_ERR(info->usbc_pe_gpiod);
	}
	info->usbc_oc_gpiod = devm_gpiod_get(dev, "usbc-oc", GPIOD_IN);
	if (IS_ERR(info->usbc_oc_gpiod)) {
		dev_err(dev, "%s: failed to get USBC_OC GPIO\n", __func__);
		return PTR_ERR(info->usbc_oc_gpiod);
	}
	info->snkdet_b_gpiod = devm_gpiod_get(dev, "snkdet-b", GPIOD_IN);
	if (IS_ERR(info->snkdet_b_gpiod)) {
		dev_err(dev, "%s: failed to get SNKDET_B GPIO\n", __func__);
		return PTR_ERR(info->snkdet_b_gpiod);
	}
	info->usbc_pe2_gpiod = devm_gpiod_get(dev, "usbc-pe2", GPIOD_OUT_LOW);
	if (IS_ERR(info->usbc_pe2_gpiod)) {
		dev_err(dev, "%s: failed to get USBC_PE2 GPIO\n", __func__);
		return PTR_ERR(info->usbc_pe2_gpiod);
	}
	info->usbc_dsc_gpiod = devm_gpiod_get(dev, "usbc-dsc", GPIOD_OUT_LOW);
	if (IS_ERR(info->usbc_dsc_gpiod)) {
		dev_err(dev, "%s: failed to get USBC_DSC GPIO\n", __func__);
		return PTR_ERR(info->usbc_dsc_gpiod);
	}

	/* allocate extcon */
	info->edev = devm_extcon_dev_allocate(dev, usb_extcon_cable);
	if (IS_ERR(info->edev)) {
		dev_err(dev, "failed to allocate extcon device\n");
		return -ENOMEM;
	}
	/* register extcon */
	ret = devm_extcon_dev_register(dev, info->edev);
	if (ret < 0) {
		dev_err(dev, "failed to register extcon device\n");
		return ret;
	}

	/* set debounce time */
	info->cc_debounce_step_jiffies = msecs_to_jiffies(CC_DEBOUNCE_STEP);
	info->src_disconnect_step_jiffies = msecs_to_jiffies(SRC_DISCONNECT_STEP);

	/* convert msecs to jiffies delta */
	info->usbc_dsc_time_jiffies = msecs_to_jiffies(USBC_DSC_TIME_MS);

	/* initialize workqueue */
	INIT_WORK(&info->usbc_oc_work, atc_usb_extcon_usbc_oc_work);
	INIT_DELAYED_WORK(&info->usbc_pe2_work, atc_usb_extcon_usbc_pe2_work);
	INIT_DELAYED_WORK(&info->usbc_dsc_start_work, atc_usb_extcon_usbc_dsc_start_work);

	/* initialize timer */
	timer_setup(&info->usbc_dsc_end_timer, atc_usb_extcon_usbc_dsc_end_func, 0);

	/* get irq */
	info->usbc_oc_irq = gpiod_to_irq(info->usbc_oc_gpiod);
	if (info->usbc_oc_irq < 0) {
		dev_err(dev, "%s: failed to get USBC_OC GPIO IRQ\n", __func__);
		return info->usbc_oc_irq;
	}
	info->snkdet_b_irq = gpiod_to_irq(info->snkdet_b_gpiod);
	if (info->snkdet_b_irq < 0) {
		dev_err(dev, "%s: failed to get SNKDET_B GPIO IRQ\n", __func__);
		return info->snkdet_b_irq;
	}

	/* register USBC_OC edge handler */
	ret = devm_request_threaded_irq(dev, info->usbc_oc_irq, NULL,
					atc_usb_extcon_usbc_oc_irq_handler,
					IRQF_TRIGGER_FALLING | IRQF_TRIGGER_RISING | IRQF_ONESHOT,
					pdev->name, info);
	if (ret < 0) {
		dev_err(dev, "%s: failed to request handler for USBC_OC GPIO IRQ\n", __func__);
		return ret;
	}

	/* register SNKDET_B edge handler */
	ret = devm_request_threaded_irq(dev, info->snkdet_b_irq, NULL,
					atc_usb_extcon_snkdet_b_irq_handler,
					IRQF_TRIGGER_FALLING | IRQF_TRIGGER_RISING | IRQF_ONESHOT,
					pdev->name, info);
	if (ret < 0) {
		dev_err(dev, "%s: failed to request handler for SNKDET_B GPIO IRQ\n", __func__);
		return ret;
	}

	platform_set_drvdata(pdev, info);
	device_init_wakeup(dev, 1);

	/* start to control USBC_PE */
	atc_usb_extcon_usbc_oc_work(&info->usbc_oc_work);
	/* Perform initial detection */
	atc_usb_extcon_usbc_pe2_work(&info->usbc_pe2_work.work);

	dev_info(dev, "%s: USB GPIO extcon driver initialized", __func__);
	return 0;
}

/* remove */
static int atc_usb_extcon_remove(struct platform_device *pdev)
{
	struct atc_usb_extcon_info *info = platform_get_drvdata(pdev);

	del_timer_sync(&info->usbc_dsc_end_timer);
	cancel_work_sync(&info->usbc_oc_work);
	cancel_delayed_work_sync(&info->usbc_dsc_start_work);
	cancel_delayed_work_sync(&info->usbc_pe2_work);

	gpiod_set_value(info->usbc_dsc_gpiod, 0);		/* USBC_DSC inactive */
	gpiod_set_value(info->usbc_pe2_gpiod, 0);		/* USBC_PE2 inactive */
	gpiod_set_value(info->usbc_pe_gpiod, 0);		/* USBC_PE inactive */

	return 0;
}

static const struct of_device_id atc_usb_extcon_dt_match[] = {
	{ .compatible = "alphatheta,extcon-atc-usb-gpio", },
	{ }
};
MODULE_DEVICE_TABLE(of, atc_usb_extcon_dt_match);

static struct platform_driver atc_usb_extcon_driver = {
	.probe		= atc_usb_extcon_probe,
	.remove		= atc_usb_extcon_remove,
	.driver		= {
		.name			= "extcon-atc-usb-gpio",
		.of_match_table = atc_usb_extcon_dt_match,
	},
};
module_platform_driver(atc_usb_extcon_driver);

MODULE_DESCRIPTION("AlphaTheta USB GPIO extcon driver");
MODULE_AUTHOR("AlphaTheta Corp.");
MODULE_LICENSE("GPL v2");
