#!/bin/sh

# This script is called from our systemd unit file to mount or umount a SD/USB drive.

usage()
{
    /bin/echo "Usage: $0 {add|remove} device_name (e.g. sda1,mmcblk1p1,...)"
    exit 0
}

ACTION=$1
UDEV_PATH=""

if [[ ${ACTION} == "add" && $# -eq 2 ]]; then
    DEVBASE=$2
    UDEV_PATH=$(udevadm info --query=path --name=/dev/${DEVBASE} --attribute-walk | \
        egrep "looking at parent device" | head -n 1 | \
        sed -e "s/.*looking at parent device '\(\/devices\/.*\)\/.*\/host.*/\1/g")
elif [[ ${ACTION} == "remove" && $# -eq 3 ]]; then
    DEVBASE=$2
    UDEV_PATH=$3
else
    usage
fi

if [ -z ${UDEV_PATH} ]; then
    usage
fi

DEVICE="/dev/${DEVBASE}"
DEVICE_KIND="none"
MOUNT_CMD="none"
WRITE_PROTECT="0"

USB1_BUS_NAME="5-1"
USB2_BUS_NAME="1-1"
DEV_BUS=$(/bin/echo ${UDEV_PATH} | /bin/sed -e "s/.*\(\/.*\)\/.*\/host.*/\1/g")
DEV_BUS_ID=${DEV_BUS##*/}
/bin/echo "DEV_BUS_ID: ${DEV_BUS_ID}"

# See if this drive is already mounted, and if so where
MOUNT_POINT=$(/bin/mount | /bin/grep ${DEVICE} | /usr/bin/awk '{ print $3 }')

do_mount()
{
    if [[ -n ${MOUNT_POINT} ]]; then
        /bin/echo "Warning: ${DEVICE} is already mounted at ${MOUNT_POINT}"
        exit 1
    fi

    if [[ ! -e ${DEVICE} ]]; then
        /bin/echo "Nothing ${DEVICE} file"
        exit 1
    fi

    # Get info for this drive: $ID_FS_TYPE
    RETBLKID="$(/sbin/blkid -o udev -p ${DEVICE} 2>&1 > /dev/null)"
    if [[ ${RETBLKID} != "" ]]; then
#        /bin/echo ${RETBLKID}
        exit 0
    fi
    eval $(/sbin/blkid -o udev -p ${DEVICE})

    # Eliminate EFI partition
    if [[ ${ID_FS_LABEL} == "EFI" ]]; then
        /bin/echo "Warning: ${DEVICE} is EFI partition"
        exit 0
    fi

    # Figure out a mount point to use
    if [[ ${DEVBASE} =~ sd[a-h][1-4] ]]; then
        if [[ ${DEV_BUS_ID} == ${USB1_BUS_NAME} ]]; then
            /bin/echo "Device is USB."
            DEVICE_KIND="usb"
        elif [[ ${DEV_BUS_ID} == ${USB2_BUS_NAME} ]]; then
            /bin/echo "Device is USB-C."
            DEVICE_KIND="usb"
        fi
        /bin/echo noop > /sys/block/${DEVBASE:0:3}/queue/scheduler
        WRITE_PROTECT=$(/bin/cat /sys/block/${DEVBASE:0:3}/ro)
    elif [[ ${DEVBASE} =~ sd[a-h] ]]; then
        if [[ ! ${ID_PART_TABLE_TYPE} == "" ]]; then
            /bin/echo "No problem."
            exit 0
        fi
        /bin/echo "Device is USB without PARTITION."
        if [[ ${DEV_BUS_ID} == ${USB1_BUS_NAME} ]]; then
            /bin/echo "Device is USB."
            DEVICE_KIND="usb"
        elif [[ ${DEV_BUS_ID} == ${USB2_BUS_NAME} ]]; then
            /bin/echo "Device is USB-C."
            DEVICE_KIND="usb"
        fi
        /bin/echo noop > /sys/block/${DEVBASE:0:3}/queue/scheduler
        WRITE_PROTECT=$(/bin/cat /sys/block/${DEVBASE:0:3}/ro)
    fi

    if [[ ${DEVBASE} =~ mmcblk[1-8]p[1-4] ]]; then
        MMC_DEVICE_PATH="/sys/devices/platform/soc/ee100000.sd/mmc_host/mmc1/"
        MMC_DEVICE_NAME=$(/bin/ls ${MMC_DEVICE_PATH} | /bin/grep mmc1)
        MMC_DEVICE_TYPE=$(/bin/cat ${MMC_DEVICE_PATH}${MMC_DEVICE_NAME}/type)
        if [[ ${MMC_DEVICE_TYPE} == "MMC" ]]; then
            /bin/echo "Warning: Device is MMC!"
#            /bin/echo -n carderr > /proc/udev_sdctn1
            exit 0
        fi
        /bin/echo "Device is SD."
        DEVICE_KIND="sd"
        /bin/echo noop > /sys/block/${DEVBASE:0:7}/queue/scheduler
        WRITE_PROTECT=$(/bin/cat /sys/block/${DEVBASE:0:7}/ro)
    elif [[ ${DEVBASE} =~ mmcblk[1-8] ]]; then
        if [[ ! ${ID_PART_TABLE_TYPE} == "" ]]; then
            /bin/echo "No problem."
            exit 0
        fi
        MMC_DEVICE_PATH="/sys/devices/platform/soc/ee100000.sd/mmc_host/mmc1/"
        MMC_DEVICE_NAME=$(/bin/ls ${MMC_DEVICE_PATH} | /bin/grep mmc1)
        MMC_DEVICE_TYPE=$(/bin/cat ${MMC_DEVICE_PATH}${MMC_DEVICE_NAME}/type)
        if [[ ${MMC_DEVICE_TYPE} == "MMC" ]]; then
            /bin/echo "Warning: Device is MMC!"
#            /bin/echo -n carderr > /proc/udev_sdctn1
            exit 0
        fi
        /bin/echo "Device is SD without PARTITION."
        DEVICE_KIND="sd"
        /bin/echo noop > /sys/block/${DEVBASE:0:7}/queue/scheduler
        WRITE_PROTECT=$(/bin/cat /sys/block/${DEVBASE:0:7}/ro)
    fi

    if [[ ${DEVICE_KIND} == "usb" || ${DEVICE_KIND} == "sd" ]]; then
        PID_LIST=$(pgrep scsi_eh_)
        for SCSI_EH_PID in ${PID_LIST}
        do
            /usr/bin/taskset -p 0f ${SCSI_EH_PID}
        done
        PID_LIST=$(pgrep usb-storage)
        for USB_STORAGE_PID in ${PID_LIST}
        do
            /usr/bin/taskset -p 0f ${USB_STORAGE_PID}
        done
    elif [[ ${DEVICE_KIND} == "none" ]]; then
        /bin/echo "Warning: ${DEVBASE} is not supported"
        exit 1
    fi

    MOUNT_POINT="/media/${DEVICE_KIND}/${DEVBASE}"
    /bin/echo "Mount point: ${MOUNT_POINT}"
    if [[ -d ${MOUNT_POINT} ]]; then
        /bin/rm -rf ${MOUNT_POINT}
    fi
    /bin/mkdir -p ${MOUNT_POINT}

    # Global mount options
    OPTS="rw,noatime"

    # File system type specific mount options
    if [[ ${ID_FS_TYPE} == "vfat" ]]; then
        OPTS+=",shortname=mixed,dmask=000,fmask=000,codepage=437,iocharset=iso8859-1,usefree,utf8,flush"
        MOUNT_CMD="/bin/mount"
    fi

    if [[ ${ID_FS_TYPE} == "exfat" ]]; then
        OPTS+=",flush"
        MOUNT_CMD="/bin/mount -t exfat-fuse"
    fi

    if [[ ${ID_FS_TYPE} == "hfsplus" ]]; then
        OPTS+=",force"
        MOUNT_CMD="/bin/mount -t hfsplus"
        /sbin/fsck.hfsplus -q -f ${DEVICE}
        RET=$? ; echo "fsck [$RET]"
        if [ $RET -gt 0 ]; then
            /sbin/fsck.hfsplus -c 32m -f ${DEVICE}
        fi
    fi

    if [[ ${MOUNT_CMD} == "none" ]]; then
        /bin/echo "Warning: ${DEVICE} is not supported format"
        /bin/rm -rf ${MOUNT_POINT}
        if [[ ${DEVICE_KIND} == "usb" ]]; then
            /bin/echo -n formatusb > /proc/udev_usbctn1
        elif [[ ${DEVICE_KIND} == "sd" && ${ID_FS_TYPE} == "" ]]; then
            /bin/echo -n carderr > /proc/udev_sdctn1
        elif [[ ${DEVICE_KIND} == "sd" ]]; then
            /bin/echo -n formatsd > /proc/udev_sdctn1
        fi
        exit 1
    fi

    if ! ${MOUNT_CMD} -o ${OPTS} ${DEVICE} ${MOUNT_POINT}; then
        echo "Error mounting ${DEVICE}"
        /bin/rm -rf ${MOUNT_POINT}
        exit 1
    fi

    # Write to /proc/udev_***
    if [[ ${DEV_BUS_ID} == ${USB1_BUS_NAME} ]]; then
        /bin/echo -n mount ${MOUNT_POINT} ${ID_FS_TYPE} protect:${WRITE_PROTECT} > /proc/udev_${DEVICE_KIND}1
    elif [[ ${DEV_BUS_ID} == ${USB2_BUS_NAME} ]]; then
        /bin/echo -n mount ${MOUNT_POINT} ${ID_FS_TYPE} protect:${WRITE_PROTECT} > /proc/udev_${DEVICE_KIND}2
    fi
}

do_unmount()
{
    if [[ -z ${MOUNT_POINT} ]]; then
        /bin/echo "Warning: ${DEVICE} is not mounted"
        exit 0
    fi

    # Figure out a mount point to delete
    if [[ ${DEVBASE} =~ sd[a-h][1-4] ]]; then
        if [[ ${DEV_BUS_ID} == ${USB1_BUS_NAME} ]]; then
            /bin/echo "Device is USB."
            DEVICE_KIND="usb"
        elif [[ ${DEV_BUS_ID} == ${USB2_BUS_NAME} ]]; then
            /bin/echo "Device is USB-C."
            DEVICE_KIND="usb"
        fi
    elif [[ ${DEVBASE} =~ sd[a-h] ]]; then
        /bin/echo "Device is USB without PARTITION."
        if [[ ${DEV_BUS_ID} == ${USB1_BUS_NAME} ]]; then
            /bin/echo "Device is USB."
            DEVICE_KIND="usb"
        elif [[ ${DEV_BUS_ID} == ${USB2_BUS_NAME} ]]; then
            /bin/echo "Device is USB-C."
            DEVICE_KIND="usb"
        fi
    fi

    if [[ ${DEVBASE} =~ mmcblk[1-8]p[1-4] ]]; then
        echo "Device is SD."
        DEVICE_KIND="sd"
    fi

    if [[ ${DEVICE_KIND} == "none" ]]; then
        /bin/echo "Warning: ${DEVBASE} is not supported"
        exit 1
    fi

    MOUNT_POINT="/media/${DEVICE_KIND}/${DEVBASE}"
    /bin/umount -l ${MOUNT_POINT}
    /bin/rm -rf ${MOUNT_POINT}

    # Write to /proc/udev_***
    if [[ ${DEV_BUS_ID} == ${USB1_BUS_NAME} ]]; then
        /bin/echo -n umount ${MOUNT_POINT} > /proc/udev_${DEVICE_KIND}1
    elif [[ ${DEV_BUS_ID} == ${USB2_BUS_NAME} ]]; then
        /bin/echo -n umount ${MOUNT_POINT} > /proc/udev_${DEVICE_KIND}2
    else
        /bin/echo "Warning: can't umount ${MOUNT_POINT}"
        exit 1
    fi

    EXFAT_PROC=$(ps aux | grep -e ${MOUNT_POINT} -e exfat | grep -v grep | awk '{print $2}')
    if [ -n "${EXFAT_PROC}" ]; then
        kill -9 ${EXFAT_PROC}
    fi
}

case "${ACTION}" in
    add)
        do_mount
        ;;
    remove)
        do_unmount
        ;;
    *)
        usage
        ;;
esac

exit 0
